--liquibase formatted sql

--changeset bjrke:add_timestamps dbms:postgresql runAlways:true splitStatements:false
CREATE OR REPLACE FUNCTION update_mtime()
RETURNS TRIGGER AS $$
BEGIN
   NEW.mtime = clock_timestamp();
   RETURN NEW;
END;
$$ language 'plpgsql';

CREATE OR REPLACE FUNCTION ctime_mtime_columns(tname TEXT)
RETURNS VOID AS $$
BEGIN
    EXECUTE 'ALTER TABLE ' || tname || ' ADD COLUMN ctime TIMESTAMP WITHOUT TIME ZONE NOT NULL DEFAULT clock_timestamp();';
    EXECUTE 'ALTER TABLE ' || tname || ' ADD COLUMN mtime TIMESTAMP WITHOUT TIME ZONE NOT NULL DEFAULT clock_timestamp();';
    EXECUTE 'CREATE TRIGGER ' || tname || '_mtrg BEFORE UPDATE ON ' || tname || ' FOR EACH ROW EXECUTE PROCEDURE update_mtime();';
END;
$$ language 'plpgsql';

--changeset bjrke:add_now_function dbms:oracle runAlways:true splitStatements:false
CREATE OR REPLACE FUNCTION now
    RETURN DATE
AS
BEGIN
    RETURN SYSDATE;
END now;

--changeset bjrke:add_clock_timestamp_function dbms:oracle runAlways:true splitStatements:false
CREATE OR REPLACE FUNCTION clock_timestamp
    RETURN TIMESTAMP WITH TIME ZONE
AS
BEGIN
    RETURN SYSTIMESTAMP;
END clock_timestamp;

--changeset bjrke:add_timestamps dbms:oracle runAlways:true splitStatements:false
CREATE OR REPLACE PROCEDURE ctime_mtime_columns(tname VARCHAR2)
IS
BEGIN
    EXECUTE IMMEDIATE 'ALTER TABLE ' || tname || ' ADD (ctime TIMESTAMP DEFAULT SYSDATE NOT NULL,mtime TIMESTAMP DEFAULT SYSTIMESTAMP NOT NULL )';
    EXECUTE IMMEDIATE 'CREATE TRIGGER ' || tname || '_mtrg BEFORE UPDATE ON ' || tname || ' FOR EACH ROW BEGIN :new.mtime := SYSTIMESTAMP; END;';
END ctime_mtime_columns;

--changeset bjrke:create_document_version_table
CREATE TABLE document_version (
    id ${uuid} NOT NULL PRIMARY KEY,
    document_id ${uuid} NOT NULL,
    version ${int32} NOT NULL,
    name ${varchar255} NOT NULL,
    change_id ${uuid} NOT NULL,
    CONSTRAINT doc_version_version_uq UNIQUE (document_id, version),
    CONSTRAINT doc_version_change_fk FOREIGN KEY (change_id) REFERENCES document_version(id)
);
${execute} ctime_mtime_columns('document_version');
CREATE INDEX doc_version_document_id_idx ON document_version(document_id);

--changeset wolfgang:create_folder_version_table
CREATE TABLE folder_version (
    id ${uuid} NOT NULL PRIMARY KEY,
    CONSTRAINT folder_version_id_fk FOREIGN KEY (id) REFERENCES document_version(id)
);
${execute} ctime_mtime_columns('folder_version');

CREATE TABLE sub_folder (
    parent_id ${uuid} NOT NULL,
    child_id ${uuid} NOT NULL,
    UNIQUE (parent_id, child_id),
    CONSTRAINT sub_folder_parent_fk FOREIGN KEY (parent_id) REFERENCES folder_version(id),
    CONSTRAINT sub_folder_child_fk FOREIGN KEY (child_id) REFERENCES document_version(id)
);
${execute} ctime_mtime_columns('sub_folder');
CREATE INDEX sub_folder_parent_idx ON sub_folder(parent_id);

--changeset bjrke:create_files_table
-- need to name it files since file is a reserved word for oracle
--TODO(JB) remove
CREATE TABLE files (
    id ${uuid} NOT NULL PRIMARY KEY,
    description ${varchar1024}
);
${execute} ctime_mtime_columns('files');

--changeset bjrke:create_file_version
CREATE TABLE file_version (
    id ${uuid} NOT NULL PRIMARY KEY,
    storage_id ${varchar64} NOT NULL,
    file_type ${varchar32}  DEFAULT 'UNKNOWN' NOT NULL,
    file_size ${int64} NOT NULL,
    deleted ${bool} DEFAULT 'f' NOT NULL,
    CONSTRAINT file_version_id_fk FOREIGN KEY (id) REFERENCES document_version(id)
);
${execute} ctime_mtime_columns('file_version');

--changeset bjrke:create_change
CREATE TABLE change (
    folder_version_id ${uuid} PRIMARY KEY,
    item_id ${varchar255} NOT NULL,
    item_version_id ${uuid},
    version ${int32} NOT NULL,
    author ${int64} NOT NULL,
    change_date ${timestamp},
    CONSTRAINT change_folder_version_id_fk FOREIGN KEY (folder_version_id) REFERENCES folder_version(id),
    CONSTRAINT change_item_version_uq UNIQUE (item_id, version)
);

${execute} ctime_mtime_columns('change');
CREATE INDEX change_parent_id_idx ON change(item_id);

--changeset christian.ewers:create_file_version_preview
CREATE TABLE file_version_preview (
    id ${uuid} NOT NULL PRIMARY KEY,
    preview_id ${varchar64} NOT NULL,
    CONSTRAINT file_version_preview_id_fk FOREIGN KEY (id) REFERENCES file_version(id)
);
${execute} ctime_mtime_columns('file_version_preview');

--changeset miriam:create_search_index_queue_seq
CREATE SEQUENCE search_index_queue_seq;

--changeset miriam:create_search_index_queue dbms:postgresql
CREATE TABLE search_index_queue (
    id ${int64} PRIMARY KEY DEFAULT nextval('search_index_queue_seq'),
    document_version_id ${uuid} NOT NULL,
    index_type ${varchar32} NOT NULL,
    in_progress ${bool} DEFAULT 'f' NOT NULL
);
${execute} ctime_mtime_columns('search_index_queue');

--changeset bjrke:create_person
CREATE TABLE person (
    id ${int64} NOT NULL PRIMARY KEY,
    firstname ${varchar255},
    lastname ${varchar255},
    fullname ${varchar1024},
    imageUrl ${varchar1024}
);
${execute} ctime_mtime_columns('person');

INSERT INTO person(id)
SELECT author FROM change GROUP BY author;

ALTER TABLE change ADD CONSTRAINT change_author_fk FOREIGN KEY(author) REFERENCES person(id);

--changeset miriam:create_search_index_queue dbms:oracle
CREATE TABLE search_index_queue (
    id ${int64} DEFAULT search_index_queue_seq.NEXTVAL,
    document_version_id ${uuid} NOT NULL,
    index_type ${varchar32} NOT NULL,
    in_progress ${bool} DEFAULT 'f' NOT NULL,
    CONSTRAINT pk_search_index_queue PRIMARY KEY (id)
);
${execute} ctime_mtime_columns('search_index_queue');


--changeset christian.ewers:use_storage_id_as_primary_key_for_file_version_preview
DROP TABLE file_version_preview;

CREATE TABLE file_version_preview (
    storage_id ${varchar64} NOT NULL PRIMARY KEY,
    preview_id ${varchar64},
    preview_status ${varchar32} NOT NULL
);
${execute} ctime_mtime_columns('file_version_preview');

--changeset miriam:search_index_queue_add_document_column
DELETE FROM search_index_queue;
ALTER TABLE search_index_queue
    ADD document_id ${uuid} NOT NULL;

--changeset miriam:search_index_queue_add_in_progress_time_column
ALTER TABLE search_index_queue
    ADD in_progress_time ${timestamp};

--changeset wolfgang:create_conversation_visibility
create table conversation_visibility(
    item_id ${varchar255} NOT NULL,
    person_id ${int64} NOT NULL,
    CONSTRAINT conversation_visibility_p_pk PRIMARY KEY (item_id, person_id),
    CONSTRAINT conversation_visibility_fk FOREIGN KEY (person_id) REFERENCES person(id)
);
${execute} ctime_mtime_columns('conversation_visibility');

--changeset wolfgang:create_poll_last_timestamp_table
create table poll_last_timestamp(
    system ${varchar255} not null primary key,
    last_poll ${timestamp} not null
)

--changeset wolfgang:remove_conversation_visibility_fk
alter table conversation_visibility drop constraint conversation_visibility_fk;

--changeset wolfgang:add_mime_type_to_file_version
alter table file_version add mime_type ${varchar255} default 'application/octet-stream' not null;

--changeset wolfgang:drop_change_author_fk
alter table change drop constraint change_author_fk;

--changeset bjrke:drop_search_index_queue_document_version_id_post
ALTER TABLE search_index_queue DROP COLUMN document_version_id;

--changeset bjrke:create_document_table
CREATE TABLE document(
    id ${uuid} NOT NULL PRIMARY KEY,
    last_version_id ${uuid} UNIQUE,
    CONSTRAINT document_last_version_id_fk FOREIGN KEY (last_version_id) REFERENCES document_version(id)
);
${execute} ctime_mtime_columns('document');
INSERT INTO document(id, last_version_id)
SELECT document_id, id FROM (
   SELECT document_id, id,
   RANK() OVER (PARTITION BY document_id ORDER BY version DESC) AS rank FROM document_version
) ranked WHERE rank = 1;
ALTER TABLE document_version ADD CONSTRAINT doc_version_document_id_fk FOREIGN KEY (document_id) REFERENCES document(id);

--changeset bjrke:drop_files
DROP TABLE files;

--changeset bjrke:create_person_role
CREATE TABLE person_role(
    person_id ${int64} NOT NULL,
    role ${varchar64} NOT NULL,
    CONSTRAINT person_role_pk PRIMARY KEY (person_id, role)
);
${execute} ctime_mtime_columns('person_role');

--changeset bjrke:drop_sub_folder_child_idx_if_exists dbms:postgresql
--this index was created during an emergency hotfix on cloud
DROP INDEX IF EXISTS sub_folder_child_idx;

--changeset bjrke:add_indices
CREATE INDEX sub_folder_child_idx ON sub_folder(child_id);
CREATE INDEX document_version_change_idx ON document_version(change_id);
CREATE INDEX change_version_idx ON change(version);

--changeset wolfgang:create_item_actions
CREATE TABLE item_actions(
    item_id ${varchar255} NOT NULL,
    action ${varchar64} NOT NULL,
    condition ${varchar4000} NOT NULL,
    CONSTRAINT item_actions_pk PRIMARY KEY (item_id, action)
);
${execute} ctime_mtime_columns('item_actions');

--changeset bjrke:add_static_role_to_person_role
ALTER TABLE person_role ADD static_role ${bool} DEFAULT 'f' NOT NULL;
UPDATE person_role SET static_role = 't' WHERE NOT role LIKE '%,%';

--changeset bjrke:create_person_role_static_role_idx dbms:oracle
CREATE INDEX person_role_static_role_idx ON person_role(person_id, static_role);

--changeset bjrke:create_person_role_static_role_idx dbms:postgresql
CREATE INDEX person_role_static_role_idx ON person_role(person_id) WHERE static_role = 't';

--changeset bjrke:create_person_role_role_idx
CREATE INDEX person_role_role_idx ON person_role(role);

--changeset bjrke:fix_boolean_oracle dbms:oracle
UPDATE file_version SET deleted = 't' WHERE deleted = '1';
UPDATE file_version SET deleted = 'f' WHERE deleted = '0';

--changeset bjrke:create_item_read_role
CREATE TABLE item_read_role(
    item_id ${varchar255} NOT NULL,
    role ${varchar64} NOT NULL,
    CONSTRAINT used_read_role_pk PRIMARY KEY (item_id, role)
);
${execute} ctime_mtime_columns('item_read_role');

--changeset bjrke:drop_poll_last_timestamp_table
DROP TABLE poll_last_timestamp;

--changeset bjrke:rename_author_owner
ALTER TABLE change RENAME COLUMN author TO owner;

--changeset wolfgang:add_profile_hash_to_person
ALTER TABLE person ADD profile_hash ${varchar255};

--changeset max:add_active_and_blocked_to_person
ALTER TABLE person ADD active ${bool} DEFAULT 't' NOT NULL;
ALTER TABLE person ADD blocked ${bool} DEFAULT 'f' NOT NULL;

--changeset bjrke:add_deleted_column_to_person
ALTER TABLE person ADD deleted ${bool} DEFAULT 'f' NOT NULL;

--changeset bjrke:create_item
CREATE TABLE item(
    id ${varchar255} NOT NULL PRIMARY KEY,
    name ${varchar255},
    parent_item ${varchar255}
);
${execute} ctime_mtime_columns('item');

--changeset cewers:add_temp_file table
CREATE TABLE temp_file (
    id ${uuid} NOT NULL PRIMARY KEY,
    storage_id ${varchar64} NOT NULL,
    file_type ${varchar32}  DEFAULT 'UNKNOWN' NOT NULL,
    mime_type ${varchar255} default 'application/octet-stream' not null,
    file_size ${int64} NOT NULL,
    name ${varchar255} NOT NULL,
    owner ${int64} NOT NULL
);
${execute} ctime_mtime_columns('temp_file');

--changeset cewers:add_deleted_flag_to_item_table
ALTER TABLE item ADD deleted ${bool} DEFAULT 'f' NOT NULL;

--changeset bjrke:add_upload_date_to_temp_file
ALTER TABLE temp_file ADD upload_date ${timestamp} DEFAULT ${now} NOT NULL;
UPDATE temp_file SET upload_date = ctime;

--changeset max:add_share_manage_right_to_share_creator
INSERT INTO item_actions(
    item_id,
    action,
    condition,
    ctime,
    mtime)
SELECT
    DISTINCT ia.item_id,
    'SHARE_MANAGE',
    'PROFILE,'||c.owner,
    c.ctime,
    c.mtime
FROM change c
LEFT JOIN item_actions ia ON ia.item_id = c.item_id
WHERE ia.item_id LIKE 'SHARE,%'
AND c.version = 1;

--changeset max:add_description_to_files
ALTER TABLE document
ADD description ${varchar1024};

--changeset max:add_tags_to_files
ALTER TABLE document
ADD tags ${varchar4000};

--changeset wolfgang:person_add_email_column
ALTER TABLE person
ADD email ${varchar255};

--changeset bjrke:item_add_new_version_public_column
ALTER TABLE item
ADD new_version_public ${bool} DEFAULT 't' NOT NULL;

--changeset wolfgang:create_person_used_syncapp_table
CREATE TABLE person_used_syncapp (
    person_id ${int64} NOT NULL PRIMARY KEY,
    document_version_id ${uuid} NOT NULL,
    download_date ${timestamp} NOT NULL
);
${execute} ctime_mtime_columns('person_used_syncapp');

--changeset bjrke:change_add_version_info_columns
UPDATE change SET change_date = ctime WHERE change_date IS NULL;
ALTER TABLE change ADD first_version ${varchar32};
ALTER TABLE change ADD first_published_version ${varchar32};
ALTER TABLE change ADD first_published_date ${timestamp};
ALTER TABLE change ADD archived_date ${timestamp};
UPDATE change SET first_published_date = change_date;

--changeset bjrke:change_date_not_null dbms:postgresql
ALTER TABLE change ALTER change_date SET NOT NULL;

--changeset bjrke:change_date_not_null dbms:oracle
ALTER TABLE change MODIFY change_date NOT NULL;

--changeset bjrke:first_published_change
ALTER TABLE document_version ADD first_published_change_id ${uuid};
ALTER TABLE document_version ADD CONSTRAINT doc_version_first_pub_chg_fk FOREIGN KEY(first_published_change_id) REFERENCES document_version(id);
UPDATE document_version SET first_published_change_id = change_id;
ALTER TABLE document ADD last_public_version_id ${uuid};
UPDATE document d SET last_public_version_id = last_version_id;

--changeset bjrke:drop_change_archive_date
ALTER TABLE change DROP COLUMN archived_date;

--changeset bjrke:add_change_expiration_date
ALTER TABLE change ADD expiration_date ${timestamp};

--changeset bjrke:increase_item_actions_conditon dbms:postgresql
ALTER TABLE item_actions ALTER condition SET DATA TYPE TEXT;

--changeset bjrke:increase_item_actions_conditon dbms:oracle
ALTER TABLE item_actions ADD (temp CLOB);
UPDATE item_actions SET temp=condition;
ALTER TABLE item_actions DROP COLUMN condition;
ALTER TABLE item_actions RENAME COLUMN temp TO condition;

--changeset bjrke:drop_item_version_id
ALTER TABLE change DROP COLUMN item_version_id;

--changeset max:add_link_share_table
CREATE TABLE public_links (
    link_id ${varchar32} NOT NULL PRIMARY KEY,
    document_id ${uuid} NOT NULL,
    expiration_date ${timestamp},
    creator ${int64} NOT NULL,
    CONSTRAINT document_id_fk FOREIGN KEY (document_id) REFERENCES document(id)
);
${execute} ctime_mtime_columns('public_links');

--changeset bjrke:migrate_my_desktop dbms:postgresql
-- the new uuid is calculated as md5 sum of the old mydesktop item id for easier statements
-- first remove empty my desktops and their root folder
WITH
    -- empty desktop contains only a single change (it does not exists another change for the same item)
    ddv AS (
        DELETE FROM change c1
        WHERE c1.item_id LIKE 'MY_DESKTOP,%'
        AND NOT EXISTS (
            SELECT 1
            FROM change c2
            WHERE c1.item_id = c2.item_id
            AND c1.version <> c2.version
        )
        RETURNING c1.folder_version_id AS id
    ),
    -- determine the root documents
    dd AS (
        SELECT document_id FROM document_version WHERE id IN (SELECT * FROM ddv)
    ),
    -- set the last_version_ids to null to be able to delete them
    ignore1 AS (
        UPDATE document d
        SET last_version_id = null, last_public_version_id = null
        WHERE id IN (SELECT * FROM dd)
    ),
    -- delete the root folder versions
    ignore2 AS (
        DELETE FROM folder_version WHERE id IN (SELECT * FROM ddv)
    ),
    -- delete the document versions
    ignore3 AS (
        DELETE FROM document_version WHERE id IN (SELECT * FROM ddv)
    )
-- delete the documents
DELETE FROM document WHERE id IN (SELECT * FROM dd);

-- all documents need to be updated in search since their parents have changed
INSERT INTO search_index_queue (document_id, index_type)
SELECT DISTINCT(v.document_id), 'METADATA'
FROM document_version v
JOIN change c ON c.folder_version_id = v.change_id
WHERE c.item_id LIKE 'MY_DESKTOP,%';

-- the my desktop owner is the manager
INSERT INTO item_actions(item_id, action, condition)
SELECT 'SHARE,' || uuid_in(md5(c.item_id)::cstring), 'SHARE_MANAGE', OVERLAY(c.item_id placing 'PROFILE,' FROM 1 FOR 11)
FROM change c
WHERE c.item_id LIKE 'MY_DESKTOP,%'
GROUP BY c.item_id;

-- and has also write permissions
INSERT INTO item_actions(item_id, action, condition)
SELECT 'SHARE,' || uuid_in(md5(c.item_id)::cstring), 'DOCUMENT_WRITE', OVERLAY(c.item_id placing 'PROFILE,' FROM 1 FOR 11)
FROM change c
WHERE c.item_id LIKE 'MY_DESKTOP,%'
GROUP BY c.item_id;

-- and read permissions
INSERT INTO item_actions(item_id, action, condition)
SELECT 'SHARE,' || uuid_in(md5(c.item_id)::cstring), 'DOCUMENT_READ', OVERLAY(c.item_id placing 'PROFILE,' FROM 1 FOR 11)
FROM change c
WHERE c.item_id LIKE 'MY_DESKTOP,%'
GROUP BY c.item_id;

-- to be able to update the search correctly the roles who may read have to be stored too
INSERT INTO item_read_role(item_id, role)
SELECT 'SHARE,' || uuid_in(md5(c.item_id)::cstring), OVERLAY(c.item_id placing 'PROFILE,' FROM 1 FOR 11)
FROM change c
WHERE c.item_id LIKE 'MY_DESKTOP,%'
GROUP BY c.item_id;

-- create the share item with the name Desktop which is readable in english and german
INSERT INTO item(id, name)
SELECT 'SHARE,' || uuid_in(md5(c.item_id)::cstring), 'Desktop'
FROM change c
WHERE c.item_id LIKE 'MY_DESKTOP,%'
GROUP BY c.item_id;

-- connect all changes to the new shares
UPDATE change
SET item_id = 'SHARE,' || uuid_in(md5(item_id)::cstring)
WHERE item_id LIKE 'MY_DESKTOP,%';

--changeset bjrke:migrate_my_desktop dbms:oracle
-- insert my_desktop items, if they do not exist
INSERT INTO item(id)
SELECT c.item_id
FROM change c
WHERE c.item_id LIKE 'MY_DESKTOP,%'
AND NOT EXISTS (
    SELECT 1 FROM item WHERE item.id = c.item_id
)
GROUP BY c.item_id;

-- mark items as deleted
UPDATE item SET deleted = 1 WHERE id LIKE 'MY_DESKTOP,%';

-- the files are not visible so update their visibility
INSERT INTO search_index_queue (document_id, index_type)
SELECT DISTINCT(v.document_id), 'METADATA'
FROM document_version v
JOIN change c ON c.folder_version_id = v.change_id
WHERE c.item_id LIKE 'MY_DESKTOP,%';

--changeset bjrke:create_person_config
CREATE TABLE person_config (
    id ${int64} NOT NULL PRIMARY KEY,
    redirect_to_first_share ${bool} DEFAULT 'f' NOT NULL
);
${execute} ctime_mtime_columns('person_config');

--changeset max:add_password_column_to_link_share_table
ALTER TABLE public_links ADD password ${varchar255};

--changeset wolfgang:create_file_lock table
CREATE TABLE file_lock (
    document_id ${uuid} NOT NULL PRIMARY KEY,
    owner ${int64} NOT NULL,
    lock_date ${timestamp} NOT NULL,
CONSTRAINT file_lock_id_fk FOREIGN KEY (document_id) REFERENCES document(id)
);
${execute} ctime_mtime_columns('file_lock');

--changeset bjrke:fix_item_delete_flag dbms:oracle
INSERT INTO search_index_queue (document_id, index_type)
SELECT DISTINCT(v.document_id), 'METADATA'
FROM document_version v
JOIN change c ON c.folder_version_id = v.change_id
JOIN item i ON c.item_id = i.id
WHERE i.deleted = '1';

UPDATE item SET deleted = 't' WHERE deleted = '1';

--changeset miriam:create_drive_version_seq
CREATE SEQUENCE drive_version_seq;

--changeset miriam:create_drive_version dbms:postgresql
CREATE TABLE drive_version (
    id ${int64} PRIMARY KEY DEFAULT nextval('drive_version_seq'),
    version ${varchar32} NOT NULL
);
${execute} ctime_mtime_columns('drive_version');

--changeset miriam:create_drive_version dbms:oracle
CREATE TABLE drive_version (
    id ${int64} DEFAULT drive_version_seq.NEXTVAL PRIMARY KEY,
    version ${varchar32} NOT NULL
);
${execute} ctime_mtime_columns('drive_version');

--changeset miriam:create_version_update_process
CREATE TABLE version_update_process (
    id ${uuid} NOT NULL PRIMARY KEY,
    drive_version_id ${int64},
    processor_type ${varchar64} NOT NULL,
    processor_version ${int32} NOT NULL,
    status ${varchar64} NOT NULL,
    CONSTRAINT v_u_p_d_version_fk FOREIGN KEY (drive_version_id) REFERENCES drive_version(id) ON DELETE CASCADE
);
${execute} ctime_mtime_columns('version_update_process');

--changeset miriam:version_id_not_null_version_update_process dbms:postgresql
ALTER TABLE version_update_process ALTER drive_version_id SET NOT NULL;

--changeset miriam:version_id_not_null_version_update_process dbms:oracle
ALTER TABLE version_update_process MODIFY drive_version_id NOT NULL;

--changeset bjrke:create_user_group
CREATE TABLE user_group (
    id ${uuid} NOT NULL PRIMARY KEY,
    name ${varchar255} NOT NULL
);
${execute} ctime_mtime_columns('user_group');

--changeset fzuellich:create_share_permission_table dbms:oracle
CREATE TABLE share_permission (
    share_id ${varchar255} NOT NULL,
    permission ${varchar32} NOT NULL,
    role_name ${varchar64} NOT NULL
);
${execute} ctime_mtime_columns('share_permission');

--changeset fzuellich:mirgrate_share_permission_data_to_new_table dbms:oracle splitStatements:false
DECLARE
    startpos    ${int32};
    eol         ${int32};
    len         ${int32};
    rolestr     ${varchar255};
BEGIN
    FOR f IN (
        SELECT
            *
        FROM
            item_actions
        WHERE
            item_id LIKE 'SHARE,%'
    ) LOOP
        len := dbms_lob.getlength(f.condition);
        startpos := 0;
        WHILE ( startpos < len ) LOOP
            eol := startpos;
            WHILE ( eol <= len AND dbms_lob.substr(f.condition, 1, eol) != ' ' ) LOOP
                eol := eol + 1;
            END LOOP;

            rolestr := dbms_lob.substr(f.condition, eol - startpos, startpos);
            IF
                ( rolestr != 'or' )
            THEN
                INSERT INTO share_permission (
                    share_id,
                    permission,
                    role_name
                )
                    SELECT
                        f.item_id,
                        CASE
                            WHEN f.action = 'DOCUMENT_READ' THEN 'READER'
                            WHEN f.action = 'DOCUMENT_WRITE' THEN 'WRITER'
                            WHEN f.action = 'SHARE_MANAGE' THEN 'MANAGER'
                        END,
                        rolestr
                    FROM
                        dual;

            END IF;

            startpos := eol + 1;
        END LOOP;

    END LOOP;
END;

--changeset fzuellich:delete_old_share_permission_data dbms:oracle

-- Remove manager and writer from document_read
DELETE FROM share_permission o
WHERE o.permission = 'READER'
AND EXISTS (SELECT 1 FROM share_permission i WHERE i.role_name = o.role_name AND i.share_id = o.share_id AND (i.permission = 'WRITER' OR i.permission = 'MANAGER'));

-- Remove manager from writer
DELETE FROM share_permission o
WHERE o.permission = 'WRITER'
AND EXISTS (SELECT 1 FROM share_permission i WHERE i.role_name = o.role_name AND i.share_id = o.share_id AND i.permission = 'MANAGER');

-- Remove left over duplicates, e.g. when we had a item_actions.condition like 'PROFILE,1 or PROFILE,2 or PROFILE,1'
-- https://stackoverflow.com/questions/529098/removing-duplicate-rows-from-table-in-oracle#529119
DELETE FROM share_permission WHERE ROWID NOT IN (SELECT MIN(ROWID) FROM share_permission GROUP BY share_id, permission, role_name);

ALTER TABLE share_permission ADD CONSTRAINT share_permission_pk PRIMARY KEY (share_id, role_name);

DELETE FROM item_actions WHERE item_id LIKE 'SHARE,%';

--changeset fzuellich:create_share_permission_table dbms:postgresql
CREATE TABLE share_permission (
    share_id ${varchar255} NOT NULL,
    permission ${varchar32} NOT NULL,
    role_name ${varchar64} NOT NULL,
    CONSTRAINT share_permission_pk PRIMARY KEY (share_id, role_name)
);
${execute} ctime_mtime_columns('share_permission');

WITH with_permission AS (

    SELECT
        item_id AS share_id,
        CASE
            WHEN action = 'SHARE_MANAGE' THEN 'MANAGER'
            WHEN action = 'DOCUMENT_READ' THEN 'READER'
            WHEN action = 'DOCUMENT_WRITE' THEN 'WRITER'
        END AS permission,
        condition
    FROM
        item_actions
    WHERE
        item_id LIKE 'SHARE,%'

), splitted AS (

    SELECT
        share_id,
        permission,
        trim(regexp_split_to_table(condition, 'or')) AS condition
    FROM
        with_permission

)
INSERT INTO share_permission (share_id, permission, role_name)
SELECT DISTINCT
    share_id,
    permission,
    condition
FROM
    splitted AS o
WHERE
    o.permission = 'MANAGER'
    OR (o.permission = 'WRITER'
        AND NOT EXISTS (SELECT 1 FROM splitted AS i WHERE i.permission = 'MANAGER' AND i.condition = o.condition AND i.share_id = o.share_id))
    OR (o.permission = 'READER'
        AND NOT EXISTS (SELECT 1 FROM splitted AS i WHERE i.permission IN ('MANAGER', 'WRITER') AND i.condition = o.condition AND i.share_id = o.share_id));

DELETE FROM item_actions WHERE item_id LIKE 'SHARE,%';

--changeset bjrke:subfolder_primary_key dbms:postgresql
ALTER TABLE sub_folder DROP CONSTRAINT sub_folder_parent_id_child_id_key;
DROP INDEX sub_folder_parent_idx;
ALTER TABLE sub_folder ADD CONSTRAINT sub_folder_pk PRIMARY KEY (parent_id, child_id);
CREATE INDEX sub_folder_child_parent_idx ON sub_folder(child_id, parent_id);

--changeset bjrke:subfolder_primary_key dbms:oracle
ALTER TABLE sub_folder DROP UNIQUE (parent_id, child_id);
DROP INDEX sub_folder_parent_idx;
ALTER TABLE sub_folder ADD CONSTRAINT sub_folder_pk PRIMARY KEY (parent_id, child_id);
CREATE INDEX sub_folder_child_parent_idx ON sub_folder(child_id, parent_id);

--changeset bjrke:move_change_date_and_owner_to_version dbms:postgresql
ALTER TABLE document_version ADD change_date ${timestamp};
ALTER TABLE document_version ADD owner ${int64};

UPDATE document_version dv
SET change_date = c.change_date, owner = c.owner
FROM change c WHERE c.folder_version_id = dv.change_id;

ALTER TABLE document_version ALTER COLUMN change_date SET NOT NULL;
ALTER TABLE document_version ALTER COLUMN owner SET NOT NULL;

ALTER TABLE change DROP COLUMN change_date;
ALTER TABLE change DROP COLUMN owner;

--changeset bjrke:move_change_date_and_owner_to_version dbms:oracle
ALTER TABLE document_version ADD change_date ${timestamp};
ALTER TABLE document_version ADD owner ${int64};

UPDATE document_version dv SET (change_date, owner) = (SELECT c.change_date, c.owner FROM change c WHERE c.folder_version_id = dv.change_id);

ALTER TABLE document_version MODIFY (change_date NOT NULL);
ALTER TABLE document_version MODIFY (owner NOT NULL);

ALTER TABLE change DROP COLUMN change_date;
ALTER TABLE change DROP COLUMN owner;

--changeset bjrke:drop_search_index_queue_in_progress
ALTER TABLE search_index_queue DROP COLUMN in_progress;

--changeset fzuellich:add_cascade_delete_to_file_lock_and_public_link_tables
ALTER TABLE file_lock DROP CONSTRAINT file_lock_id_fk;
ALTER TABLE file_lock
ADD CONSTRAINT file_lock_id_fk
FOREIGN KEY (document_id) REFERENCES document (id) ON DELETE CASCADE;

ALTER TABLE public_links DROP CONSTRAINT document_id_fk;
ALTER TABLE public_links
ADD CONSTRAINT document_id_fk
FOREIGN KEY (document_id) REFERENCES document (id) ON DELETE CASCADE;

--changeset waller:create_pending_access_request
CREATE TABLE pending_access_request (
    person_id ${int64} NOT NULL,
    share_id ${varchar255} NOT NULL,
    CONSTRAINT person_id_fk FOREIGN KEY (person_id) REFERENCES person(id),
    CONSTRAINT share_id_fk FOREIGN KEY (share_id) REFERENCES item(id),
    CONSTRAINT access_request_pk PRIMARY KEY (person_id, share_id)

);
${execute} ctime_mtime_columns('pending_access_request');
CREATE INDEX person_id_idx ON pending_access_request(person_id);
CREATE INDEX share_id_idx ON pending_access_request(share_id);

--changeset fzuellich:add_share_subscription_table
CREATE TABLE share_subscription (
    share_id ${varchar255} NOT NULL,
    person_id ${int64} NOT NULL,
    CONSTRAINT share_subscription_pk PRIMARY KEY (share_id, person_id),
    CONSTRAINT share_subscription_share_fk FOREIGN KEY (share_id) REFERENCES item(id),
    CONSTRAINT share_subscription_person_fk FOREIGN KEY (person_id) REFERENCES person(id)
);
${execute} ctime_mtime_columns('share_subscription');

--changeset fzuellich:remove_person_config_table
DROP TABLE person_config;

--changeset azottnick:remove_file_type_column dbms:postgresql
ALTER TABLE file_version DROP file_type;
ALTER TABLE temp_file DROP file_type;

--changeset azottnick:remove_file_type_column dbms:oracle
ALTER TABLE file_version DROP COLUMN file_type;
ALTER TABLE temp_file DROP COLUMN file_type;

--changeset fzuellich:create_document_version_count_table
CREATE TABLE document_version_count (
  id ${uuid},
  count ${int32},
  CONSTRAINT document_version_count_pk PRIMARY KEY (id)
);
${execute} ctime_mtime_columns('document_version_count');

--changeset fzuellich:rectify_expiration_date_for_public_links
UPDATE public_links
SET expiration_date = current_timestamp + interval '4' MONTH
WHERE
      expiration_date IS null
      OR expiration_date > current_timestamp + interval '4' MONTH;

--changeset fzuellich:make_expiration_date_non_null dbms:postgresql
ALTER TABLE public_links ALTER expiration_date SET NOT NULL;

--changeset fzuellich:make_expiration_date_non_null dbms:oracle
ALTER TABLE public_links MODIFY expiration_date NOT NULL;

--changeset fzuellich:set_images_to_present
UPDATE file_version_preview p
   SET preview_status = 'PRESENT'
 WHERE preview_status = 'UNSUPPORTED'
       AND EXISTS (SELECT 1
                     FROM file_version f
                    WHERE f.storage_id = p.storage_id
                      AND f.mime_type LIKE 'image/%');

--changeset fzuellich:retrigger_preview_generation_for_unsupported_videos
DELETE FROM file_version_preview p
WHERE preview_status = 'UNSUPPORTED'
  AND EXISTS (SELECT 1
              FROM file_version f
              WHERE f.storage_id = p.storage_id
                AND f.mime_type LIKE 'video/%');

-- changeset jkaiser:add_indexes
CREATE INDEX item_parent_item_idx ON item(parent_item);
CREATE INDEX document_version_owner_idx ON document_version(owner,id);
CREATE INDEX search_idx_queue_idx_type_idx ON search_index_queue(index_type);

-- changeset jkaiser:add_index_oracle dbms:oracle
CREATE INDEX doc_ver_fst_pub_change_id_idx ON document_version(first_published_change_id,id);

-- changeset fzuellich:add_missing_document_version_index_for_postgres dbms:postgresql
CREATE INDEX doc_ver_fst_pub_change_id_idx ON document_version(first_published_change_id,id);

-- changeset tsubklewe:migrate_wiki_ids
update item set id  = REPLACE(id, 'WIKI_CONTAINER,', 'WIKI_WIKI,') where id like 'WIKI_CONTAINER,%';
update item set parent_item  = REPLACE(parent_item, 'WIKI_CONTAINER,', 'WIKI_WIKI,') where parent_item like 'WIKI_CONTAINER,%';
update change set item_id  = REPLACE(item_id, 'WIKI_CONTAINER,', 'WIKI_WIKI,') where item_id like 'WIKI_CONTAINER,%';

-- changeset fzuellich:extend_user_group_with_tenant_information
TRUNCATE TABLE user_group;
ALTER TABLE user_group ADD is_all_users_group ${bool} NOT NULL;
ALTER TABLE user_group ADD tenant_id ${uuid} NOT NULL;
ALTER TABLE user_group ADD CONSTRAINT all_users_group_tenant_uq UNIQUE (is_all_users_group, tenant_id);

-- changeset fzuellich:remove_user_group_unique_constraint
ALTER TABLE user_group DROP CONSTRAINT all_users_group_tenant_uq;

-- changeset fzuellich:add_person_config_table
CREATE TABLE person_config (
    person_id ${int64} NOT NULL PRIMARY KEY,
    document_list_sort_params ${varchar255} NOT NULL,
    CONSTRAINT person_config_person_id_fk FOREIGN KEY (person_id) REFERENCES person(id) ON DELETE CASCADE
);

-- changeset cewers:delete_profile_item_actions
DELETE FROM item_actions WHERE item_id LIKE 'PROFILE,%';

-- changeset tsubklewe:remove_parent_item_ids_when_cyclic
UPDATE item SET parent_item = null WHERE parent_item = id AND (parent_item LIKE 'WIKI_WIKI,%' OR parent_item LIKE 'TENANT,%');

-- changeset maxse:add_timestamps_to_person_config
${execute} ctime_mtime_columns('person_config');

-- changeset christian.ewers:drop_conversation_visibility_table
DROP TABLE conversation_visibility;

-- changeset christian.ewers:mark_all_newsblog_article_items_deleted
DELETE FROM item WHERE id LIKE 'NEWSBLOG_ARTICLE,%';
INSERT INTO item (id) SELECT DISTINCT(item_id) FROM change WHERE item_id LIKE 'NEWSBLOG_ARTICLE,%';
UPDATE item SET deleted='t' WHERE id LIKE 'NEWSBLOG_ARTICLE,%';

-- changeset argo.liivanurm:add_tenant_id_to_item
ALTER TABLE item ADD tenant_id ${uuid};

--changeset argo.liivanurm:person_add_tenant_id
ALTER TABLE person ADD tenant_id ${uuid};

--changeset argo.liivanurm:add_tenant_id_to_shares dbms:postgresql
UPDATE item SET tenant_id = (SELECT tenant_id FROM person WHERE tenant_id IS NOT NULL LIMIT 1)
    WHERE item.id LIKE 'SHARE%' AND tenant_id IS NULL AND deleted = 'f';

--changeset argo.liivanurm:add_tenant_id_to_shares dbms:oracle
UPDATE item SET tenant_id = (SELECT tenant_id FROM person WHERE tenant_id IS NOT NULL FETCH NEXT 1 ROWS ONLY)
    WHERE item.id LIKE 'SHARE%' AND tenant_id IS NULL AND deleted = 'f';

--changeset christian.ewers:fix_unpublished_news_items
UPDATE document set last_public_version_id=last_version_id
    WHERE id in (select document_id from document_version
        WHERE change_id IN (select folder_version_id from change
            WHERE item_id IN (select id from item where deleted='f' and (id LIKE 'POST,%' or id LIKE 'COMMENT,%')
    ))) AND last_public_version_id IS NULL;

UPDATE document_version set first_published_change_id=change_id
    WHERE change_id in (select folder_version_id from change
    WHERE item_id IN (select id from item where deleted='f' and (id LIKE 'POST,%' or id LIKE 'COMMENT,%')))
    AND first_published_change_id is null;

UPDATE change set first_published_date=ctime
    WHERE item_id IN (select id from item where deleted='f' and (id LIKE 'POST,%' or id LIKE 'COMMENT,%'))
    AND first_published_date is null;

UPDATE item set new_version_public='t'
    WHERE id IN (select id from item WHERE deleted='f' and (id LIKE 'POST,%' or id LIKE 'COMMENT,%'));

--changeset christian.ewers:delete_all_document_attached_to_profiles dbms:postgresql
INSERT INTO item (id, deleted) (SELECT DISTINCT(item_id), true FROM change WHERE item_id LIKE'PROFILE,%') ON CONFLICT(id) DO UPDATE SET deleted=true;

--changeset christian.ewers:cascade_delete_pending_access_request dbms:postgresql
ALTER TABLE pending_access_request DROP CONSTRAINT share_id_fk;
ALTER TABLE pending_access_request ADD CONSTRAINT share_id_fk FOREIGN KEY(share_id) REFERENCES item(id) ON DELETE CASCADE;

--changeset maxse:create_settings_table
CREATE TABLE settings (
    tenant_id ${uuid} PRIMARY KEY,
    public_links_enabled ${bool} DEFAULT 'f' NOT NULL
);
${execute} ctime_mtime_columns('settings');

--changeset maxse:add_tenant_id_to_public_links dbms:postgresql
-- Step 1: Add a nullable tenant_id column to public_links
ALTER TABLE public_links
    ADD COLUMN tenant_id ${uuid} NULL;

-- Step 2: Update the tenant_id for all public_links by looking up the tenant_id
-- on either the creator of the public link or the item it links to. tenant_id on person
-- is nullable and not all public_links point to an item, so we need to coalesce here.
WITH tenant_data AS (
    SELECT pl.link_id AS public_link_id, COALESCE(p.tenant_id, i.tenant_id) AS tenant_id
    FROM public_links pl
             JOIN person p ON pl.creator = p.id
             LEFT JOIN document d ON pl.document_id = d.id
             LEFT JOIN document_version dv ON d.last_public_version_id = dv.id
             LEFT JOIN change c ON dv.change_id = c.folder_version_id
             LEFT JOIN item i ON c.item_id = i.id
)
UPDATE public_links
SET tenant_id = tenant_data.tenant_id
FROM tenant_data
WHERE public_links.link_id = tenant_data.public_link_id;

-- Step 3: Delete public_links where tenant_id is still null
-- We might lose some data here, but we don't want any public links that we can't link to a tenant
DELETE FROM public_links
WHERE tenant_id IS NULL;

-- Step 4: Make the tenant_id column non-nullable
ALTER TABLE public_links
    ALTER COLUMN tenant_id SET NOT NULL;


--changeset maxse:add_tenant_id_to_public_links dbms:oracle
-- see postgresl migration for more explanation
ALTER TABLE public_links
    ADD tenant_id ${uuid};

MERGE INTO public_links pl
USING (
    SELECT pl.link_id AS public_link_id, COALESCE(p.tenant_id, i.tenant_id) AS tenant_id
    FROM public_links pl
             JOIN person p ON pl.creator = p.id
             LEFT JOIN document d ON pl.document_id = d.id
             LEFT JOIN document_version dv ON d.last_public_version_id = dv.id
             LEFT JOIN change c ON dv.change_id = c.folder_version_id
             LEFT JOIN item i ON c.item_id = i.id
) tenant_data
ON (pl.link_id = tenant_data.public_link_id)
WHEN MATCHED THEN
    UPDATE SET pl.tenant_id = tenant_data.tenant_id;

DELETE FROM public_links
WHERE tenant_id IS NULL;

ALTER TABLE public_links MODIFY tenant_id ${uuid} NOT NULL;

--changeset christian.ewers:delete_search_index_queue
DROP table search_index_queue;

--changeset christian.ewers:make_old_entity_files_visible_to_superadmin
-- files in entities are authorized via the item_actions table. Just make all documents that are still visible accessable for superadmin only
UPDATE item_actions set condition='SUPERADMIN';

--changeset argo.liivanurm:delete_all_channel_items
UPDATE item SET deleted = 't' WHERE id LIKE 'CHANNEL,%';

--changeset maxse:add_video_teaser_id
alter table temp_file add video_teaser_id uuid default null;
alter table file_version add video_teaser_id uuid default null;

--changeset christian.ewers:mark_all_legacy_data_as_deleted
UPDATE item SET deleted='t' WHERE id LIKE 'ENTITY,%';
UPDATE item SET deleted='t' WHERE id LIKE 'WORKSTREAM_MESSAGE,%';
UPDATE item SET deleted='t' WHERE id LIKE 'PROFILE,%';
-- missed to drop sequence in older changeset
DROP sequence if exists search_index_queue_seq;

--changeset maxse:change_video_teaser_id_to_varchar
update temp_file set video_teaser_id = null;
update file_version set video_teaser_id = null;

alter table temp_file
    alter column video_teaser_id type varchar(64);
alter table file_version
    alter column video_teaser_id type varchar(64);

--changeset christian.ewers:drop_entity_related_stuff
DROP TABLE item_actions;
DROP TABLE item_read_role;
DROP TABLE person_role;

--changeset christian.ewers:drop_column_new_version_public
ALTER TABLE item DROP COLUMN new_version_public;
ALTER TABLE document DROP COLUMN last_public_version_id;

--changeset argo.liivanurm:drop_column_parent_item
ALTER TABLE item DROP COLUMN parent_item;

--changeset argo.liivanurm:add_column_antivirus_enabled
--validCheckSum: 9:f19d07e5d0084f5cdaafdf96ff95a246
ALTER TABLE settings ADD COLUMN antivirus_enabled ${bool} DEFAULT 'f' NOT NULL;

--changeset argo.liivanurm:create_file_version_virus_scan
CREATE TABLE file_version_virus_scan (
    id UUID NOT NULL PRIMARY KEY,
    file_version_id UUID NOT NULL,
    virus_scan_status VARCHAR(64) NOT NULL,
    CONSTRAINT file_version_virus_scan_fk FOREIGN KEY (file_version_id) REFERENCES file_version(id)
);
${execute} ctime_mtime_columns('file_version_virus_scan');

--changeset argo.liivanurm:recreate_virus_scan_with_storage_id_instead_of_file_version_id
DROP TABLE file_version_virus_scan;

CREATE TABLE virus_scan_status (
    storage_id VARCHAR(64) NOT NULL PRIMARY KEY,
    status VARCHAR(64) NOT NULL,
    scan_date TIMESTAMP NULL
);
${execute} ctime_mtime_columns('virus_scan_status');

--changeset christian.ewers:make_constraints_deferable
ALTER TABLE change
    ALTER CONSTRAINT change_folder_version_id_fk DEFERRABLE INITIALLY DEFERRED;

--changeset christian.ewers:set_tenant_id_for_tenant_items runOnChange:true
INSERT INTO item(id, name, tenant_id)(
    SELECT distinct(item_id) as item_id,'',split_part(item_id, ',', 2)::uuid as tenant_id
        FROM change c WHERE item_id LIKE 'TENANT,%'
) ON CONFLICT(id) DO UPDATE set tenant_id =excluded.tenant_id where item.tenant_id is null;
-- update existing items without changes
UPDATE item set tenant_id=split_part(id, ',', 2)::uuid where item.tenant_id IS NULL AND id like 'TENANT,%';

--changeset argo.liivanurm:add_column_create_date_to_virus_scan_status
ALTER TABLE virus_scan_status ADD COLUMN create_date TIMESTAMP NOT NULL DEFAULT current_timestamp;

--changeset argo.liivanurm:add_all_storage_ids_to_virus_scan_status_newest_file_version_first
INSERT INTO virus_scan_status (storage_id, status, scan_date, create_date)
    SELECT distinct on (fv.storage_id) fv.storage_id, 'NOT_SCANNED', null, fv.ctime
    FROM file_version fv
    WHERE NOT EXISTS (SELECT * FROM virus_scan_status vss WHERE fv.storage_id = vss.storage_id)
    ORDER BY fv.storage_id, fv.ctime DESC;

--changeset argo.liivanurm:add_columns_status_update_by_and_at_to_virus_scan_status
ALTER TABLE virus_scan_status ADD COLUMN status_updated_by VARCHAR(255) NULL;
ALTER TABLE virus_scan_status ADD COLUMN status_updated_at TIMESTAMP NULL;

--changeset argo.liivanurm:clean_up_virus_scan_status_if_storage_id_is_deleted
DELETE FROM virus_scan_status v WHERE NOT EXISTS (SELECT 1 FROM file_version fv WHERE v.storage_id = fv.storage_id)
                                  AND NOT EXISTS (SELECT 1 FROM temp_file tf WHERE tf.storage_id = v.storage_id);
