--liquibase formatted sql

--changeset tobias.subklewe:create_initial_tables
CREATE TABLE wiki (
    id ${uuid} NOT NULL PRIMARY KEY,
    name ${varchar1024} NOT NULL,
    description ${text} NOT NULL,
    tenant_id ${uuid} NOT NULL,
    created_at ${timestamp} NOT NULL,
    updated_at ${timestamp} NOT NULL
);

CREATE TABLE permission (
    id ${uuid} NOT NULL PRIMARY KEY,
    wiki_id ${uuid} NOT NULL,
    permission_type ${varchar1024} NOT NULL,
    grantee_id ${varchar1024} NOT NULL,
    created_at ${timestamp} NOT NULL,
    updated_at ${timestamp} NOT NULL
);

CREATE TABLE article (
    id ${uuid} NOT NULL PRIMARY KEY,
    name ${varchar1024} NOT NULL,
    content ${text} NOT NULL,
    wiki_id ${uuid} NOT NULL,
    parent_id ${uuid} NOT NULL,
    parent_type ${varchar1024} NOT NULL,
    created_at ${timestamp} NOT NULL,
    updated_at ${timestamp} NOT NULL
);

CREATE TABLE chapter (
    id ${uuid} NOT NULL PRIMARY KEY,
    name ${varchar1024} NOT NULL,
    wiki_id ${uuid} NOT NULL,
    parent_id ${uuid} NOT NULL,
    parent_type ${varchar1024} NOT NULL,
    created_at ${timestamp} NOT NULL,
    updated_at ${timestamp} NOT NULL
);

--changeset tobias.subklewe:rename_permission_table_columns
ALTER TABLE permission RENAME COLUMN permission_type to type;
ALTER TABLE permission RENAME COLUMN grantee_id to grantee;
CREATE UNIQUE INDEX permission_wiki_id_role_index on permission (wiki_id, grantee);
CREATE INDEX permission_wiki_id_index ON permission(wiki_id);

--changeset vinh.phu.phan:add_constraints
CREATE INDEX chapter_parent_id_idx ON chapter(parent_id);
CREATE INDEX article_parent_id_idx on article (parent_id);
ALTER TABLE article ADD CONSTRAINT fk_article_wiki_id FOREIGN KEY (wiki_id) REFERENCES wiki(id);
ALTER TABLE chapter ADD CONSTRAINT fk_chapter_wiki_id FOREIGN KEY (wiki_id) REFERENCES wiki(id);
ALTER TABLE permission ADD CONSTRAINT fk_permission_wiki_id FOREIGN KEY (wiki_id) REFERENCES wiki(id);

--changeset alexander.zottnick:rename_name_to_title
ALTER TABLE wiki RENAME COLUMN name TO title;
ALTER TABLE article RENAME COLUMN name TO title;
ALTER TABLE chapter RENAME COLUMN name TO title;

--changeset tobias.subklewe:create_article_version_table
CREATE TABLE article_version (
    id ${uuid} NOT NULL PRIMARY KEY,
    article_id ${uuid},
    title ${varchar1024} NOT NULL,
    content ${text} NOT NULL,
    created_at ${timestamp} NOT NULL,
    updated_at ${timestamp} NOT NULL
);

CREATE INDEX article_version_article_id_idx ON article_version(article_id);

INSERT INTO article_version (id, article_id, title, content, created_at, updated_at)
SELECT id, id, title, content, created_at, updated_at
FROM article;

ALTER TABLE article DROP COLUMN title;
ALTER TABLE article DROP COLUMN content;

ALTER TABLE article ADD latest_version ${uuid};
UPDATE article SET latest_version = id;

--changeset max.semmler:create_subscription_table
CREATE TABLE subscription (
    item_id ${uuid} NOT NULL,
    item_type ${varchar1024} NOT NULL,
    subscriber ${varchar1024} NOT NULL,
    is_subscribed ${bool} NOT NULL,
    created_at ${timestamp} NOT NULL,
    updated_at ${timestamp} NOT NULL,
    PRIMARY KEY(item_id, subscriber)
);

--changeset vinh.phu.phan:create_profile_table
CREATE TABLE profile (
    id ${int64} NOT NULL PRIMARY KEY,
    email ${varchar1024} NOT NULL,
    fullname ${varchar1024} NOT NULL,
    firstname ${varchar255} NOT NULL,
    lastname ${varchar255} NOT NULL,
    image_url ${varchar4000} NOT NULL,
    profile_url ${varchar4000} NOT NULL,
    active ${bool} NOT NULL,
    blocked ${bool} NOT NULL,
    deleted ${bool} NOT NULL,
    created_at ${timestamp} NOT NULL,
    updated_at ${timestamp} NOT NULL
);
--changeset tobias.subklewe:add_text_content_postgres dbms:postgresql
ALTER TABLE article_version ADD text_content ${text};
UPDATE article_version SET text_content = content;
ALTER TABLE article_version ALTER COLUMN text_content SET NOT NULL;

--changeset tobias.subklewe:add_text_content_oracle dbms:oracle
ALTER TABLE article_version ADD text_content ${text};
UPDATE article_version SET text_content = content;
ALTER TABLE article_version MODIFY text_content NOT NULL;

--changeset vinh.phu.phan:migrate_subscriber_to_profile_id
ALTER TABLE subscription RENAME TO subscription_backup;
CREATE TABLE subscription (
    item_id ${uuid} NOT NULL,
    item_type ${varchar1024} NOT NULL,
    subscriber ${int64} NOT NULL,
    is_subscribed ${bool} NOT NULL,
    created_at ${timestamp} NOT NULL,
    updated_at ${timestamp} NOT NULL,
    PRIMARY KEY(item_id, subscriber)
);
INSERT INTO subscription(item_id, item_type, subscriber, is_subscribed, created_at, updated_at)
    SELECT
        item_id,
        item_type,
        CAST(replace(subscriber, 'PROFILE,', '') AS ${int64}),
        is_subscribed,
        created_at,
        updated_at
    FROM subscription_backup;
DROP TABLE subscription_backup;

--changeset tobias.subkelwe:add_modifiedby_postgres dbms:postgresql
ALTER TABLE wiki ADD modified_by ${int64};
UPDATE wiki SET modified_by = 0;
ALTER TABLE wiki ALTER COLUMN modified_by SET NOT NULL;

ALTER TABLE chapter ADD modified_by ${int64};
UPDATE chapter SET modified_by = 0;
ALTER TABLE chapter ALTER COLUMN modified_by SET NOT NULL;

ALTER TABLE article_version ADD modified_by ${int64};
UPDATE article_version SET modified_by = 0;
ALTER TABLE article_version ALTER COLUMN modified_by SET NOT NULL;

--changeset tobias.subkelwe:add_modifiedby_oracle dbms:oracle
ALTER TABLE wiki ADD modified_by ${int64};
UPDATE wiki SET modified_by = 0;
ALTER TABLE wiki MODIFY modified_by NOT NULL;

ALTER TABLE chapter ADD modified_by ${int64};
UPDATE chapter SET modified_by = 0;
ALTER TABLE chapter MODIFY modified_by NOT NULL;

ALTER TABLE article_version ADD modified_by ${int64};
UPDATE article_version SET modified_by = 0;
ALTER TABLE article_version MODIFY modified_by NOT NULL;

--changeset vinh.phu.phan:create_user_group_table
CREATE TABLE user_group (
    id ${uuid} NOT NULL PRIMARY KEY,
    name ${varchar255} NOT NULL,
    created_at ${timestamp} NOT NULL,
    updated_at ${timestamp} NOT NULL
);

--changeset maximilian.tschoepe:added_foreign_key_for_articleVersion
ALTER TABLE article_version ADD CONSTRAINT fk_article_version_article_id FOREIGN KEY (article_id) REFERENCES article(id);

--changeset vinh.phu.phan:drop_lastest_version_from_aticle_table
ALTER TABLE article DROP COLUMN latest_version;
CREATE UNIQUE INDEX av_created_at_article_id_idx on article_version(article_id, created_at);
CREATE INDEX av_created_at_idx ON article_version(created_at);

--changeset vinh.phu.phan:set_not_null_on_article_version_aticle_id dbms:postgresql
ALTER TABLE article_version ALTER COLUMN article_id SET NOT NULL;

--changeset vinh.phu.phan:set_not_null_on_article_version_aticle_id dbms:oracle
ALTER TABLE article_version MODIFY article_id NOT NULL;

--changeset tobias.subklewe:prefix_ids
UPDATE article SET parent_type = 'WIKI_WIKI' WHERE parent_type = 'WIKI';
UPDATE article SET parent_type = 'WIKI_CHAPTER' WHERE parent_type = 'CHAPTER';
UPDATE chapter SET parent_type = 'WIKI_WIKI' WHERE parent_type = 'WIKI';
UPDATE chapter SET parent_type = 'WIKI_CHAPTER' WHERE parent_type = 'CHAPTER';

--changeset tobias.subklewe:prefix_subscription_type
UPDATE subscription SET item_type = 'WIKI_WIKI' WHERE item_type = 'WIKI';

--changeset sebastian.fleischer:add_editing_started_at_and_editing_started_by_to_article_table
ALTER TABLE article ADD editing_started_at ${timestamp};
ALTER TABLE article ADD editing_started_by ${int64};

--changeset tsubklewe:set_null_on_profile_fields dbms:postgresql
ALTER TABLE profile ALTER COLUMN email DROP NOT NULL;
ALTER TABLE profile ALTER COLUMN profile_url DROP NOT NULL;

--changeset tsubklewe:set_null_on_profile_fields dbms:oracle
ALTER TABLE profile MODIFY email NULL;
ALTER TABLE profile MODIFY profile_url NULL;

--changeset msemmler:add_is_all_tenant_users_group_on_user_group
DELETE FROM user_group;
ALTER TABLE user_group ADD is_all_tenant_users_group ${bool} DEFAULT ${false} NOT NULL;
ALTER TABLE user_group ADD tenant_id ${uuid} NOT NULL;

--changeset mtschoepe:add_new_deletion_table
CREATE TABLE deletion (
    id ${uuid} NOT NULL PRIMARY KEY,
    item_id ${uuid} NOT NULL,
    item_type ${varchar1024} NOT NULL,
    deleted_at ${timestamp} NOT NULL,
    deleted_by ${int64} NOT NULL
);

ALTER TABLE article ADD deletion_id ${uuid} DEFAULT NULL;
ALTER TABLE article ADD CONSTRAINT fk_article_deletion_id FOREIGN KEY (deletion_id) REFERENCES deletion(id);
CREATE INDEX article_deletion_id_idx ON article(deletion_id);

ALTER TABLE chapter ADD deletion_id ${uuid} DEFAULT NULL;
ALTER TABLE chapter ADD CONSTRAINT fk_chapter_deletion_id FOREIGN KEY (deletion_id) REFERENCES deletion(id);
CREATE INDEX chapter_deletion_id_idx ON chapter(deletion_id);

ALTER TABLE wiki ADD deletion_id ${uuid} DEFAULT NULL;
ALTER TABLE wiki ADD CONSTRAINT fk_wiki_deletion_id FOREIGN KEY (deletion_id) REFERENCES deletion(id);
CREATE INDEX wiki_deletion_id_idx ON wiki(deletion_id);

--changeset tsubklewe:make_oracle_save_empty_articles dbms:oracle
--Oracle saves empty strings as NULL as it uses CLOB internally
ALTER TABLE article_version MODIFY text_content NULL;

--changeset sfleischer:add_relative_position
ALTER TABLE article ADD relative_position ${int32} DEFAULT -1 NOT NULL;
ALTER TABLE chapter ADD relative_position ${int32} DEFAULT -1 NOT NULL;

--changeset sfleischer:migrate_position dbms:postgresql
UPDATE article
SET relative_position = relative_positions.pos
FROM (SELECT articles_and_chapters.id, RANK() OVER (PARTITION BY articles_and_chapters.parent_id ORDER BY articles_and_chapters.title) - 1 AS pos
    FROM (
        SELECT a.id, a.parent_id, av.title FROM article a
        INNER JOIN article_version av ON a.id = av.article_id
        WHERE av.created_at IN (SELECT MAX(al.created_at) FROM article_version al WHERE al.article_id = av.article_id)
        UNION
        SELECT c.id, c.parent_id, c.title FROM chapter c
    ) articles_and_chapters) relative_positions
WHERE article.id = relative_positions.id;

UPDATE chapter
SET relative_position = relative_positions.pos
FROM (SELECT articles_and_chapters.id, RANK() OVER (PARTITION BY articles_and_chapters.parent_id ORDER BY articles_and_chapters.title) - 1 AS pos
    FROM (
        SELECT a.id, a.parent_id, av.title FROM article a
        INNER JOIN article_version av ON a.id = av.article_id
        WHERE av.created_at IN (SELECT MAX(al.created_at) FROM article_version al WHERE al.article_id = av.article_id)
        UNION
        SELECT c.id, c.parent_id, c.title FROM chapter c
    ) articles_and_chapters) relative_positions
WHERE chapter.id = relative_positions.id;

--changeset sfleischer:migrate_position dbms:oracle
MERGE INTO article a
USING (SELECT articles_and_chapters.id, RANK() OVER (PARTITION BY articles_and_chapters.parent_id ORDER BY articles_and_chapters.title) - 1 AS pos
           FROM (
               SELECT a.id, a.parent_id, av.title FROM article a
               INNER JOIN article_version av ON a.id = av.article_id
               WHERE av.created_at IN (SELECT MAX(al.created_at) FROM article_version al WHERE al.article_id = av.article_id)
               UNION
               SELECT c.id, c.parent_id, c.title FROM chapter c
           ) articles_and_chapters) relative_positions
ON (a.id = relative_positions.id)
WHEN MATCHED THEN UPDATE SET a.relative_position = relative_positions.pos;

MERGE INTO chapter c
USING (SELECT articles_and_chapters.id, RANK() OVER (PARTITION BY articles_and_chapters.parent_id ORDER BY articles_and_chapters.title) - 1 AS pos
           FROM (
               SELECT a.id, a.parent_id, av.title FROM article a
               INNER JOIN article_version av ON a.id = av.article_id
               WHERE av.created_at IN (SELECT MAX(al.created_at) FROM article_version al WHERE al.article_id = av.article_id)
               UNION
               SELECT c.id, c.parent_id, c.title FROM chapter c
           ) articles_and_chapters) relative_positions
ON (c.id = relative_positions.id)
WHEN MATCHED THEN UPDATE SET c.relative_position = relative_positions.pos;

--changeset maximilian.tschoepe:rename_profile_columns
ALTER TABLE profile RENAME COLUMN fullname to full_name;
ALTER TABLE profile RENAME COLUMN firstname to first_name;
ALTER TABLE profile RENAME COLUMN lastname to last_name;

--changeset mtschoepe:add_contactPersonId_to_article
ALTER TABLE article ADD contact_person ${int64} DEFAULT NULL;

--changeset mtschoepe:add_contactPersonId_to_wiki
ALTER TABLE wiki ADD contact_person ${int64} DEFAULT NULL;

--changeset tobias.subklewe:rename_permission_table_columns_again
ALTER TABLE permission RENAME COLUMN grantee to grantee_id;

--changeset tobias.subklewe:cleanup_unused_permissions dbms:postgresql
delete
from permission
where id in (SELECT permission.id
             from permission
                      left join user_group on cast(substr(grantee_id, length('USER_GROUP,') + 1) as ${uuid}) = user_group.id
             where grantee_id like 'USER_GROUP,%'
               and user_group.id is null);

delete
from permission
where id in (SELECT permission.id
             from permission
                      left join profile on cast(substr(grantee_id, length('PROFILE,') + 1) as ${int64}) = profile.id
             where grantee_id like 'PROFILE,%'
               and profile.deleted = true);

--changeset tobias.subklewe:cleanup_unused_permissions dbms:oracle
delete
from permission
where id in (SELECT permission.id
             from permission
                      left join user_group on UPPER(REPLACE(SUBSTR(grantee_id, LENGTH('USER_GROUP,') + 1), '-', '')) = user_group.id
             where grantee_id like 'USER_GROUP,%'
               and user_group.id is null);

delete
from permission
where id in (SELECT permission.id
             from permission
                      left join profile on CAST(SUBSTR(grantee_id, LENGTH('PROFILE,') + 1) as ${int64}) = profile.id
             where grantee_id like 'PROFILE,%'
               and profile.deleted = ${true});

--changeset tsubklewe:add_drive_file_migration_table
CREATE TABLE drive_file_migration (
    old_file_id ${varchar1024} NOT NULL PRIMARY KEY,
    new_file_id ${varchar1024},
    article_id ${uuid} NOT NULL,
    created_at ${timestamp} NOT NULL,
    updated_at ${timestamp} NOT NULL
);

CREATE INDEX drive_file_migration_article_id_index ON drive_file_migration(article_id);

--changeset tsubklewe:add_migration_status_table
CREATE TABLE migration_status (
    migration ${varchar1024} NOT NULL PRIMARY KEY,
    status ${varchar1024} NOT NULL,
    created_at ${timestamp} NOT NULL,
    updated_at ${timestamp}
);

INSERT INTO migration_status (migration, status, created_at) VALUES ('DRIVE_FILES', 'STARTED', ${now})

--changeset tsubklewe:set_migration_start_to_now
UPDATE migration_status SET created_at = ${now} WHERE migration = 'DRIVE_FILES';

--changeset tsubklewe:add_service_config_table
CREATE TABLE config (
    config_type ${varchar1024} NOT NULL,
    tenant_id ${uuid} NOT NULL,
    config_value ${varchar1024},
    created_at ${timestamp} NOT NULL,
    updated_at ${timestamp} NOT NULL,
    PRIMARY KEY(config_type, tenant_id)
);

--changeset tsubklewe:add_sort_table
CREATE TABLE wiki_overview (
    id ${int64} NOT NULL PRIMARY KEY,
    sort ${varchar1024},
    direction ${varchar1024},
    display ${varchar1024},
    created_at ${timestamp} NOT NULL,
    updated_at ${timestamp} NOT NULL
);

--changeset tsubklewe:drop_migration
DROP TABLE migration_status;

--changeset azottnick:add_heading_migration_status_table
CREATE TABLE article_heading_migration_status (
    article_id ${uuid} NOT NULL,
    status ${varchar1024} NOT NULL
);

CREATE INDEX article_heading_migration_status_status_index ON article_heading_migration_status(status);

--changeset azottnick:insert_article_versions_to_migration_table
INSERT INTO article_heading_migration_status (article_id, status)
SELECT article_id, 'PENDING' FROM article_version a WHERE a.created_at IN (SELECT max(al.created_at) FROM article_version al WHERE al.article_id = a.article_id);

--changeset azottnick:add_article_heading_table
CREATE TABLE article_heading (
    id ${uuid} NOT NULL,
    article_id ${uuid} NOT NULL,
    text ${varchar1024} NOT NULL,
    PRIMARY KEY(id, article_id),
    created_at ${timestamp} NOT NULL,
    updated_at ${timestamp} NOT NULL
)

--changeset azottnick:reset_article_heading_migration
TRUNCATE TABLE article_heading_migration_status;
TRUNCATE TABLE article_heading;

--changeset azottnick:add_order_to_article_heading_table
ALTER TABLE article_heading ADD sort_order ${int32} NOT NULL;

--changeset azottnick:reinsert_article_versions_to_migration_table
INSERT INTO article_heading_migration_status (article_id, status)
SELECT article_id, 'PENDING' FROM article_version a WHERE a.created_at IN (SELECT max(al.created_at) FROM article_version al WHERE al.article_id = a.article_id);

--changeset azottnick:add_draft_flag_to_article_version
ALTER TABLE article_version ADD is_draft ${bool} DEFAULT ${false} NOT NULL;
CREATE INDEX article_version_is_draft_index ON article_version(is_draft);
CREATE INDEX article_version_article_id_is_draft_index ON article_version(article_id, is_draft);

--changeset sward:set_wiki_title_collation_to_german
ALTER TABLE wiki ALTER COLUMN title SET DATA TYPE ${varchar1024} COLLATE "de-DE-x-icu";

--changeset vinh.phu.phan:drop_contact_person_column
ALTER TABLE article DROP COLUMN contact_person;
ALTER TABLE wiki DROP COLUMN contact_person;

--changeset vinh.phu.phan:add_translation_tables
CREATE TABLE translation (
    id ${uuid} NOT NULL PRIMARY KEY,
    item_id ${varchar1024} NOT NULL,
    field_id ${varchar1024} NOT NULL,
    language ${varchar32} NOT NULL,
    text ${text} NOT NULL,
    created_at ${timestamp} NOT NULL,
    updated_at ${timestamp} NOT NULL
);
CREATE UNIQUE INDEX translation_item_id_field_id_language_index on translation (item_id, field_id, language);
CREATE INDEX translation_item_id_index ON translation(item_id);
CREATE INDEX translation_item_id_language_index ON translation(item_id, language);

CREATE TABLE original_language (
    item_id ${varchar1024} NOT NULL PRIMARY KEY,
    language ${varchar32} NOT NULL,
    created_at ${timestamp} NOT NULL,
    updated_at ${timestamp} NOT NULL
);
