#!/bin/sh

cleanUpAndExit () {
    echo "[Clean up] Killing container '$DOCKER_CONVERTER_NAME' ..."
    docker kill "$DOCKER_CONVERTER_NAME"

    echo "[Clean up] Removing temporary data '$BASE_FOLDER'"
    rm -r "$BASE_FOLDER"

    exit 1
}

_ONE_GB_IN_KB=1024000

cd $(dirname "$0")
JUDO_SCRIPTS=$(pwd)

BASE_FOLDER=$(dirname "$1") # corresponds to /tmp/jdoc/date/uuid
INPUT=$1
OUTPUT=$2
TEMP=$3
PAGE_LIMIT=$4
CONTAINER_VERSION=$5
# Extract PDFs uuid to uniquely identify the converter (helps with killing it)
DOCKER_CONVERTER_NAME="pdf2jpeghtml-$(basename "$INPUT")"

mkdir -p "$TEMP"
chmod 777 "$TEMP" # Make sure Docker can write data to this directory.

export TEMP
export TMP="$TEMP"
export TMPDIR="$TEMP"
export MAGICK_TMPDIR="$TEMP"

echo "convert $INPUT to html"

EXTRA_PARAMS=""

END_OF_PREVIEW_NAME="end_of_preview.png"
END_OF_PREVIEW_FILE="${JUDO_SCRIPTS}/$END_OF_PREVIEW_NAME"
ADD_END_OF_PREVIEW="NO"

# NOTE: The unusual line read here is necessary due to possibly illegal chars in title, which fuck up the output.
# At least this way only the title gets somewhat broken ...
PDF_PAGE_COUNT=$(pdfinfo "$INPUT" | while read line; do echo "$line";done | grep "Pages:" | cut -d ":" -f 2 | sed 's/ *$//' | sed 's/^ *//')
TITLE=$(pdfinfo "$INPUT" | while read line; do echo "$line";done | grep Title | cut -d ":" -f 2 | sed 's/ *$//' | sed 's/^ *//')

if [ "$PDF_PAGE_COUNT" = "" ]
then
    echo "can't read pdf : $INPUT"
    # exit unsupported
    exit 2
fi

PAGES_TO_WRITE=$PDF_PAGE_COUNT
PAGES_WRITTEN=0

if [ "$PAGE_LIMIT" -ge 1 ] && [ "$PDF_PAGE_COUNT" -gt "$PAGE_LIMIT" ]
then
    EXTRA_PARAMS="$EXTRA_PARAMS -l $PAGE_LIMIT"
    ADD_END_OF_PREVIEW="YES"
    PAGES_TO_WRITE=$(($PAGE_LIMIT + 1))
fi

cat <<EOF
docker run \
       --name $DOCKER_CONVERTER_NAME \
       --rm --network none -u justsoftware \
       -v $TEMP:/home/justsoftware/workspace/ \
       -v $BASE_FOLDER:$BASE_FOLDER \
       justsocial/just-pdftoppm:$CONTAINER_VERSION \
       -c -m $_ONE_GB_IN_KB pdftoppm -png $INPUT -r 100 $EXTRA_PARAMS /home/justsoftware/workspace/bg
EOF

# Enforce killing the container when the script is killed.
trap cleanUpAndExit 15

docker run \
       --name "$DOCKER_CONVERTER_NAME" \
       --rm --network none -u justsoftware \
       -v "$TEMP":/home/justsoftware/workspace/ \
       -v "$BASE_FOLDER":"$BASE_FOLDER" \
       justsocial/just-pdftoppm:$CONTAINER_VERSION \
       -c -m "$_ONE_GB_IN_KB" pdftoppm -png "$INPUT" -r 100 $EXTRA_PARAMS /home/justsoftware/workspace/bg

DOCKER_EXIT=$?

if [ "$DOCKER_EXIT" -ge "128" ]
then
    echo "[pdftoppm] Script exhausted resources limits and was terminated."
    exit 1
fi

mkdir -p "$OUTPUT"
rm -f "$OUTPUT/*"

write() {
  echo "$1" >> "$TEMP/index.html"
}

writeCSS() {
  write "    <style type=\"text/css\">"
  write ""
  write "      html, body {"
  write "        height: 100%;"
  write "        width: 100%;"
  write "      }"
  write ""
  write "      .siteStyle {"
  write "        background-color: transparent;"
  write "        margin: 0;"
  write "      }"
  write ""
  write "      .judo-wrapper {"
  write "        height: 100%;"
  write "        width: 100%;"
  write "        overflow: auto;"
  write "      }"
  write ""
  write "      .pageStyle {"
  write "        margin: auto;"
  write "        padding: 5px 5px 5px 5px;"
  write "        text-align: center;"
  write "      }"
  write ""
  write "    </style>"
}

writeJS() {
  write "<script>"
  write "function init() {"
  write "    var wrapperElement = document.querySelector('.judo-wrapper');"
  write "    wrapperElement.addEventListener('click', function(ev) {"
  write "        if(ev.target === wrapperElement) {"
  write "            parent.postMessage('outerContentClicked', parent.location.origin);"
  write "        }"
  write "    });"
  write "}"
  write "</script>"
}

writeHead() {
  write "<!DOCTYPE html>"
  write "<html xmlns=\"http://www.w3.org/1999/xhtml\" lang=\"\" xml:lang=\"\">"
  write "  <head>"
  write "    <title>$TITLE</title>"
  write ""
  write "    <meta http-equiv=\"Content-Type\" content=\"text/html; charset=UTF-8\"/>"
  writeCSS
  writeJS
  write "  </head>"
  write "  <body onload=\"init();\" class=\"siteStyle\">"
  write "    <div class=\"judo-wrapper\">"
}

writeTail() {
  write "    </div>"
  write "  </body>"
  write "</html>"
}

writePage() {
  write "    <div class=\"pageStyle\">"
  write "      <img  class=\"imageStyle\" src=\"$1\" alt=\"$1\"/>"
  write "    </div>"
}

writeHead

# add a page for each background image
#  NOTE: the 'sort' command here achieves:
#   - virtually splitting the filename at '-' into columns 1 and 2 (-t)
#   - then sorts by column 1 ignoring cases (-k1,1i)
#   - and afterwards by column 2 numerically (-k2,2n)
#  So "bg-1.jpg" gets in front of "bg-10.jpg" and "bg-10.jpg" is still before "end-of-preview.jpg". If the
#  names of preview images change, this sorting needs to be adjusted as well!
ls -1 "$TEMP" | grep "png" | grep "bg" | sort -t- -k1,1i -k2,2n | while read src
do
    trgt=${src%.*}.jpg
    nice -n 15 convert "$TEMP/$src" -resize 904800@ "$OUTPUT/$trgt"
    writePage "$trgt"
done

if [ "$ADD_END_OF_PREVIEW" = "YES" ] && [ -f "$END_OF_PREVIEW_FILE" ]
then
    cp "$END_OF_PREVIEW_FILE" "$OUTPUT/$END_OF_PREVIEW_NAME"
    writePage "$END_OF_PREVIEW_NAME"
fi

writeTail

# count pages written
PAGES_WRITTEN=$(cat "$TEMP/index.html" | grep "div" | grep -c "pageStyle")

[ "$PAGES_WRITTEN" -eq "$PAGES_TO_WRITE" ] && cp "$TEMP/index.html" "$OUTPUT/index.html"

[ "$(ls -1 $OUTPUT | wc -l)" -eq 1 ] && rm -r "$OUTPUT"

rm -r "$TEMP"

# index.html is copied last and only on success, so if it doesn't exist, we failed to convert....
# so check that we didn't fail and return
if [ -f "$OUTPUT/index.html" ]
then
    exit 0
else
    echo "conversion failed, no output created"
    exit 1
fi
