/*
 * Decompiled with CFR 0.152.
 */
package de.justsoftware.storageserver.service;

import de.justsoftware.storageserver.config.StorageserverConfiguration;
import de.justsoftware.storageserver.service.FileManagementException;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.FileTime;
import java.nio.file.attribute.PosixFilePermission;
import java.nio.file.attribute.PosixFilePermissions;
import java.time.Clock;
import java.time.Instant;
import java.util.Set;
import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.FileSystemUtils;

@Service
@ParametersAreNonnullByDefault
public class FileManagementService {
    private static final String FILES_DIR = "files";
    private static final String TRASH_DIR = "trash";
    private static final Logger LOG = LoggerFactory.getLogger(FileManagementService.class);
    private final Clock _clock;
    private final String _filesPath;
    private final String _trashPath;
    private final Set<PosixFilePermission> _filePermissions;
    private final Set<PosixFilePermission> _directoryPermissions;
    private final FileAttribute<?> _directoryPermissionsAttribute;

    @Autowired
    public FileManagementService(StorageserverConfiguration config, Clock clock) {
        Path rootPath = config.getRootPath().toPath().normalize();
        Path filesDirectory = rootPath.resolve(FILES_DIR);
        this._filesPath = filesDirectory.toString();
        this._trashPath = rootPath.resolve(TRASH_DIR).toString();
        this._filePermissions = PosixFilePermissions.fromString(config.getFilePermissions());
        this._directoryPermissions = PosixFilePermissions.fromString(config.getDirectoryPermissions());
        this._directoryPermissionsAttribute = PosixFilePermissions.asFileAttribute(this._directoryPermissions);
        this._clock = clock;
        this.ensureFilesDirectory(filesDirectory);
    }

    private void ensureFilesDirectory(Path filesDirectory) {
        if (Files.isDirectory(filesDirectory, new LinkOption[0])) {
            return;
        }
        try {
            Files.createDirectories(filesDirectory, this._directoryPermissionsAttribute);
            LOG.info("Created directory {}", (Object)filesDirectory);
        }
        catch (IOException e) {
            LOG.error("Could not create directory " + String.valueOf(filesDirectory), (Throwable)e);
        }
    }

    @Nonnull
    public File getFilesPath() {
        return new File(this._filesPath);
    }

    @Nonnull
    public File ensureDirectory(String path) throws IOException {
        return this.ensureDirectory(this._filesPath, path).toFile();
    }

    public void ensureFilePermissions(File file) throws IOException {
        Files.setPosixFilePermissions(file.toPath(), this._filePermissions);
    }

    @Nonnull
    private Path ensureDirectory(String basePath, String path) throws IOException {
        Path directory = Paths.get(basePath, path);
        Files.createDirectories(directory, this._directoryPermissionsAttribute);
        return directory;
    }

    public void deleteFile(String path, String filename) throws IOException {
        Path targetFile = Paths.get(this._filesPath, path, filename);
        if (!Files.isRegularFile(targetFile, new LinkOption[0])) {
            return;
        }
        Files.delete(targetFile);
    }

    public void moveFileToTrash(String path, String filename) throws IOException, FileManagementException {
        Path srcFile = Paths.get(this._filesPath, path, filename);
        if (Files.isDirectory(srcFile, new LinkOption[0])) {
            throw new FileManagementException("Moving directories to the trash is not allowed");
        }
        if (!Files.isRegularFile(srcFile, new LinkOption[0])) {
            return;
        }
        Path targetPath = this.ensureDirectory(this._trashPath, path);
        this.moveAndUpdateLastModified(srcFile, targetPath.resolve(filename));
    }

    private void moveAndUpdateLastModified(Path srcFile, Path targetFile) throws FileManagementException {
        Instant now = Instant.now(this._clock);
        try {
            Files.move(srcFile, targetFile, StandardCopyOption.REPLACE_EXISTING);
            Files.setLastModifiedTime(targetFile, FileTime.from(now));
            FileTime lastModifiedTime = Files.getLastModifiedTime(targetFile, LinkOption.NOFOLLOW_LINKS);
            if (now.isAfter(lastModifiedTime.toInstant())) {
                LOG.warn("Could not set last-modified-time on trash file. Moving '{}' back to '{}', to prevent unintended data loss.", (Object)targetFile.toString(), (Object)srcFile.toString());
                Files.move(targetFile, srcFile, StandardCopyOption.REPLACE_EXISTING);
            }
        }
        catch (IOException | IllegalStateException | SecurityException e) {
            LOG.error("Could not move '" + srcFile.toString() + "' to trash", (Throwable)e);
            throw new FileManagementException((Throwable)e);
        }
    }

    public void deleteDirectory(String path) throws IOException {
        Path targetDir = Paths.get(this._filesPath, path);
        if (!Files.isDirectory(targetDir, new LinkOption[0])) {
            return;
        }
        FileSystemUtils.deleteRecursively((Path)targetDir);
    }

    public void copyFile(String srcPath, String destPath) throws IOException, FileManagementException {
        Path srcFile = Paths.get(this._filesPath, srcPath);
        Path destFile = Paths.get(this._filesPath, destPath);
        if (!Files.isRegularFile(srcFile, new LinkOption[0])) {
            throw new FileManagementException("File " + String.valueOf(srcFile.normalize()) + " does not exist or is not a regular file");
        }
        Path destDir = destFile.getParent();
        if (destDir != null) {
            Files.createDirectories(destDir, this._directoryPermissionsAttribute);
        }
        Files.copy(srcFile, destFile, StandardCopyOption.REPLACE_EXISTING);
        Files.setPosixFilePermissions(destFile, this._filePermissions);
    }

    public void moveFile(String srcPath, String destPath) throws IOException, FileManagementException {
        Path srcFile = Paths.get(this._filesPath, srcPath);
        Path destFile = Paths.get(this._filesPath, destPath);
        if (!Files.isRegularFile(srcFile, new LinkOption[0])) {
            throw new FileManagementException("File " + String.valueOf(srcFile.normalize()) + " does not exist or is not a regular file");
        }
        Path destDir = destFile.getParent();
        if (destDir != null) {
            Files.createDirectories(destDir, this._directoryPermissionsAttribute);
        }
        Files.move(srcFile, destFile, StandardCopyOption.REPLACE_EXISTING);
    }
}

