--liquibase formatted sql

--changeset christian.ewers:add_timestamps dbms:postgresql runAlways:true splitStatements:false
CREATE OR REPLACE FUNCTION update_mtime()
    RETURNS TRIGGER AS
$$
BEGIN
    NEW.mtime = clock_timestamp();
    RETURN NEW;
END;
$$ language 'plpgsql';

CREATE OR REPLACE FUNCTION ctime_mtime_columns(tname TEXT)
    RETURNS VOID AS
$$
BEGIN
    EXECUTE 'ALTER TABLE ' || tname ||
            ' ADD COLUMN ctime TIMESTAMP WITHOUT TIME ZONE NOT NULL DEFAULT clock_timestamp();';
    EXECUTE 'ALTER TABLE ' || tname ||
            ' ADD COLUMN mtime TIMESTAMP WITHOUT TIME ZONE NOT NULL DEFAULT clock_timestamp();';
    EXECUTE 'CREATE TRIGGER ' || tname || '_mtrg BEFORE UPDATE ON ' || tname ||
            ' FOR EACH ROW EXECUTE PROCEDURE update_mtime();';
END;
$$ language 'plpgsql';

--changeset christian.ewers:create_migration_function dbms:postgresql splitStatements:false
CREATE OR REPLACE FUNCTION run_if_table_exists(table_to_check text, statement TEXT)
    RETURNS VOID AS
$$
BEGIN
    IF EXISTS
        (SELECT 1
         FROM information_schema.tables
         WHERE table_schema = 'public'
           AND table_name = table_to_check)
    THEN
        EXECUTE statement;
    END IF;
END;
$$ language 'plpgsql';


--changeset christian.ewers:add_oauth2_tables
CREATE TABLE oauth2_registered_client
(
    id                                   UUID          NOT NULL,
    client_id                            VARCHAR(255)  NOT NULL,
    client_name                          VARCHAR(255)  NOT NULL,
    client_secret                        VARCHAR(255)           DEFAULT NULL,
    client_authentication_methods        VARCHAR(1024) NOT NULL,
    authorization_grant_types            VARCHAR(1024) NOT NULL,
    redirect_uris                        VARCHAR(1024)          DEFAULT NULL,
    post_logout_redirect_uris            VARCHAR(1024)          DEFAULT NULL,
    scopes                               VARCHAR(1024) NOT NULL,
    tenant_id                            UUID                   DEFAULT NULL,

    -- token-settings
    authorization_code_time_to_live      BIGINT        NOT NULL DEFAULT 300000000000, -- 5 minutes
    access_token_time_to_live            BIGINT        NOT NULL DEFAULT 300000000000, -- 5 minutes,
    access_token_format                  VARCHAR(255)  NOT NULL DEFAULT 'reference',
    device_code_time_to_live             BIGINT        NOT NULL DEFAULT 300000000000, -- 5 minutes,
    reuse_refresh_tokens                 BOOLEAN       NOT NULL DEFAULT false,
    refresh_token_time_to_live           BIGINT        NOT NULL DEFAULT 1.8e+12,      -- 30 minutes,
    id_token_signature_algorithm         VARCHAR(255)  NOT NULL DEFAULT 'RS256',
    x509_certificate_bound_access_tokens VARCHAR(255)  NOT NULL DEFAULT false,

    -- client-settings
    require_authorization_consent        BOOLEAN       NOT NULL DEFAULT false,
    require_proof_key                    BOOLEAN       NOT NULL DEFAULT false,
    PRIMARY KEY (id),
    UNIQUE (client_id)
);
SELECT ctime_mtime_columns('oauth2_registered_client');

CREATE TABLE oauth2_authorization_consent
(
    registered_client_id VARCHAR(255)  NOT NULL,
    principal_name       VARCHAR(255)  NOT NULL,
    authorities          VARCHAR(1024) NOT NULL,
    PRIMARY KEY (registered_client_id, principal_name)
);
SELECT ctime_mtime_columns('oauth2_authorization_consent');

CREATE TABLE oauth2_authorization
(
    id                            UUID         NOT NULL,
    registered_client_id          UUID         NOT NULL references oauth2_registered_client (id) ON DELETE CASCADE,
    principal_name                VARCHAR(255) NOT NULL,
    principal_profile_id          VARCHAR(255)  DEFAULT NULL,
    principal_tenant_id           UUID         NOT NULL,
    authorization_grant_type      VARCHAR(255) NOT NULL,
    authorized_scopes             VARCHAR(1024) DEFAULT NULL,
    authorization_attributes      TEXT          DEFAULT NULL,
    authorization_state           VARCHAR(1024) DEFAULT NULL,
    authorization_code_value      VARCHAR(1024) DEFAULT NULL,
    authorization_code_issued_at  TIMESTAMP     DEFAULT NULL,
    authorization_code_expires_at TIMESTAMP     DEFAULT NULL,
    authorization_code_metadata   TEXT          DEFAULT NULL,
    access_token_value            VARCHAR(4000) DEFAULT NULL,
    access_token_issued_at        TIMESTAMP     DEFAULT NULL,
    access_token_expires_at       TIMESTAMP     DEFAULT NULL,
    access_token_metadata         TEXT          DEFAULT NULL,
    access_token_type             VARCHAR(255)  DEFAULT NULL,
    access_token_scopes           VARCHAR(1024) DEFAULT NULL,
    oidc_id_token_value           VARCHAR(4000) DEFAULT NULL,
    oidc_id_token_issued_at       TIMESTAMP     DEFAULT NULL,
    oidc_id_token_expires_at      TIMESTAMP     DEFAULT NULL,
    oidc_id_token_metadata        TEXT          DEFAULT NULL,
    refresh_token_value           VARCHAR(4000) DEFAULT NULL,
    refresh_token_issued_at       TIMESTAMP     DEFAULT NULL,
    refresh_token_expires_at      TIMESTAMP     DEFAULT NULL,
    refresh_token_metadata        TEXT          DEFAULT NULL,
    PRIMARY KEY (id)

);
SELECT ctime_mtime_columns('oauth2_authorization');

--changeset christian.ewers:migrate_oauth_clients dbms:postgresql
SELECT run_if_table_exists('oauth_client_details', '
    INSERT INTO oauth2_registered_client(id, client_id, client_name, client_secret, authorization_grant_types, client_authentication_methods, redirect_uris, scopes, tenant_id)
    SELECT
        gen_random_uuid(),
        ocd.client_id,
        ocd.client_id as client_name,
        CONCAT(''{bcrypt}'',ocd.client_secret),
        ocd.authorized_grant_types,
        case when ocd.client_secret is null then ''none'' else ''client_secret_post'' end as methods,
        ocd.web_server_redirect_uri, CONCAT(ocd.authorities,'','',ocd.scope),
        coalesce(ocd.tenant_id, (CASE WHEN tenantCount.count = 1 THEN global_tenant.id ELSE NULL END))
    FROM
        public.oauth_client_details ocd,
        (SELECT count(id) as count FROM public.tenant) as tenantCount,
        (SELECT id from public.tenant LIMIT 1) as global_tenant
    where ocd.client_id NOT IN (''toro'', ''just-login'', ''just-login-mobile'', ''mobile-apps'', ''swagger'')
');

--changeset argo.liivanurm:add_remember_me_token dbms:postgresql
CREATE TABLE remember_me_token
(
    series    VARCHAR(255) NOT NULL,
    username  VARCHAR(255) NOT NULL,
    token     VARCHAR(255) NOT NULL,
    last_used TIMESTAMP    NOT NULL,
    PRIMARY KEY (series)
);
SELECT ctime_mtime_columns('remember_me_token');

--changeset christian.ewers:make_tenant_id_optional_again
ALTER TABLE oauth2_authorization
    ALTER COLUMN principal_tenant_id DROP NOT NULL;

--changeset argo.liivanurm:add_totp_device
CREATE TABLE totp_device
(
    id          UUID         NOT NULL,
    profile_id  VARCHAR(255) NOT NULL,
    secret      VARCHAR(255) NOT NULL,
    confirmed   BOOLEAN      NOT NULL DEFAULT false,
    created_at  TIMESTAMP    NOT NULL,
    PRIMARY KEY (id),
    UNIQUE (profile_id)
);
SELECT ctime_mtime_columns('totp_device');

--changeset argo.liivanurm:create_settings_table
CREATE TABLE settings
(
    tenant_id     UUID PRIMARY KEY,
    mfa_mandatory BOOLEAN NOT NULL
);
SELECT ctime_mtime_columns('settings');

--changeset argo.liivanurm:add_confirmed_at_timestamp
ALTER TABLE totp_device ADD COLUMN confirmed_at TIMESTAMP NULL;

--changeset argo.liivanurm:set_confirmed_at_timestamp_for_already_confirmed_devices
UPDATE totp_device SET confirmed_at = mtime WHERE confirmed = true;

--changeset argo.liivanurm:create_account_lock_failed_login
CREATE TABLE account_lock
(
    id           UUID         NOT NULL,
    profile_id   VARCHAR(255) NOT NULL,
    locked_until TIMESTAMP    NOT NULL,
    PRIMARY KEY (id),
    UNIQUE (profile_id)
);
SELECT ctime_mtime_columns('account_lock');

CREATE TABLE account_failed_login
(
    id            UUID         NOT NULL,
    profile_id    VARCHAR(255) NOT NULL,
    failed_count  INT          NOT NULL,
    counted_since TIMESTAMP    NOT NULL,
    PRIMARY KEY (id),
    UNIQUE (profile_id)
);
SELECT ctime_mtime_columns('account_failed_login');
