--liquibase formatted sql

--changeset christian.ewers:add_timestamps runAlways:true splitStatements:false
CREATE OR REPLACE FUNCTION update_mtime()
RETURNS TRIGGER AS $$
BEGIN
   NEW.mtime = clock_timestamp();
   RETURN NEW;
END;
$$ language 'plpgsql';

CREATE OR REPLACE FUNCTION ctime_mtime_columns(tname TEXT)
RETURNS VOID AS $$
BEGIN
    EXECUTE 'ALTER TABLE ' || tname || ' ADD COLUMN ctime TIMESTAMP WITHOUT TIME ZONE NOT NULL DEFAULT clock_timestamp();';
    EXECUTE 'ALTER TABLE ' || tname || ' ADD COLUMN mtime TIMESTAMP WITHOUT TIME ZONE NOT NULL DEFAULT clock_timestamp();';
    EXECUTE 'CREATE TRIGGER ' || tname || '_mtrg BEFORE UPDATE ON ' || tname || ' FOR EACH ROW EXECUTE PROCEDURE update_mtime();';
END;
$$ language 'plpgsql';

--changeset baron:create_initial_tables
CREATE TABLE profile (
    id VARCHAR(64) NOT NULL PRIMARY KEY,
    first_name TEXT NOT NULL,
    last_name TEXT NOT NULL,
    email TEXT NOT NULL,
    image_url TEXT NOT NULL,
    status TEXT NOT NULL
);
SELECT ctime_mtime_columns('profile');

CREATE TABLE channel (
    id VARCHAR(64) NOT NULL PRIMARY KEY,
    title TEXT NOT NULL,
    tenant_id VARCHAR(64) NOT NULL,
    deleted BOOLEAN NOT NULL,
    deleted_at TIMESTAMP WITHOUT TIME ZONE DEFAULT NULL,
    deleted_by VARCHAR(64) DEFAULT NULL,
    create_date TIMESTAMP WITHOUT TIME ZONE NOT NULL,
    modify_date TIMESTAMP WITHOUT TIME ZONE NOT NULL
);
SELECT ctime_mtime_columns('channel');

CREATE TABLE personal_subscription (
    id VARCHAR(64) NOT NULL PRIMARY KEY,
    channel_id VARCHAR(64) NOT NULL,
    profile_id VARCHAR(64) NOT NULL,
    notifications_enabled BOOLEAN NOT NULL,
    user_has_unsubscribed BOOLEAN NOT NULL,
    CONSTRAINT unique__channel__profile
        UNIQUE(channel_id, profile_id),
    CONSTRAINT fk__channel__personal_subscription
        FOREIGN KEY(channel_id)
        REFERENCES channel(id)
        ON DELETE CASCADE,
    CONSTRAINT fk__profile__subscription
        FOREIGN KEY(profile_id)
        REFERENCES profile(id)
        ON DELETE CASCADE
);
SELECT ctime_mtime_columns('personal_subscription');

CREATE TABLE managed_subscription (
    id VARCHAR(64) NOT NULL PRIMARY KEY,
    channel_id VARCHAR(64) NOT NULL,
    type VARCHAR(64) NOT NULL,
    user_id VARCHAR(64) NOT NULL,
    CONSTRAINT unique_managed_subscription__channel__user
        UNIQUE (channel_id, user_id),
    CONSTRAINT fk__channel__managed_subscription
        FOREIGN KEY(channel_id)
        REFERENCES channel(id)
        ON DELETE CASCADE
);
SELECT ctime_mtime_columns('managed_subscription');

CREATE TABLE user_group (
    id VARCHAR(64) NOT NULL PRIMARY KEY,
    name TEXT NOT NULL,
    tenant_id VARCHAR(64) NOT NULL
);
SELECT ctime_mtime_columns('user_group');

CREATE TABLE post (
    id VARCHAR(64) NOT NULL PRIMARY KEY,
    author_id VARCHAR(64) NOT NULL,
    channel_id VARCHAR(64) NOT NULL,
    create_date TIMESTAMP WITHOUT TIME ZONE NOT NULL,
    modify_date TIMESTAMP WITHOUT TIME ZONE NOT NULL,
    title TEXT NOT NULL,
    text TEXT NOT NULL,
    json_content TEXT NOT NULL,
    cover_element_id VARCHAR(64),
    pinned BOOLEAN DEFAULT false NOT NULL,
    pinned_at TIMESTAMP WITHOUT TIME ZONE,
    publish_date TIMESTAMP WITHOUT TIME ZONE NOT NULL,
    published BOOLEAN NOT NULL,
    deleted BOOLEAN NOT NULL,
    deleted_at TIMESTAMP WITHOUT TIME ZONE DEFAULT NULL,
    deleted_by VARCHAR(64) DEFAULT NULL,
    CONSTRAINT fk__author__post
        FOREIGN KEY(author_id)
        REFERENCES profile(id),
    CONSTRAINT fk__channel__post
        FOREIGN KEY(channel_id)
        REFERENCES channel(id)
        ON DELETE CASCADE
);
SELECT ctime_mtime_columns('post');

CREATE TABLE attachment (
    id VARCHAR(64) NOT NULL PRIMARY KEY,
    post_id VARCHAR(64) NOT NULL,
    mime_type TEXT NOT NULL,
    name TEXT NOT NULL,
    file_size BIGINT DEFAULT 0 NOT NULL,
    CONSTRAINT fk__post__attachment
        FOREIGN KEY(post_id)
        REFERENCES post(id)
        ON DELETE CASCADE
);
SELECT ctime_mtime_columns('attachment');

CREATE TABLE poll (
    id VARCHAR(64) NOT NULL PRIMARY KEY,
    question TEXT NOT NULL,
    allow_multiple_answers BOOLEAN NOT NULL,
    post_id VARCHAR(64) NOT NULL,
    CONSTRAINT fk__post__poll
        FOREIGN KEY(post_id)
        REFERENCES post(id)
        ON DELETE CASCADE
);
SELECT ctime_mtime_columns('poll');

CREATE TABLE poll_answer (
    id VARCHAR(64) NOT NULL PRIMARY KEY,
    text TEXT NOT NULL,
    position INTEGER NOT NULL,
    poll_id VARCHAR(64) NOT NULL,
    CONSTRAINT fk__poll__poll_answer
        FOREIGN KEY(poll_id)
        REFERENCES poll(id)
        ON DELETE CASCADE
);
SELECT ctime_mtime_columns('poll_answer');

CREATE TABLE poll_vote (
    id VARCHAR(64) NOT NULL PRIMARY KEY,
    poll_id VARCHAR(64) NOT NULL,
    answer_id VARCHAR(64) NOT NULL,
    profile_id VARCHAR(64) NOT NULL,
    CONSTRAINT unique__answer__profile
        UNIQUE (answer_id, profile_id),
    CONSTRAINT fk__poll_answer__poll_vote
        FOREIGN KEY(answer_id)
        REFERENCES poll_answer(id)
        ON DELETE CASCADE
);
SELECT ctime_mtime_columns('poll_vote');

CREATE TABLE post_comment (
    id VARCHAR(64) NOT NULL PRIMARY KEY,
    post_id VARCHAR(64) NOT NULL,
    parent_id VARCHAR(64) NOT NULL,
    reference_id VARCHAR(64) NOT NULL,
    author_id VARCHAR(64) NOT NULL,
    created_at TIMESTAMP WITHOUT TIME ZONE NOT NULL,
    updated_at TIMESTAMP WITHOUT TIME ZONE NOT NULL,
    last_updated_by VARCHAR(64),
    json_content TEXT NOT NULL,
    text TEXT NOT NULL,
    deleted BOOLEAN DEFAULT false NOT NULL,
    deleted_at TIMESTAMP WITHOUT TIME ZONE DEFAULT NULL,
    deleted_by VARCHAR(64) DEFAULT NULL,
    CONSTRAINT fk__author__post_comment
        FOREIGN KEY(author_id)
        REFERENCES profile(id),
    CONSTRAINT fk__post__post_comment
        FOREIGN KEY(post_id)
        REFERENCES post(id)
        ON DELETE CASCADE
);
SELECT ctime_mtime_columns('post_comment');

CREATE TABLE reaction (
    id VARCHAR(64) NOT NULL PRIMARY KEY,
    emoji VARCHAR(1024) NOT NULL,
    item_id VARCHAR(64) NOT NULL,
    profile_id VARCHAR(64) NOT NULL
);
SELECT ctime_mtime_columns('reaction');

CREATE TABLE permission (
    id VARCHAR(64) NOT NULL PRIMARY KEY,
    item_id VARCHAR(64) NOT NULL,
    grantee_id VARCHAR(64) NOT NULL,
    role VARCHAR(1024) NOT NULL,
    CONSTRAINT unique__item_id__grantee_id__role
        UNIQUE (item_id, grantee_id, role)
);
SELECT ctime_mtime_columns('permission');

CREATE TABLE tenant (
    id VARCHAR(64) NOT NULL PRIMARY KEY,
    deepl_key VARCHAR(64)
);
SELECT ctime_mtime_columns('tenant');

--changeset baron:create_table__user_group_member
CREATE TABLE user_group_member (
    id VARCHAR(1024) NOT NULL PRIMARY KEY,
    user_group_id VARCHAR(64) NOT NULL,
    profile_id VARCHAR(64) NOT NULL,
    CONSTRAINT unique__user_group_id__profile_id
        UNIQUE (user_group_id, profile_id)
);
SELECT ctime_mtime_columns('user_group_member');

--changeset baron:create_table_news_cover
CREATE TABLE news_cover (
    tenant_id VARCHAR(64) NOT NULL PRIMARY KEY,
    open_on_start BOOLEAN NOT NULL,
    cycle_delay INTEGER NOT NULL,
    CONSTRAINT fk__tenant__news_cover
        FOREIGN KEY(tenant_id)
        REFERENCES tenant(id)
        ON DELETE CASCADE
);
SELECT ctime_mtime_columns('news_cover');

CREATE TABLE news_cover_section (
    id VARCHAR(64) NOT NULL PRIMARY KEY,
    title TEXT NOT NULL,
    pinned_only BOOLEAN NOT NULL,
    position INTEGER NOT NULL,
    tenant_id VARCHAR(64) NOT NULL,
    CONSTRAINT fk__news_cover__news_cover_section
        FOREIGN KEY(tenant_id)
        REFERENCES news_cover(tenant_id)
        ON DELETE CASCADE
);
SELECT ctime_mtime_columns('news_cover_section');

CREATE TABLE news_cover_section_channel (
    section_id VARCHAR(64) NOT NULL,
    channel_id VARCHAR(64) NOT NULL,
    CONSTRAINT fk__news_cover_section__news_cover_section_channel
        FOREIGN KEY(section_id)
        REFERENCES news_cover_section(id)
        ON DELETE CASCADE,
    CONSTRAINT unique__news_cover_section__channel
        UNIQUE (section_id, channel_id)
);
SELECT ctime_mtime_columns('news_cover_section_channel');

--changeset baron:add_post_read_mark
CREATE TABLE post_read_mark(
    id VARCHAR(64) NOT NULL PRIMARY KEY,
    post_id VARCHAR(64) NOT NULL,
    profile_id VARCHAR(64) NOT NULL,
    post_published_date TIMESTAMP WITHOUT TIME ZONE NOT NULL,
    CONSTRAINT fk__post__post_read_mark
        FOREIGN KEY(post_id)
        REFERENCES post(id)
        ON DELETE CASCADE,
    CONSTRAINT fk__profile__post_read_mark
        FOREIGN KEY(profile_id)
        REFERENCES profile(id)
        ON DELETE CASCADE,
    CONSTRAINT unique__post_id__profile_id
        UNIQUE (post_id, profile_id)
);
SELECT ctime_mtime_columns('post_read_mark');

--changeset baron:permission__alter__column__item_id
ALTER TABLE permission RENAME COLUMN item_id to channel_id;
ALTER TABLE permission ADD CONSTRAINT fk__channel__permission
    FOREIGN KEY(channel_id)
    REFERENCES channel(id)
    ON DELETE CASCADE;

--changeset baron:permission__rename_table
ALTER TABLE permission DROP CONSTRAINT unique__item_id__grantee_id__role;
ALTER TABLE permission ADD CONSTRAINT unique__channel_id__grantee_id
    UNIQUE (channel_id, grantee_id);
ALTER TABLE permission RENAME to channel_permission;

--changeset liivanurm:add_migrated_item
CREATE TABLE migrated_item (
    item_id VARCHAR(64) NOT NULL PRIMARY KEY,
    legacy_id VARCHAR(255) NOT NULL,
    CONSTRAINT unique__item_id__legacy_id
        UNIQUE (item_id, legacy_id)
);
SELECT ctime_mtime_columns('migrated_item');

--changeset maxse:add_migration_error
CREATE TABLE migration_error (
    id VARCHAR(64) NOT NULL PRIMARY KEY,
    item_id VARCHAR(64),
    reason TEXT,
    data TEXT
);
SELECT ctime_mtime_columns('migration_error');

--changeset liivanurm:alter_attachments_add_position
ALTER TABLE attachment ADD position INTEGER DEFAULT 0 NOT NULL;

--changeset baron:alter_profile_add_prefix_suffix
ALTER TABLE profile ADD prefix TEXT;
ALTER TABLE profile ADD suffix TEXT;

--changeset baron:add_settings
CREATE TABLE settings (
    tenant_id VARCHAR(64) NOT NULL PRIMARY KEY,
    all_may_read_subscription_profiles BOOLEAN DEFAULT 't' NOT NULL,
    CONSTRAINT fk__tenant__settings
        FOREIGN KEY(tenant_id)
        REFERENCES tenant(id)
        ON DELETE CASCADE
);
SELECT ctime_mtime_columns('settings');
INSERT INTO settings (tenant_id) SELECT id FROM tenant;

--changeset liivanurm:add_managed_subscription_where_channel_permission_exists
INSERT INTO managed_subscription(
SELECT concat('MANAGED_SUBSCRIPTION,', gen_random_uuid()), cp.channel_id, 'NO_RECOMMENDATION', cp.grantee_id
FROM channel_permission cp
WHERE NOT EXISTS (SELECT * FROM managed_subscription ms WHERE cp.grantee_id = ms.user_id));

--changeset liivanurm:fix_add_managed_subscription_for_all_channels
INSERT INTO managed_subscription(
SELECT concat('MANAGED_SUBSCRIPTION,', gen_random_uuid()), cp.channel_id, 'NO_RECOMMENDATION', cp.grantee_id
FROM channel_permission cp
WHERE NOT EXISTS (SELECT * FROM managed_subscription ms WHERE cp.channel_id = ms.channel_id));

--changeset maxse:alter_poll_add_privacy
ALTER TABLE poll ADD privacy VARCHAR(1024) DEFAULT 'PUBLIC' NOT NULL;

--changeset twuebbena:alter_settings_add_allow_additional_info_in_mail_notification
ALTER TABLE settings ADD allow_additional_info_in_mail_notification BOOLEAN DEFAULT 'f' NOT NULL;

--changeset cewers:alter_types_of_profile_table
ALTER TABLE profile ALTER COLUMN last_name TYPE VARCHAR(255);
ALTER TABLE profile ALTER COLUMN first_name TYPE VARCHAR(255);
ALTER TABLE profile ALTER COLUMN email TYPE VARCHAR(256);
ALTER TABLE profile ALTER COLUMN image_url TYPE VARCHAR(1024);
ALTER TABLE profile ALTER COLUMN status TYPE VARCHAR(255);
ALTER TABLE profile ALTER COLUMN prefix TYPE VARCHAR(255);
ALTER TABLE profile ALTER COLUMN suffix TYPE VARCHAR(255);

--changeset liivanurm:rename_post_mark_read_to_post_UNread_mark
ALTER TABLE post_read_mark RENAME to post_unread_mark;

--changeset baron:post__add__column__edited_by
ALTER TABLE post ADD edited_by VARCHAR(64) DEFAULT NULL;

--changeset baron:drop__table__migration_error
DROP TABLE migration_error;

--changeset maxse:add_created_by_to_channel
ALTER TABLE channel ADD created_by VARCHAR(64) DEFAULT NULL;

--changeset maxse:add_edited_by_to_channel
ALTER TABLE channel ADD edited_by VARCHAR(64) DEFAULT NULL;

--changeset maxse:post_drop__pinned__add__pin_expiration
ALTER TABLE post ADD COLUMN pin_expiration TIMESTAMP WITHOUT TIME ZONE DEFAULT NULL;
UPDATE post
    SET pin_expiration =
        CASE
            WHEN published = true THEN (now() + INTERVAL '30 days')
            ELSE (publish_date + INTERVAL '30 days')
        END
    WHERE pinned = true;
ALTER TABLE post DROP COLUMN pinned;

--changeset christian.ewers:add_index_to_reactions_and_comments
DROP INDEX IF EXISTS reaction_item_idx;
CREATE INDEX reaction_item_idx ON reaction (item_id);
DROP INDEX IF EXISTS post_comment_parent_idx;
CREATE INDEX post_comment_parent_idx ON post_comment (parent_id);

--changeset christian.ewers:add_publish_date_idx
create index post_publish_date_idx ON post(publish_date);

--changeset baron:create__table__event
CREATE TABLE event (
    id VARCHAR(64) NOT NULL PRIMARY KEY,
    post_id VARCHAR(64) NOT NULL,
    start_date TIMESTAMP WITHOUT TIME ZONE NOT NULL,
    end_date TIMESTAMP WITHOUT TIME ZONE NOT NULL,
    CONSTRAINT fk__post__event
        FOREIGN KEY(post_id)
        REFERENCES post(id)
        ON DELETE CASCADE
);
SELECT ctime_mtime_columns('event');

--changeset baron:create__index__event__post_id
create index event_post_id_idx ON event(post_id);

--changeset baron:add__column__delete_date
ALTER TABLE post ADD COLUMN delete_date TIMESTAMP WITHOUT TIME ZONE DEFAULT NULL;

--changeset baron:alter__poll_vote__add__column__vote_date
ALTER TABLE poll_vote ADD vote_date TIMESTAMP WITHOUT TIME ZONE DEFAULT clock_timestamp() NOT NULL;
UPDATE poll_vote SET vote_date = ctime;

--changeset liivanurm:add__column__comments_disabled
ALTER TABLE post ADD comments_disabled BOOLEAN DEFAULT 'f' NOT NULL;

--changeset liivanurm:create__indexes__for__post_unread_mark__attachment__post__channel
DROP INDEX IF EXISTS post_unread_mark_profile_id_idx;
CREATE INDEX post_unread_mark_profile_id_idx ON post_unread_mark (profile_id);
DROP INDEX IF EXISTS attachment_post_id_idx;
CREATE INDEX attachment_post_id_idx ON attachment (post_id);
DROP INDEX IF EXISTS post_channel_id_idx;
CREATE INDEX post_channel_id_idx ON post (channel_id);

--changeset baron:create__table__comment_notifications
CREATE TABLE comment_notifications (
    id BIGSERIAL PRIMARY KEY NOT NULL,
    profile_id VARCHAR(64) NOT NULL,
    post_id VARCHAR(64) NOT NULL,
    enabled BOOLEAN NOT NULL,
    CONSTRAINT fk__post__comment_notifications
        FOREIGN KEY(post_id)
        REFERENCES post(id)
        ON DELETE CASCADE,
    CONSTRAINT unique__post__profile
        UNIQUE(profile_id, post_id)
);
SELECT ctime_mtime_columns('comment_notifications');

--changeset baron:add__profile_idx__to__comment_notifications
CREATE INDEX profile_idx ON comment_notifications (profile_id);

--changeset baron:add__post_idx__to__comment_notifications
CREATE INDEX post_idx ON comment_notifications (post_id);

--changeset maxse:drop__column_deepl_key__on__tenant_table
ALTER TABLE tenant DROP COLUMN deepl_key;

--changeset liivanurm:allow__only__unique__reactions
ALTER TABLE reaction ADD CONSTRAINT unique__item_id__profile_id__emoji UNIQUE (item_id, profile_id, emoji);

--changeset baron:attachment__alter__column__post_id
ALTER TABLE attachment DROP CONSTRAINT fk__post__attachment;
ALTER TABLE attachment RENAME COLUMN post_id to item_id;

--changeset baron:add__item_idx__to__attachment
DROP INDEX IF EXISTS attachment_post_id_idx;
CREATE INDEX attachment_item_id_idx ON attachment (item_id);

--changeset baron:post_comment__alter__column__json_content
ALTER TABLE post_comment ALTER COLUMN json_content DROP NOT NULL;

--changeset baron:post_comment__alter__column__text
ALTER TABLE post_comment ALTER COLUMN text DROP NOT NULL;

--changeset liivanurm:add__tenant_id__to__profile
ALTER TABLE profile ADD tenant_id VARCHAR(64) NULL;

--changeset baron:table__attachment__add__column__alt_text
ALTER TABLE attachment ADD COLUMN alt_text TEXT DEFAULT NULL;

--changeset maxse:add__column_deepl_key__on__tenant_table
ALTER TABLE tenant ADD COLUMN deepl_key VARCHAR(255);

--changeset cdusse:post__add__translation
CREATE TABLE translation (
    item_id TEXT NOT NULL,
    field_id TEXT NOT NULL,
    language TEXT NOT NULL,
    text TEXT NOT NULL,
    PRIMARY KEY (item_id, field_id, language)
);
SELECT ctime_mtime_columns('translation');
ALTER TABLE post ADD COLUMN source_language TEXT NULL;

--changeset cdusse:add__source_language__to__post_comment
ALTER TABLE post_comment ADD COLUMN source_language TEXT NULL;

--changeset baron:channel__add__column_only_managers_may_pin
ALTER TABLE channel ADD COLUMN only_managers_may_pin BOOLEAN DEFAULT false NOT NULL;

--changeset baron:add__event_participation
CREATE TABLE event_participation (
    id VARCHAR(64) NOT NULL PRIMARY KEY,
    event_id VARCHAR(64) NOT NULL,
    profile_id VARCHAR(64) NOT NULL,
    created_at TIMESTAMP WITHOUT TIME ZONE NOT NULL,
    updated_at TIMESTAMP WITHOUT TIME ZONE NOT NULL,
    CONSTRAINT unique__event__profile
        UNIQUE (event_id, profile_id),
    CONSTRAINT fk__event__event_participation
        FOREIGN KEY(event_id)
        REFERENCES event(id)
        ON DELETE CASCADE,
    CONSTRAINT fk__profile__event_participation
        FOREIGN KEY(profile_id)
        REFERENCES profile(id)
        ON DELETE CASCADE
);
SELECT ctime_mtime_columns('event_participation');

--changeset baron:table__event__add__column__participation_enabled
ALTER TABLE event ADD COLUMN participation_enabled BOOLEAN DEFAULT false NOT NULL;

--changeset baron:table__event__add__column__max_participants
ALTER TABLE event ADD COLUMN max_participants INTEGER DEFAULT NULL;

--changeset liivanurm:table__event_participation__add__participating
ALTER TABLE event_participation ADD COLUMN participating BOOLEAN DEFAULT NULL;

--changeset liivanurm:change__event_participating__to__non_null
DELETE FROM event_participation WHERE participating is null;
ALTER TABLE event_participation ALTER COLUMN participating SET DEFAULT false;
ALTER TABLE event_participation ALTER COLUMN participating SET NOT NULL;

--changeset maxse:rename__event_participation__to__event_attendance
ALTER TABLE event_participation
    RENAME TO event_attendance;

ALTER TABLE event_attendance
    RENAME CONSTRAINT unique__event__profile TO unique__event__profile__attendance;
ALTER TABLE event_attendance
    RENAME CONSTRAINT fk__event__event_participation TO fk__event__event_attendance;
ALTER TABLE event_attendance
    RENAME CONSTRAINT fk__profile__event_participation TO fk__profile__event_attendance;

ALTER TABLE event
    RENAME COLUMN participation_enabled TO attendance_enabled;

ALTER TABLE event
    RENAME COLUMN max_participants TO max_attendees;

ALTER TABLE event_attendance
    RENAME COLUMN participating TO attending;

--changeset maxse:truncate__table__event_attendance
TRUNCATE TABLE event_attendance;
