#!/bin/sh

cd $(dirname "$0")
JUDO_SCRIPTS=`pwd`

INPUT=$1
OUTPUT=$2
TEMP=$3
#page limit is currently not used since it is not supported by libre-office
PAGE_LIMIT=$4
CONTAINER_VERSION=$5

mkdir -p $TEMP

export TEMP
export TMP="$TEMP"
export TMPDIR="$TEMP"
export MAGICK_TMPDIR="$TEMP"

cleanup() {
  # purge abandoned temp files (pipes that are older than 2 days):
  find /tmp -maxdepth 1 -user $USER -mtime 2 -iname 'OSL_PIPE_900_SingleOfficeIPC_*' -exec rm -r {} \;
}

# Check if file to be converted can be converted by checking file type against a whitelist and blacklist.
# Blacklist takes precedence.
checkIfConversionIsPossible() {
  INPUT=$1

  DETECTED_TYPE="$(file "$INPUT" | cut -d ":" -f 2- | cut -d "," -f 1 | sed -e 's| ||' | sed -e 's| \+| |g')"

  # Only compare IS_ON_BLACKLIST using string comparison, because we might get an empty string here
  # and that might produce errors that can be misunderstood.
  IS_ON_BLACKLIST="$(grep -sc "$DETECTED_TYPE" "$JUDO_SCRIPTS"/default.blacklist)"
  IS_ON_WHITELIST="$(grep -sc "$DETECTED_TYPE" "$JUDO_SCRIPTS"/default.whitelist)"
  IS_MS_OFFICE_ARCHIVE=$(unzip -v "$INPUT" | grep -c "docProps/app.xml")

  # Deny conversion if file is either blacklisted, OR is neither a whitelisted type nor an MS Office archive
  if  [ "$IS_ON_BLACKLIST" = 1 ] || ([ "$IS_ON_WHITELIST" -eq 0 ] && [ "$IS_MS_OFFICE_ARCHIVE" -eq 0 ])
  then
    # document is not supported
    echo "$INPUT of type '$DETECTED_TYPE' can not be converted to PDF."
    exit 2
  fi
}

succeedFastIfPdf() {
  INPUT=$1
  if [ `file $INPUT | grep ": PDF document, version" | wc -l` -eq 1 ]
  then
    # is already a pdf file so just copy
    cp $INPUT $OUTPUT
    echo "SUCCESS: $INPUT of type PDF was moved to $OUTPUT"
    exit 0
  fi
}

killDocker() {
    NAME=$1
    echo "Killing container '$NAME' ..."
    docker kill "$NAME"
    exit 1
}

doConvertWithLibreOffice() {
  CONVERT=$1                                    # Path of the form /tmp/jdoc/{YYYY-mm-dd}/{uuid}/{uuid}.{filetype}
  ABSOLUTE_TEMP=$2                              # Path of the form /tmp/jdoc/{YYYY-mm-dd}/{uuid}/temp
  CONVERTER_WORKSPACE=$(dirname "$ABSOLUTE_TEMP") # Path of the form /tmp/jdoc/{YYYY-mm-dd}/{uuid}/

  # Convert /tmp paths to /home/justsoftware/workspace paths for use inside the container
  DOCKER_WORKSPACE=/home/justsoftware/workspace
  DOCKER_WORKSPACE_TEMP="$DOCKER_WORKSPACE"/$(basename "$ABSOLUTE_TEMP")
  DOCKER_WORKSPACE_CONVERT="$DOCKER_WORKSPACE"/$(basename "$CONVERT")
  DOCKER_CONVERTER_NAME=$(basename "$CONVERTER_WORKSPACE")

  # configure libreoffice
  mkdir -p "$ABSOLUTE_TEMP"/user
  cp "$JUDO_SCRIPTS"/libreoffice-config.xml "$ABSOLUTE_TEMP"/user/registrymodifications.xcu
  chmod -R 777 "$CONVERTER_WORKSPACE"

  # set a trap for the kill signal to ensure the running docker container is killed when the watchdog kills this script
  trap 'killDocker ${DOCKER_CONVERTER_NAME}' 15

  # execute convert
  echo "timeout --preserve-status --signal=TERM --kill-after=30s 15m docker run --name $DOCKER_CONVERTER_NAME --rm --network none -v $CONVERTER_WORKSPACE:$DOCKER_WORKSPACE justsocial/just-pdf-converter:$CONTAINER_VERSION soffice -env:UserInstallation=\"file://$DOCKER_WORKSPACE_TEMP\" --headless --invisible --convert-to pdf $DOCKER_WORKSPACE_CONVERT --outdir $DOCKER_WORKSPACE_TEMP"
  timeout --preserve-status --signal=TERM --kill-after=30s 15m docker run --name "$DOCKER_CONVERTER_NAME" --rm --network none -v "$CONVERTER_WORKSPACE":"$DOCKER_WORKSPACE" justsocial/just-pdf-converter:$CONTAINER_VERSION soffice -env:UserInstallation="file://$DOCKER_WORKSPACE_TEMP" --headless --invisible --convert-to pdf "$DOCKER_WORKSPACE_CONVERT" --outdir "$DOCKER_WORKSPACE_TEMP"
}

doConvertToPdf() {
  # check if we should transform a pdf to pdf and just copy it
  succeedFastIfPdf $INPUT
  # check if we actually should convert the input file and fail if we don't
  checkIfConversionIsPossible $INPUT
  # do the conversion

  # convert to pdf
  cd $TEMP
  ABSOULUTE_TEMP=`pwd`

  # prepare conversion
  echo ">$INPUT<"
  IDENTIFY_STRING=`identify $INPUT || echo "$INPUT binary"`
  DETECTED_EXTENSION=`echo $IDENTIFY_STRING | tr A-Z a-z | cut -d " " -f 2`
  # rename input to file with extension
  CONVERT="${INPUT}.${DETECTED_EXTENSION}"
  echo "rename input to typed input: mv ${INPUT} ${CONVERT}"
  mv ${INPUT} ${CONVERT}

  # trigger conversion
  echo "convert ${CONVERT} with libreoffice to PDF"
  doConvertWithLibreOffice ${CONVERT} ${ABSOULUTE_TEMP}

  echo "restore input to preconversion state: mv ${CONVERT} ${INPUT}"
  mv ${CONVERT} ${INPUT}

  echo "find output pdf"
  find . -name "*.pdf" | while read line
  do
      echo "found output pdf and place result: mv $line $OUTPUT"
      mv $line $OUTPUT
  done

  if [ -f $OUTPUT ]
  then
    echo "SUCCESS: $INPUT of type $DETECTED_TYPE converted to $OUTPUT"
    exit 0
  else
    echo "ERROR: $INPUT of type $DETECTED_TYPE could not be converted to PDF by Just Documents PDF Conversion"
    exit 1
  fi
}

cleanup

doConvertToPdf
