/*
 * Decompiled with CFR 0.152.
 */
package de.justsoftware.jdoc.services.conversion.impl;

import com.google.common.base.MoreObjects;
import com.google.common.collect.EvictingQueue;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Queues;
import de.justsoftware.jdoc.configuration.JustDocumentsEnvironment;
import de.justsoftware.jdoc.model.DocumentStatus;
import de.justsoftware.jdoc.model.DocumentType;
import de.justsoftware.jdoc.services.conversion.exceptions.InvalidShellScriptException;
import de.justsoftware.jdoc.services.conversion.exceptions.ShellScriptExecutionException;
import de.justsoftware.jdoc.services.conversion.impl.AbstractDocumentConverterService;
import de.justsoftware.jdoc.services.tasks.processes.ProcessWatchdogService;
import de.justsoftware.jdoc.services.tasks.processes.model.WatchedProcessId;
import de.justsoftware.jdoc.util.CommandLineOutputUtil;
import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.MeterRegistry;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.util.Map;
import java.util.Queue;
import java.util.concurrent.TimeUnit;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import org.apache.commons.exec.CommandLine;
import org.apache.commons.exec.DefaultExecutor;
import org.apache.commons.exec.ExecuteException;
import org.apache.commons.exec.ExecuteStreamHandler;
import org.apache.commons.exec.PumpStreamHandler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.actuate.health.Health;
import org.springframework.boot.actuate.health.Status;

@ParametersAreNonnullByDefault
public class ShellScriptExecutingConverterService
extends AbstractDocumentConverterService {
    private static final Logger LOG = LoggerFactory.getLogger(ShellScriptExecutingConverterService.class);
    private static final int[] VALID_EXIT_CODES = new int[]{0, 1, 2};
    private static final Map<Integer, DocumentStatus> EXIT_CODE_RESULT_MAP = ImmutableMap.of((Object)VALID_EXIT_CODES[0], (Object)DocumentStatus.PRESENT, (Object)VALID_EXIT_CODES[1], (Object)DocumentStatus.ERROR, (Object)VALID_EXIT_CODES[2], (Object)DocumentStatus.UNSUPPORTED);
    private static final DocumentStatus ERROR_RESULT = DocumentStatus.ERROR;
    private static final long VERY_FAST_CALL = TimeUnit.SECONDS.toMillis(10L);
    private final String _scriptRootPath;
    private final ProcessWatchdogService _watchdogService;
    private final String _pageLimit;
    private final Counter _errorCounter;
    private final Counter _totalCounter;
    private final int _statusQueueSize;
    private final int _errorCountThreshold;
    private final Queue<DocumentStatus> _monitoringStatusQueue;
    private final String _containerVersion;

    public ShellScriptExecutingConverterService(DocumentType sourceType, DocumentType targetType, JustDocumentsEnvironment env, String scriptName, ProcessWatchdogService watchdogService, MeterRegistry meterRegistry, int monitoringQueueSize, int monitoringErrorThreshold) {
        super(scriptName, env.getVersion(), sourceType, targetType);
        this._scriptRootPath = env.getConverterScriptPath();
        this._watchdogService = watchdogService;
        this._pageLimit = String.valueOf(env.getPageLimit());
        this._containerVersion = env.getContainerVersion();
        this._totalCounter = meterRegistry.counter(scriptName + ".conversion.total", new String[0]);
        this._errorCounter = meterRegistry.counter(scriptName + ".conversion.error", new String[0]);
        this._statusQueueSize = monitoringQueueSize;
        this._errorCountThreshold = monitoringErrorThreshold;
        this._monitoringStatusQueue = Queues.synchronizedQueue((Queue)EvictingQueue.create((int)monitoringQueueSize));
    }

    public DocumentStatus process(File source, File target, File tempDirectory, long maxRuntime) {
        this._totalCounter.increment();
        try {
            CommandLine commandLine = this.getScriptCommand().addArgument(source.getAbsolutePath(), false).addArgument(target.getAbsolutePath(), false).addArgument(tempDirectory.getAbsolutePath(), false).addArgument(this._pageLimit, false).addArgument(this._containerVersion, false);
            DocumentStatus result = this.doExecute(this.modifyScriptCommand(commandLine), maxRuntime);
            this.handleResult(result);
            return result;
        }
        catch (ShellScriptExecutionException e) {
            LOG.error("execution of script failed", (Throwable)e);
        }
        catch (InvalidShellScriptException e) {
            LOG.error("script configuration invalid", (Throwable)e);
        }
        this.handleResult(ERROR_RESULT);
        return ERROR_RESULT;
    }

    private void handleResult(DocumentStatus result) {
        this._monitoringStatusQueue.add(result);
        if (result == DocumentStatus.ERROR) {
            this._errorCounter.increment();
        }
    }

    @Nonnull
    protected CommandLine modifyScriptCommand(CommandLine commandLine) {
        return commandLine;
    }

    @Nonnull
    private CommandLine getScriptCommand() throws InvalidShellScriptException {
        File scriptFile = this.getScriptFile();
        if (!scriptFile.exists() || !scriptFile.isFile()) {
            this.exitWithError(scriptFile, "is missing.");
        }
        if (!scriptFile.canRead()) {
            this.exitWithError(scriptFile, "can't be read.");
        }
        if (!this.makeExecutable(scriptFile)) {
            this.exitWithError(scriptFile, "can't execute.");
        }
        return new CommandLine(scriptFile);
    }

    @Nonnull
    private DocumentStatus doExecute(CommandLine cmdLine, long maxMillisToRun) throws ShellScriptExecutionException {
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        ByteArrayOutputStream err = new ByteArrayOutputStream();
        DefaultExecutor executor = new DefaultExecutor();
        executor.setStreamHandler((ExecuteStreamHandler)new PumpStreamHandler((OutputStream)out, (OutputStream)err));
        executor.setExitValues(VALID_EXIT_CODES);
        WatchedProcessId watchedId = this._watchdogService.addProcessToWatch(cmdLine, maxMillisToRun);
        DocumentStatus resultStatus = ERROR_RESULT;
        try {
            int exitCode = executor.execute(cmdLine);
            resultStatus = (DocumentStatus)MoreObjects.firstNonNull((Object)((DocumentStatus)EXIT_CODE_RESULT_MAP.get(exitCode)), (Object)ERROR_RESULT);
            if (LOG.isTraceEnabled() || !DocumentStatus.PRESENT.equals((Object)resultStatus)) {
                LOG.info(CommandLineOutputUtil.format((CommandLine)cmdLine, (ByteArrayOutputStream)out, (ByteArrayOutputStream)err));
            }
        }
        catch (ExecuteException e) {
            LOG.error(CommandLineOutputUtil.format((CommandLine)cmdLine, (ByteArrayOutputStream)out, (ByteArrayOutputStream)err) + " -> errorCode : " + e.getExitValue());
        }
        catch (IOException e) {
            throw new ShellScriptExecutionException(cmdLine, out, err, (Exception)e);
        }
        finally {
            this._watchdogService.removeProcessFromWatch(watchedId);
        }
        return resultStatus;
    }

    private boolean makeExecutable(File scriptFile) {
        CommandLine cmdLine = new CommandLine("chmod");
        cmdLine.addArgument("+x");
        cmdLine.addArgument(scriptFile.getAbsolutePath(), false);
        try {
            this.doExecute(cmdLine, VERY_FAST_CALL);
            return true;
        }
        catch (ShellScriptExecutionException e) {
            LOG.error("script " + scriptFile.getAbsolutePath() + " not executable", (Throwable)e);
            return false;
        }
    }

    private void exitWithError(File scriptFile, String specialMessage) throws InvalidShellScriptException {
        this.exitWithError(scriptFile, specialMessage, null);
    }

    private void exitWithError(File scriptFile, String specialMessage, @Nullable Exception e) throws InvalidShellScriptException {
        String errorString = "conversion script " + scriptFile.getAbsolutePath() + " for conversion from " + String.valueOf(this.getSourceType()) + " to " + String.valueOf(this.getTargetType()) + " " + specialMessage;
        if (e != null) {
            throw new InvalidShellScriptException(errorString, (Throwable)e);
        }
        throw new InvalidShellScriptException(errorString);
    }

    public boolean isActivated() {
        File scriptFile = this.getScriptFile();
        return scriptFile.exists() && scriptFile.isFile() && scriptFile.canRead();
    }

    @Nonnull
    private File getScriptFile() {
        return new File(this._scriptRootPath, this.getName());
    }

    protected void doHealthCheck(Health.Builder builder) {
        Status status = Status.UP;
        long errorCount = this._monitoringStatusQueue.stream().filter(arg_0 -> DocumentStatus.ERROR.equals(arg_0)).count();
        builder.withDetail("monitoring state error/conversions", (Object)String.format("%d/%d", errorCount, this._monitoringStatusQueue.size()));
        builder.withDetail("error threshold", (Object)String.format("%d out of %d", this._errorCountThreshold, this._statusQueueSize));
        builder.withDetail("errors since startup", (Object)this._errorCounter.count());
        builder.withDetail("total since startup", (Object)this._totalCounter.count());
        if (errorCount >= (long)this._errorCountThreshold) {
            status = Status.DOWN;
        }
        builder.status(status);
    }

    public void resetStatus() {
        LOG.info("{} resetting monitoringStatusQueue by clearing entries", (Object)this.getName());
        this._monitoringStatusQueue.clear();
    }
}

