--liquibase formatted sql

--changeset christian.ewers:add_timestamps dbms:postgresql runAlways:true splitStatements:false
CREATE OR REPLACE FUNCTION update_mtime()
RETURNS TRIGGER AS $$
BEGIN
   NEW.mtime = clock_timestamp();
   RETURN NEW;
END;
$$ language 'plpgsql';

CREATE OR REPLACE FUNCTION ctime_mtime_columns(tname TEXT)
RETURNS VOID AS $$
BEGIN
    EXECUTE 'ALTER TABLE ' || tname || ' ADD COLUMN ctime TIMESTAMP WITHOUT TIME ZONE NOT NULL DEFAULT clock_timestamp();';
    EXECUTE 'ALTER TABLE ' || tname || ' ADD COLUMN mtime TIMESTAMP WITHOUT TIME ZONE NOT NULL DEFAULT clock_timestamp();';
    EXECUTE 'CREATE TRIGGER ' || tname || '_mtrg BEFORE UPDATE ON ' || tname || ' FOR EACH ROW EXECUTE PROCEDURE update_mtime();';
END;
$$ language 'plpgsql';

--changeset christian.ewers:add_now_function dbms:oracle runAlways:true splitStatements:false
CREATE OR REPLACE FUNCTION now
    RETURN DATE
AS
BEGIN
    RETURN SYSDATE;
END now;

--changeset christian.ewers:add_clock_timestamp_function dbms:oracle runAlways:true splitStatements:false
CREATE OR REPLACE FUNCTION clock_timestamp
    RETURN TIMESTAMP WITH TIME ZONE
AS
BEGIN
    RETURN SYSTIMESTAMP;
END clock_timestamp;

--changeset christian.ewers:add_timestamps dbms:oracle runAlways:true splitStatements:false
CREATE OR REPLACE PROCEDURE ctime_mtime_columns(tname VARCHAR2)
IS
BEGIN
    EXECUTE IMMEDIATE 'ALTER TABLE ' || tname || ' ADD (ctime TIMESTAMP DEFAULT SYSDATE NOT NULL,mtime TIMESTAMP DEFAULT SYSTIMESTAMP NOT NULL )';
    EXECUTE IMMEDIATE 'CREATE TRIGGER ' || tname || '_mtrg BEFORE UPDATE ON ' || tname || ' FOR EACH ROW BEGIN :new.mtime := SYSTIMESTAMP; END;';
END ctime_mtime_columns;

--changeset baron:create_initial_tables
CREATE TABLE profile (
    id ${varchar64} NOT NULL PRIMARY KEY,
    first_name ${text} NOT NULL,
    last_name ${text} NOT NULL,
    email ${text} NOT NULL,
    image_url ${text} NOT NULL,
    status ${text} NOT NULL
);
${execute} ctime_mtime_columns('profile');

CREATE TABLE channel (
    id ${varchar64} NOT NULL PRIMARY KEY,
    title ${text} NOT NULL,
    tenant_id ${varchar64} NOT NULL,
    deleted ${bool} NOT NULL,
    deleted_at ${timestamp} DEFAULT NULL,
    deleted_by ${varchar64} DEFAULT NULL,
    create_date ${timestamp} NOT NULL,
    modify_date ${timestamp} NOT NULL
);
${execute} ctime_mtime_columns('channel');

CREATE TABLE personal_subscription (
    id ${varchar64} NOT NULL PRIMARY KEY,
    channel_id ${varchar64} NOT NULL,
    profile_id ${varchar64} NOT NULL,
    notifications_enabled ${bool} NOT NULL,
    user_has_unsubscribed ${bool} NOT NULL,
    CONSTRAINT unique__channel__profile
        UNIQUE(channel_id, profile_id),
    CONSTRAINT fk__channel__personal_subscription
        FOREIGN KEY(channel_id)
        REFERENCES channel(id)
        ON DELETE CASCADE,
    CONSTRAINT fk__profile__subscription
        FOREIGN KEY(profile_id)
        REFERENCES profile(id)
        ON DELETE CASCADE
);
${execute} ctime_mtime_columns('personal_subscription');

CREATE TABLE managed_subscription (
    id ${varchar64} NOT NULL PRIMARY KEY,
    channel_id ${varchar64} NOT NULL,
    type ${varchar64} NOT NULL,
    user_id ${varchar64} NOT NULL,
    CONSTRAINT unique_managed_subscription__channel__user
        UNIQUE (channel_id, user_id),
    CONSTRAINT fk__channel__managed_subscription
        FOREIGN KEY(channel_id)
        REFERENCES channel(id)
        ON DELETE CASCADE
);
${execute} ctime_mtime_columns('managed_subscription');

CREATE TABLE user_group (
    id ${varchar64} NOT NULL PRIMARY KEY,
    name ${text} NOT NULL,
    tenant_id ${varchar64} NOT NULL
);
${execute} ctime_mtime_columns('user_group');

CREATE TABLE post (
    id ${varchar64} NOT NULL PRIMARY KEY,
    author_id ${varchar64} NOT NULL,
    channel_id ${varchar64} NOT NULL,
    create_date ${timestamp} NOT NULL,
    modify_date ${timestamp} NOT NULL,
    title ${text} NOT NULL,
    text ${text} NOT NULL,
    json_content ${text} NOT NULL,
    cover_element_id ${varchar64},
    pinned ${bool} DEFAULT ${false} NOT NULL,
    pinned_at ${timestamp},
    publish_date ${timestamp} NOT NULL,
    published ${bool} NOT NULL,
    deleted ${bool} NOT NULL,
    deleted_at ${timestamp} DEFAULT NULL,
    deleted_by ${varchar64} DEFAULT NULL,
    CONSTRAINT fk__author__post
        FOREIGN KEY(author_id)
        REFERENCES profile(id),
    CONSTRAINT fk__channel__post
        FOREIGN KEY(channel_id)
        REFERENCES channel(id)
        ON DELETE CASCADE
);
${execute} ctime_mtime_columns('post');

CREATE TABLE attachment (
    id ${varchar64} NOT NULL PRIMARY KEY,
    post_id ${varchar64} NOT NULL,
    mime_type ${text} NOT NULL,
    name ${text} NOT NULL,
    file_size ${int64} DEFAULT 0 NOT NULL,
    CONSTRAINT fk__post__attachment
        FOREIGN KEY(post_id)
        REFERENCES post(id)
        ON DELETE CASCADE
);
${execute} ctime_mtime_columns('attachment');

CREATE TABLE poll (
    id ${varchar64} NOT NULL PRIMARY KEY,
    question ${text} NOT NULL,
    allow_multiple_answers ${bool} NOT NULL,
    post_id ${varchar64} NOT NULL,
    CONSTRAINT fk__post__poll
        FOREIGN KEY(post_id)
        REFERENCES post(id)
        ON DELETE CASCADE
);
${execute} ctime_mtime_columns('poll');

CREATE TABLE poll_answer (
    id ${varchar64} NOT NULL PRIMARY KEY,
    text ${text} NOT NULL,
    position ${int32} NOT NULL,
    poll_id ${varchar64} NOT NULL,
    CONSTRAINT fk__poll__poll_answer
        FOREIGN KEY(poll_id)
        REFERENCES poll(id)
        ON DELETE CASCADE
);
${execute} ctime_mtime_columns('poll_answer');

CREATE TABLE poll_vote (
    id ${varchar64} NOT NULL PRIMARY KEY,
    poll_id ${varchar64} NOT NULL,
    answer_id ${varchar64} NOT NULL,
    profile_id ${varchar64} NOT NULL,
    CONSTRAINT unique__answer__profile
        UNIQUE (answer_id, profile_id),
    CONSTRAINT fk__poll_answer__poll_vote
        FOREIGN KEY(answer_id)
        REFERENCES poll_answer(id)
        ON DELETE CASCADE
);
${execute} ctime_mtime_columns('poll_vote');

CREATE TABLE post_comment (
    id ${varchar64} NOT NULL PRIMARY KEY,
    post_id ${varchar64} NOT NULL,
    parent_id ${varchar64} NOT NULL,
    reference_id ${varchar64} NOT NULL,
    author_id ${varchar64} NOT NULL,
    created_at ${timestamp} NOT NULL,
    updated_at ${timestamp} NOT NULL,
    last_updated_by ${varchar64},
    json_content ${text} NOT NULL,
    text ${text} NOT NULL,
    deleted ${bool} DEFAULT ${false} NOT NULL,
    deleted_at ${timestamp} DEFAULT NULL,
    deleted_by ${varchar64} DEFAULT NULL,
    CONSTRAINT fk__author__post_comment
        FOREIGN KEY(author_id)
        REFERENCES profile(id),
    CONSTRAINT fk__post__post_comment
        FOREIGN KEY(post_id)
        REFERENCES post(id)
        ON DELETE CASCADE
);
${execute} ctime_mtime_columns('post_comment');

CREATE TABLE reaction (
    id ${varchar64} NOT NULL PRIMARY KEY,
    emoji ${varchar1024} NOT NULL,
    item_id ${varchar64} NOT NULL,
    profile_id ${varchar64} NOT NULL
);
${execute} ctime_mtime_columns('reaction');

CREATE TABLE permission (
    id ${varchar64} NOT NULL PRIMARY KEY,
    item_id ${varchar64} NOT NULL,
    grantee_id ${varchar64} NOT NULL,
    role ${varchar1024} NOT NULL,
    CONSTRAINT unique__item_id__grantee_id__role
        UNIQUE (item_id, grantee_id, role)
);
${execute} ctime_mtime_columns('permission');

CREATE TABLE tenant (
    id ${varchar64} NOT NULL PRIMARY KEY,
    deepl_key ${varchar64}
);
${execute} ctime_mtime_columns('tenant');

--changeset baron:create_table__user_group_member
CREATE TABLE user_group_member (
    id ${varchar1024} NOT NULL PRIMARY KEY,
    user_group_id ${varchar64} NOT NULL,
    profile_id ${varchar64} NOT NULL,
    CONSTRAINT unique__user_group_id__profile_id
        UNIQUE (user_group_id, profile_id)
);
${execute} ctime_mtime_columns('user_group_member');

--changeset baron:create_table_news_cover
CREATE TABLE news_cover (
    tenant_id ${varchar64} NOT NULL PRIMARY KEY,
    open_on_start ${bool} NOT NULL,
    cycle_delay ${int32} NOT NULL,
    CONSTRAINT fk__tenant__news_cover
        FOREIGN KEY(tenant_id)
        REFERENCES tenant(id)
        ON DELETE CASCADE
);
${execute} ctime_mtime_columns('news_cover');

CREATE TABLE news_cover_section (
    id ${varchar64} NOT NULL PRIMARY KEY,
    title ${text} NOT NULL,
    pinned_only ${bool} NOT NULL,
    position ${int32} NOT NULL,
    tenant_id ${varchar64} NOT NULL,
    CONSTRAINT fk__news_cover__news_cover_section
        FOREIGN KEY(tenant_id)
        REFERENCES news_cover(tenant_id)
        ON DELETE CASCADE
);
${execute} ctime_mtime_columns('news_cover_section');

CREATE TABLE news_cover_section_channel (
    section_id ${varchar64} NOT NULL,
    channel_id ${varchar64} NOT NULL,
    CONSTRAINT fk__news_cover_section__news_cover_section_channel
        FOREIGN KEY(section_id)
        REFERENCES news_cover_section(id)
        ON DELETE CASCADE,
    CONSTRAINT unique__news_cover_section__channel
        UNIQUE (section_id, channel_id)
);
${execute} ctime_mtime_columns('news_cover_section_channel');

--changeset baron:add_post_read_mark
CREATE TABLE post_read_mark(
    id ${varchar64} NOT NULL PRIMARY KEY,
    post_id ${varchar64} NOT NULL,
    profile_id ${varchar64} NOT NULL,
    post_published_date ${timestamp} NOT NULL,
    CONSTRAINT fk__post__post_read_mark
        FOREIGN KEY(post_id)
        REFERENCES post(id)
        ON DELETE CASCADE,
    CONSTRAINT fk__profile__post_read_mark
        FOREIGN KEY(profile_id)
        REFERENCES profile(id)
        ON DELETE CASCADE,
    CONSTRAINT unique__post_id__profile_id
        UNIQUE (post_id, profile_id)
);
${execute} ctime_mtime_columns('post_read_mark');

--changeset baron:permission__alter__column__item_id
ALTER TABLE permission RENAME COLUMN item_id to channel_id;
ALTER TABLE permission ADD CONSTRAINT fk__channel__permission
    FOREIGN KEY(channel_id)
    REFERENCES channel(id)
    ON DELETE CASCADE;

--changeset baron:permission__rename_table
ALTER TABLE permission DROP CONSTRAINT unique__item_id__grantee_id__role;
ALTER TABLE permission ADD CONSTRAINT unique__channel_id__grantee_id
    UNIQUE (channel_id, grantee_id);
ALTER TABLE permission RENAME to channel_permission;

--changeset liivanurm:add_migrated_item
CREATE TABLE migrated_item (
    item_id ${varchar64} NOT NULL PRIMARY KEY,
    legacy_id ${varchar255} NOT NULL,
    CONSTRAINT unique__item_id__legacy_id
        UNIQUE (item_id, legacy_id)
);
${execute} ctime_mtime_columns('migrated_item');

--changeset maxse:add_migration_error
CREATE TABLE migration_error (
    id ${varchar64} NOT NULL PRIMARY KEY,
    item_id ${varchar64},
    reason ${text},
    data ${text}
);
${execute} ctime_mtime_columns('migration_error');

--changeset liivanurm:alter_attachments_add_position
ALTER TABLE attachment ADD position ${int32} DEFAULT 0 NOT NULL;

--changeset baron:alter_profile_add_prefix_suffix
ALTER TABLE profile ADD prefix ${text};
ALTER TABLE profile ADD suffix ${text};

--changeset baron:add_settings
CREATE TABLE settings (
    tenant_id ${varchar64} NOT NULL PRIMARY KEY,
    all_may_read_subscription_profiles ${bool} DEFAULT 't' NOT NULL,
    CONSTRAINT fk__tenant__settings
        FOREIGN KEY(tenant_id)
        REFERENCES tenant(id)
        ON DELETE CASCADE
);
${execute} ctime_mtime_columns('settings');
INSERT INTO settings (tenant_id) SELECT id FROM tenant;

--changeset liivanurm:add_managed_subscription_where_channel_permission_exists dbms:postgresql
INSERT INTO managed_subscription(
SELECT concat('MANAGED_SUBSCRIPTION,', gen_random_uuid()), cp.channel_id, 'NO_RECOMMENDATION', cp.grantee_id
FROM channel_permission cp
WHERE NOT EXISTS (SELECT * FROM managed_subscription ms WHERE cp.grantee_id = ms.user_id));

--changeset liivanurm:fix_add_managed_subscription_for_all_channels dbms:postgresql
INSERT INTO managed_subscription(
SELECT concat('MANAGED_SUBSCRIPTION,', gen_random_uuid()), cp.channel_id, 'NO_RECOMMENDATION', cp.grantee_id
FROM channel_permission cp
WHERE NOT EXISTS (SELECT * FROM managed_subscription ms WHERE cp.channel_id = ms.channel_id));

--changeset maxse:alter_poll_add_privacy
ALTER TABLE poll ADD privacy ${varchar1024} DEFAULT 'PUBLIC' NOT NULL;

--changeset twuebbena:alter_settings_add_allow_additional_info_in_mail_notification
ALTER TABLE settings ADD allow_additional_info_in_mail_notification ${bool} DEFAULT 'f' NOT NULL;

--changeset cewers:alter_types_of_profile_table dbms:oracle
ALTER TABLE profile ADD last_name_TEMP ${varchar255};
UPDATE profile SET last_name_TEMP = DBMS_LOB.SUBSTR (last_name, 255);
ALTER TABLE profile MODIFY last_name_TEMP NOT NULL;
ALTER TABLE profile DROP COLUMN last_name;
ALTER TABLE profile RENAME COLUMN last_name_TEMP TO last_name;

ALTER TABLE profile ADD first_name_TEMP ${varchar255};
UPDATE profile  SET first_name_TEMP = DBMS_LOB.SUBSTR (first_name, 255);
ALTER TABLE profile MODIFY first_name_TEMP NOT NULL;
ALTER TABLE profile DROP COLUMN first_name;
ALTER TABLE profile RENAME COLUMN first_name_TEMP TO first_name;

ALTER TABLE profile ADD email_TEMP ${emailAddress};
UPDATE profile SET email_TEMP = DBMS_LOB.SUBSTR (email, 256);
ALTER TABLE profile MODIFY email_TEMP NOT NULL;
ALTER TABLE profile DROP COLUMN email;
ALTER TABLE profile RENAME COLUMN email_TEMP TO email;

ALTER TABLE profile ADD image_url_TEMP ${varchar1024};
UPDATE profile SET image_url_TEMP = DBMS_LOB.SUBSTR (image_url, 1024);
ALTER TABLE profile MODIFY image_url_TEMP NOT NULL;
ALTER TABLE profile DROP COLUMN image_url;
ALTER TABLE profile RENAME COLUMN image_url_TEMP TO image_url;

ALTER TABLE profile ADD status_TEMP ${varchar255};
UPDATE profile SET status_TEMP = DBMS_LOB.SUBSTR (status, 255);
ALTER TABLE profile MODIFY status_TEMP NOT NULL;
ALTER TABLE profile DROP COLUMN status;
ALTER TABLE profile RENAME COLUMN status_TEMP TO status;

ALTER TABLE profile ADD prefix_TEMP ${varchar255};
UPDATE profile SET prefix_TEMP = DBMS_LOB.SUBSTR (prefix, 255);
ALTER TABLE profile DROP COLUMN prefix;
ALTER TABLE profile RENAME COLUMN prefix_TEMP TO prefix;

ALTER TABLE profile ADD suffix_TEMP ${varchar255};
UPDATE profile SET suffix_TEMP = DBMS_LOB.SUBSTR (suffix, 255);
ALTER TABLE profile DROP COLUMN suffix;
ALTER TABLE profile RENAME COLUMN suffix_TEMP TO suffix;

--changeset cewers:alter_types_of_profile_table dbms:postgresql
ALTER TABLE profile ALTER COLUMN last_name TYPE ${varchar255};
ALTER TABLE profile ALTER COLUMN first_name TYPE ${varchar255};
ALTER TABLE profile ALTER COLUMN email TYPE ${emailAddress};
ALTER TABLE profile ALTER COLUMN image_url TYPE ${varchar1024};
ALTER TABLE profile ALTER COLUMN status TYPE ${varchar255};
ALTER TABLE profile ALTER COLUMN prefix TYPE ${varchar255};
ALTER TABLE profile ALTER COLUMN suffix TYPE ${varchar255};

--changeset liivanurm:rename_post_mark_read_to_post_UNread_mark
ALTER TABLE post_read_mark RENAME to post_unread_mark;

--changeset baron:post__add__column__edited_by
ALTER TABLE post ADD edited_by ${varchar64} DEFAULT NULL;

--changeset baron:drop__table__migration_error
DROP TABLE migration_error;

--changeset maxse:add_created_by_to_channel
ALTER TABLE channel ADD created_by ${varchar64} DEFAULT NULL;

--changeset maxse:add_edited_by_to_channel
ALTER TABLE channel ADD edited_by ${varchar64} DEFAULT NULL;

--changeset maxse:post_drop__pinned__add__pin_expiration dbms:postgresql
ALTER TABLE post ADD COLUMN pin_expiration ${timestamp} DEFAULT NULL;
UPDATE post
    SET pin_expiration =
        CASE
            WHEN published = ${true} THEN (now() + INTERVAL '30 days')
            ELSE (publish_date + INTERVAL '30 days')
        END
    WHERE pinned = ${true};
ALTER TABLE post DROP COLUMN pinned;

--changeset maxse:post_drop__pinned__add__pin_expiration dbms:oracle
ALTER TABLE post ADD pin_expiration ${timestamp} DEFAULT NULL;
UPDATE post
    SET pin_expiration =
        CASE
            WHEN published = ${true} THEN (SELECT (sysdate + 30) FROM dual)
            ELSE (publish_date + 30)
        END
    WHERE pinned = ${true};
ALTER TABLE post DROP COLUMN pinned;

--changeset christian.ewers:add_index_to_reactions_and_comments dbms:postgresql
DROP INDEX IF EXISTS reaction_item_idx;
CREATE INDEX reaction_item_idx ON reaction (item_id);
DROP INDEX IF EXISTS post_comment_parent_idx;
CREATE INDEX post_comment_parent_idx ON post_comment (parent_id);

--changeset christian.ewers:add_index_to_reactions_and_comments dbms:oracle
CREATE INDEX reaction_item_idx ON reaction (item_id);
CREATE INDEX post_comment_parent_idx ON post_comment (parent_id);

--changeset christian.ewers:add_publish_date_idx
create index post_publish_date_idx ON post(publish_date);

--changeset baron:create__table__event
CREATE TABLE event (
    id ${varchar64} NOT NULL PRIMARY KEY,
    post_id ${varchar64} NOT NULL,
    start_date ${timestamp} NOT NULL,
    end_date ${timestamp} NOT NULL,
    CONSTRAINT fk__post__event
        FOREIGN KEY(post_id)
        REFERENCES post(id)
        ON DELETE CASCADE
);
${execute} ctime_mtime_columns('event');

--changeset baron:create__index__event__post_id
create index event_post_id_idx ON event(post_id);

--changeset baron:add__column__delete_date dbms:postgresql
ALTER TABLE post ADD COLUMN delete_date ${timestamp} DEFAULT NULL;

--changeset baron:add__column__delete_date dbms:oracle
ALTER TABLE post ADD delete_date ${timestamp} DEFAULT NULL;

--changeset baron:alter__poll_vote__add__column__vote_date
ALTER TABLE poll_vote ADD vote_date ${timestamp} DEFAULT ${now} NOT NULL;
UPDATE poll_vote SET vote_date = ctime;

--changeset liivanurm:add__column__comments_disabled
ALTER TABLE post ADD comments_disabled ${bool} DEFAULT 'f' NOT NULL;

--changeset liivanurm:create__indexes__for__post_unread_mark__attachment__post__channel dbms:postgresql
DROP INDEX IF EXISTS post_unread_mark_profile_id_idx;
CREATE INDEX post_unread_mark_profile_id_idx ON post_unread_mark (profile_id);
DROP INDEX IF EXISTS attachment_post_id_idx;
CREATE INDEX attachment_post_id_idx ON attachment (post_id);
DROP INDEX IF EXISTS post_channel_id_idx;
CREATE INDEX post_channel_id_idx ON post (channel_id);

--changeset liivanurm:create__indexes__for__post_unread_mark__attachment__post__channel dbms:oracle
CREATE INDEX post_unread_mark_profile_id_idx ON post_unread_mark (profile_id);
CREATE INDEX attachment_post_id_idx ON attachment (post_id);
CREATE INDEX post_channel_id_idx ON post (channel_id);

--changeset liivanurm:fix_add__column__comments_disabled__in__oracle dbms:oracle
UPDATE post SET comments_disabled = 0 where comments_disabled = 'f';
ALTER TABLE post MODIFY comments_disabled DEFAULT 0;

--changeset liivanurm:fix_settings_boolean_values dbms:oracle
UPDATE settings SET all_may_read_subscription_profiles  = 0 where all_may_read_subscription_profiles = 'f';
UPDATE settings SET all_may_read_subscription_profiles  = 1 where all_may_read_subscription_profiles = 't';
UPDATE settings SET allow_additional_info_in_mail_notification  = 0 where allow_additional_info_in_mail_notification = 'f';
UPDATE settings SET allow_additional_info_in_mail_notification  = 1 where allow_additional_info_in_mail_notification = 't';
ALTER TABLE settings MODIFY allow_additional_info_in_mail_notification DEFAULT 0;
ALTER TABLE settings MODIFY all_may_read_subscription_profiles DEFAULT 1;

--changeset baron:create__table__comment_notifications dbms:postgresql
CREATE TABLE comment_notifications (
    id BIGSERIAL PRIMARY KEY NOT NULL,
    profile_id ${varchar64} NOT NULL,
    post_id ${varchar64} NOT NULL,
    enabled ${bool} NOT NULL,
    CONSTRAINT fk__post__comment_notifications
        FOREIGN KEY(post_id)
        REFERENCES post(id)
        ON DELETE CASCADE,
    CONSTRAINT unique__post__profile
        UNIQUE(profile_id, post_id)
);
${execute} ctime_mtime_columns('comment_notifications');

--changeset baron:create__table__comment_notifications dbms:oracle
CREATE TABLE comment_notifications (
    id NUMBER GENERATED by default on null as IDENTITY PRIMARY KEY NOT NULL,
    profile_id ${varchar64} NOT NULL,
    post_id ${varchar64} NOT NULL,
    enabled ${bool} NOT NULL,
    CONSTRAINT fk__post__comment_notifications
        FOREIGN KEY(post_id)
        REFERENCES post(id)
        ON DELETE CASCADE,
    CONSTRAINT unique__post__profile
        UNIQUE(profile_id, post_id)
);
${execute} ctime_mtime_columns('comment_notifications');

--changeset baron:add__profile_idx__to__comment_notifications
CREATE INDEX profile_idx ON comment_notifications (profile_id);

--changeset baron:add__post_idx__to__comment_notifications
CREATE INDEX post_idx ON comment_notifications (post_id);

--changeset maxse:drop__column_deepl_key__on__tenant_table
ALTER TABLE tenant DROP COLUMN deepl_key;

--changeset liivanurm:allow__only__unique__reactions
ALTER TABLE reaction ADD CONSTRAINT unique__item_id__profile_id__emoji UNIQUE (item_id, profile_id, emoji);

--changeset baron:attachment__alter__column__post_id
ALTER TABLE attachment DROP CONSTRAINT fk__post__attachment;
ALTER TABLE attachment RENAME COLUMN post_id to item_id;

--changeset baron:add__item_idx__to__attachment dbms:postgresql
DROP INDEX IF EXISTS attachment_post_id_idx;
CREATE INDEX attachment_item_id_idx ON attachment (item_id);

--changeset baron:add__item_idx__to__attachment dbms:oracle
DROP INDEX attachment_post_id_idx;
CREATE INDEX attachment_item_id_idx ON attachment (item_id);

--changeset baron:post_comment__alter__column__json_content dbms:postgresql
ALTER TABLE post_comment ALTER COLUMN json_content DROP NOT NULL;

--changeset baron:post_comment__alter__column__text dbms:postgresql
ALTER TABLE post_comment ALTER COLUMN text DROP NOT NULL;

--changeset baron:post_comment__alter__column__json_content dbms:oracle
ALTER TABLE post_comment MODIFY json_content NULL;

--changeset baron:post_comment__alter__column__text dbms:oracle
ALTER TABLE post_comment MODIFY text NULL;
