--liquibase formatted sql

--changeset fzuellich:add_timestamps dbms:postgresql runAlways:true splitStatements:false
CREATE OR REPLACE FUNCTION update_mtime()
RETURNS TRIGGER AS $$
BEGIN
   NEW.mtime = clock_timestamp();
   RETURN NEW;
END;
$$ language 'plpgsql';

CREATE OR REPLACE FUNCTION ctime_mtime_columns(tname TEXT)
RETURNS VOID AS $$
BEGIN
    EXECUTE 'ALTER TABLE ' || tname || ' ADD COLUMN ctime TIMESTAMP WITHOUT TIME ZONE NOT NULL DEFAULT clock_timestamp();';
    EXECUTE 'ALTER TABLE ' || tname || ' ADD COLUMN mtime TIMESTAMP WITHOUT TIME ZONE NOT NULL DEFAULT clock_timestamp();';
    EXECUTE 'CREATE TRIGGER ' || tname || '_mtrg BEFORE UPDATE ON ' || tname || ' FOR EACH ROW EXECUTE PROCEDURE update_mtime();';
END;
$$ language 'plpgsql';

--changeset fzuellich:add_now_function dbms:oracle runAlways:true splitStatements:false
CREATE OR REPLACE FUNCTION now
    RETURN DATE
AS
BEGIN
    RETURN SYSDATE;
END now;

--changeset fzuellich:add_clock_timestamp_function dbms:oracle runAlways:true splitStatements:false
CREATE OR REPLACE FUNCTION clock_timestamp
    RETURN TIMESTAMP WITH TIME ZONE
AS
BEGIN
    RETURN SYSTIMESTAMP;
END clock_timestamp;

--changeset fzuellich:add_timestamps dbms:oracle runAlways:true splitStatements:false
CREATE OR REPLACE PROCEDURE ctime_mtime_columns(tname VARCHAR2)
IS
BEGIN
    EXECUTE IMMEDIATE 'ALTER TABLE ' || tname || ' ADD (ctime TIMESTAMP DEFAULT SYSDATE NOT NULL,mtime TIMESTAMP DEFAULT SYSTIMESTAMP NOT NULL )';
    EXECUTE IMMEDIATE 'CREATE TRIGGER ' || tname || '_mtrg BEFORE UPDATE ON ' || tname || ' FOR EACH ROW BEGIN :new.mtime := SYSTIMESTAMP; END;';
END ctime_mtime_columns;

--changeset jkaiser:add_tenant
CREATE TABLE tenant (
    id ${uuid} NOT NULL PRIMARY KEY,
    name ${varchar1024} NOT NULL
);

--changeset jkaiser:add_app_settings
CREATE TABLE app_settings (
    PRIMARY KEY (tenant_id, app),
    tenant_id ${uuid} NOT NULL REFERENCES tenant (id) ON DELETE CASCADE,
    app ${varchar1024} NOT NULL,
    app_color ${varchar1024} NOT NULL
);

--changeset jkaiser:truncate_tenants
TRUNCATE TABLE tenant CASCADE;

--changeset dwaller:shorten_tenant_name dbms:oracle
ALTER TABLE tenant MODIFY name ${varchar255};

--changeset dwaller:shorten_tenant_name dbms:postgresql
ALTER TABLE tenant ALTER COLUMN name type ${varchar255};

--changeset dwaller:shorten_app_color dbms:oracle
ALTER TABLE app_settings MODIFY app_color ${colorHex};

--changeset dwaller:shorten_app_color dbms:postgresql
ALTER TABLE app_settings ALTER COLUMN app_color type ${colorHex};

--changeset jkaiser:add_tenant_settings
CREATE TABLE tenant_settings (
    PRIMARY KEY (tenant_id, settings_attribute),
    tenant_id ${uuid} NOT NULL REFERENCES tenant (id) ON DELETE CASCADE,
    settings_attribute ${varchar1024} NOT NULL,
    settings_value ${varchar1024} NOT NULL
)

--changeset jkaiser:rename_core_app
UPDATE app_settings SET app = 'TASKS' WHERE app = 'LIST';

--changeset jkaiser:create_table_tenant_invite
CREATE TABLE tenant_invitation (
    PRIMARY KEY (tenant_id, tenant_invitation_code),
    tenant_id ${uuid} NOT NULL REFERENCES tenant (id) ON DELETE CASCADE,
    tenant_invitation_code ${varchar64} NOT NULL
);


--changeset baron:create_table_profile
CREATE TABLE profile (
    id ${int64} NOT NULL PRIMARY KEY,
    firstname ${varchar1024} NOT NULL,
    lastname ${varchar1024} NOT NULL,
    image_url ${varchar1024} NOT NULL,
    active ${bool} NOT NULL,
    blocked ${bool} NOT NULL
)

--changeset baron:create_table_user_group
CREATE TABLE user_group (
    id ${uuid} NOT NULL PRIMARY KEY,
    name ${varchar1024} NOT NULL,
    tenant_id ${uuid} NOT NULL REFERENCES tenant (id) ON DELETE CASCADE,
    is_all_tenant_users_group ${bool} NOT NULL
)

--changeset maxse:create_table_app_permission
CREATE TABLE app_permission (
    PRIMARY KEY (tenant_id, app, grantee),
    tenant_id ${uuid} NOT NULL,
    app ${varchar255} NOT NULL,
    grantee ${varchar255} NOT NULL,
    grantee_type ${varchar255} NOT NULL,
    role ${varchar255} NOT NULL
)

--changeset dwaller:delete_old_tenant_settings_table
DROP TABLE tenant_settings

--changeset dwaller:create_tenant_settings
CREATE TABLE tenant_settings (
    PRIMARY KEY (tenant_id),
    tenant_id ${uuid} NOT NULL REFERENCES tenant (id) ON DELETE CASCADE,
    global_highlight_color ${colorHex},
    logo_id ${varchar1024}
)

--changeset dwaller:add_hasManualPermissions_toggle
ALTER TABLE app_settings ADD has_manual_permissions ${bool} DEFAULT 'f' NOT NULL

--changeset dwaller:rename_app_color_to_color
ALTER TABLE app_settings RENAME COLUMN app_color TO color

--changeset baron:create_table_tenant_permission
CREATE TABLE tenant_permission (
    PRIMARY KEY (tenant_id, grantee_id),
    tenant_id ${uuid} NOT NULL REFERENCES tenant (id) ON DELETE CASCADE,
    grantee_id ${varchar255} NOT NULL,
    role ${varchar255} NOT NULL
)

--changeset baron:alter_table_app_permission_drop_grantee_type
ALTER TABLE app_permission DROP COLUMN grantee_type

--changeset jkaiser:update_tenant_settings_logo_id_to_document_id
UPDATE tenant_settings
SET logo_id = 'DOCUMENT,' || logo_id
WHERE
logo_id <> ''
AND logo_id IS NOT NULL
AND logo_id NOT LIKE 'DOCUMENT,%'

--changeset baron:migrate_tenant_settings_global_highlight_color_to_not_null
UPDATE tenant_settings SET global_highlight_color = '#EC004A' WHERE global_highlight_color IS NULL;

--changeset baron:alter_table_tenant_settings_global_highlight_color_not_null dbms:postgresql
ALTER TABLE tenant_settings ALTER COLUMN global_highlight_color SET NOT NULL;

--changeset baron:alter_table_tenant_settings_global_highlight_color_not_null dbms:oracle
ALTER TABLE tenant_settings MODIFY (global_highlight_color NOT NULL);

--changeset msemmler:add_timestamps_to_existing_tables
${execute} ctime_mtime_columns('tenant');
${execute} ctime_mtime_columns('app_settings');
${execute} ctime_mtime_columns('tenant_settings');
${execute} ctime_mtime_columns('tenant_invitation');
${execute} ctime_mtime_columns('profile');
${execute} ctime_mtime_columns('user_group');
${execute} ctime_mtime_columns('app_permission');
${execute} ctime_mtime_columns('tenant_permission');

--changeset msemmler:create_table_deleted_tenants
CREATE TABLE deleted_tenants (
    id ${uuid} NOT NULL PRIMARY KEY,
    name ${varchar1024} NOT NULL,
    deleted_by ${int64} NOT NULL
);
${execute} ctime_mtime_columns('deleted_tenants');

--changeset dwaller:create_table_idp_configuration
CREATE TABLE idp_configuration (
    config_id ${uuid} PRIMARY KEY,
    tenant_id ${uuid} NOT NULL REFERENCES tenant (id) ON DELETE CASCADE,
    client_id ${varchar255} NOT NULL,
    client_secret ${varchar255} NOT NULL,
    redirect_uri ${varchar1024} NOT NULL,
    idp_name ${varchar255} NOT NULL,
    username_attribute ${varchar255} NOT NULL,
    grant_type ${varchar64} NOT NULL,
    scope ${varchar255},
    issuer_uri ${varchar1024},
    authorization_uri ${varchar1024},
    token_uri ${varchar1024},
    user_info_uri ${varchar1024}
);
${execute} ctime_mtime_columns('idp_configuration');

--changeset maxse:drop_column_redirect_uri
ALTER TABLE idp_configuration DROP COLUMN redirect_uri;

--changeset baron:create_table_platform_settings
CREATE TABLE platform_settings (
    id ${int32} PRIMARY KEY,
    is_microsoft_teams_integration_enabled ${bool} NOT NULL
);
${execute} ctime_mtime_columns('platform_settings');

--changeset baron:insert_default_platform_settings dbms:postgresql
INSERT INTO platform_settings (id, is_microsoft_teams_integration_enabled) VALUES (0, false)
--changeset baron:insert_default_platform_settings dbms:oracle
INSERT INTO platform_settings (id, is_microsoft_teams_integration_enabled) VALUES (0, 0)

--changeset baron:drop_platform_settings
DROP table platform_settings;

--changeset baron:tenant_settings_add_column_deepl_key
ALTER TABLE tenant_settings ADD deepl_key ${varchar64}

--changeset dwaller:delete_all_profile_data
DROP table profile;

--changeset dwaller:add_column_email_to_profile_table
CREATE TABLE profile (
    id ${int64} NOT NULL PRIMARY KEY,
    firstname ${varchar1024} NOT NULL,
    lastname ${varchar1024} NOT NULL,
    email ${emailAddress} NOT NULL,
    image_url ${varchar1024} NOT NULL,
    active ${bool} NOT NULL,
    blocked ${bool} NOT NULL
)
--changeset dwaller:make_all_data_nullable dbms:postgresql
ALTER TABLE idp_configuration ALTER COLUMN client_id DROP NOT NULL;
ALTER TABLE idp_configuration ALTER COLUMN client_secret DROP NOT NULL;
ALTER TABLE idp_configuration ALTER COLUMN idp_name DROP NOT NULL;
ALTER TABLE idp_configuration ALTER COLUMN username_attribute DROP NOT NULL;
ALTER TABLE idp_configuration ALTER COLUMN grant_type DROP NOT NULL;

--changeset dwaller:make_all_data_nullable dbms:oracle
ALTER TABLE idp_configuration MODIFY client_id ${varchar255} NULL;
ALTER TABLE idp_configuration MODIFY client_secret ${varchar255} NULL;
ALTER TABLE idp_configuration MODIFY idp_name ${varchar255} NULL;
ALTER TABLE idp_configuration MODIFY username_attribute ${varchar255} NULL;
ALTER TABLE idp_configuration MODIFY grant_type ${varchar64} NULL;

--changeset dwaller:add_column_for_jwk_set_uri
ALTER TABLE idp_configuration ADD jwk_set_uri ${varchar1024};

--changeset cewers:add_client_target_and_mode_switch
ALTER TABLE idp_configuration ADD targets ${varchar1024} DEFAULT 'DESKTOP' NOT NULL ;
ALTER TABLE idp_configuration ADD automatic_mode ${bool} DEFAULT 't' NOT NULL;

--changeset cewers:add_more_idp_toggles
ALTER TABLE idp_configuration ADD default_provider ${bool} DEFAULT 'f' NOT NULL;
ALTER TABLE idp_configuration ADD enabled ${bool} DEFAULT 't' NOT NULL;

--changeset baron:create_table_people_settings
CREATE TABLE people_settings (
    tenant_id ${uuid} NOT NULL REFERENCES tenant (id) ON DELETE CASCADE,
    has_manual_exporters ${bool} DEFAULT 'f' NOT NULL
);
INSERT INTO people_settings (tenant_id) SELECT id FROM tenant;

--changeset cewers:change_default_for_export_permissions
UPDATE people_settings SET has_manual_exporters = 't'

--changeset dwaller:create_profile_tenant_table
CREATE TABLE profile_tenant (
    PRIMARY KEY (profile_id, tenant_id),
    profile_id ${int64} UNIQUE NOT NULL,
    tenant_id ${uuid} NOT NULL REFERENCES tenant (id) ON DELETE CASCADE
);

--changeset dwaller:remove_redundant_MANAGER_permission_for_just_superadmin
DELETE FROM tenant_permission where grantee_id = 'PROFILE,0';

--changeset dwaller:split_all_MOBILE_targets_into_MOBILE_SOCIAL_and_MOBILE_CONNECT dbms:postgresql
UPDATE idp_configuration SET targets = REPLACE(targets, 'MOBILE', 'MOBILE_SOCIAL,MOBILE_CONNECT') WHERE targets like '%MOBILE%';

--changeset baron:tenant_permission_replace_primary_key dbms:postgresql
ALTER TABLE tenant_permission DROP CONSTRAINT tenant_permission_pkey;
ALTER TABLE tenant_permission ADD CONSTRAINT unique__tenant_id__grantee_id__role UNIQUE (tenant_id, grantee_id, role);

--changeset baron:tenant_permission_replace_primary_key dbms:oracle
ALTER TABLE tenant_permission DROP PRIMARY KEY;
ALTER TABLE tenant_permission ADD CONSTRAINT unique__tenant_id__grantee_id__role UNIQUE (tenant_id, grantee_id, role);

--changeset baron:tenant_permission_insert_default_all_users_may_use_all_tenant_users_group
INSERT INTO tenant_permission (tenant_id, grantee_id, role)
SELECT tenant_id, CONCAT('USER_GROUP,', id), 'USE_ALL_TENANT_USERS_GROUP'
FROM user_group WHERE is_all_tenant_users_group = 't';

--changeset baron:app_permission_replace_primary_key dbms:postgresql
ALTER TABLE app_permission DROP CONSTRAINT app_permission_pkey;
ALTER TABLE app_permission ADD CONSTRAINT unique__tenant_id__app__grantee__role UNIQUE (tenant_id, app, grantee, role);

--changeset baron:app_permission_replace_primary_key dbms:oracle
ALTER TABLE app_permission DROP PRIMARY KEY;
ALTER TABLE app_permission ADD CONSTRAINT unique__tenant_id__app__grantee__role UNIQUE (tenant_id, app, grantee, role);

--changeset baron:migrate_people_exporters
INSERT INTO app_permission (tenant_id, app, grantee, role)
SELECT tenant_id, 'PEOPLE', CONCAT('USER_GROUP,', id), 'EXPORTER' FROM user_group WHERE is_all_tenant_users_group = 't'
AND EXISTS (SELECT 1 FROM people_settings WHERE tenant_id = tenant_id AND has_manual_exporters = 'f');

DROP TABLE people_settings;

--changeset baron:remove_manual_permissions_toggle
INSERT INTO app_permission (tenant_id, app, grantee, role)
SELECT tenant_id, 'NEWS', CONCAT('USER_GROUP,', id), 'WRITER' FROM user_group WHERE is_all_tenant_users_group = 't'
AND EXISTS (SELECT 1 FROM app_settings WHERE tenant_id = tenant_id AND app = 'NEWS' AND has_manual_permissions = 'f');

INSERT INTO app_permission (tenant_id, app, grantee, role)
SELECT tenant_id, 'PEOPLE', CONCAT('USER_GROUP,', id), 'WRITER' FROM user_group WHERE is_all_tenant_users_group = 't'
AND EXISTS (SELECT 1 FROM app_settings WHERE tenant_id = tenant_id AND app = 'PEOPLE' AND has_manual_permissions = 'f');

INSERT INTO app_permission (tenant_id, app, grantee, role)
SELECT tenant_id, 'DRIVE', CONCAT('USER_GROUP,', id), 'WRITER' FROM user_group WHERE is_all_tenant_users_group = 't'
AND EXISTS (SELECT 1 FROM app_settings WHERE tenant_id = tenant_id AND app = 'DRIVE' AND has_manual_permissions = 'f');

INSERT INTO app_permission (tenant_id, app, grantee, role)
SELECT tenant_id, 'WIKI', CONCAT('USER_GROUP,', id), 'WRITER' FROM user_group WHERE is_all_tenant_users_group = 't'
AND EXISTS (SELECT 1 FROM app_settings WHERE tenant_id = tenant_id AND app = 'WIKI' AND has_manual_permissions = 'f');

ALTER TABLE app_settings DROP COLUMN has_manual_permissions;

--changeset baron:grant__all_tenant_users_group_chat_create_roles
INSERT INTO app_permission (tenant_id, app, grantee, role)
SELECT tenant_id, 'CHAT', CONCAT('USER_GROUP,', id), 'CHAT_CREATE_ONE_TO_ONE'
FROM user_group WHERE is_all_tenant_users_group = 't';

INSERT INTO app_permission (tenant_id, app, grantee, role)
SELECT tenant_id, 'CHAT', CONCAT('USER_GROUP,', id), 'CHAT_CREATE_MANY_TO_MANY'
FROM user_group WHERE is_all_tenant_users_group = 't';

--changeset baron:rename__column__config_id__to__id
ALTER TABLE idp_configuration RENAME COLUMN config_id TO id;

--changeset baron:create__table__external_app_settings
CREATE TABLE external_app_settings(
    id ${varchar64} NOT NULL PRIMARY KEY,
    name ${varchar4000} NOT NULL,
    url ${varchar4000} NOT NULL,
    color ${colorHex} NOT NULL,
    tenant_id ${uuid} NOT NULL,
    CONSTRAINT fk__tenant__external_app_settings
        FOREIGN KEY(tenant_id)
        REFERENCES tenant(id)
        ON DELETE CASCADE
);

--changeset baron:add__column__position__to__external_app_settings
ALTER TABLE external_app_settings ADD position ${int32} DEFAULT 0 NOT NULL;
ALTER TABLE external_app_settings ADD CONSTRAINT unique__id__position UNIQUE (id, position);

--changeset baron:add__column__id__to__app_settings dbms:postgresql
ALTER TABLE app_settings ADD column id ${varchar64} NULL;
UPDATE app_settings SET id = (SELECT CONCAT('APP,', LOWER(app), '_', tenant_id));
ALTER TABLE app_settings alter column id SET NOT NULL;
ALTER TABLE app_settings DROP CONSTRAINT app_settings_pkey;
ALTER TABLE app_settings ADD PRIMARY KEY (id);

--changeset baron:add__column__id__to__app_settings dbms:oracle
ALTER TABLE app_settings ADD id ${varchar64};
UPDATE app_settings a SET a.id = (SELECT 'APP,' || LOWER(app) || '_' || tenant_id FROM app_settings
WHERE a.TENANT_ID = TENANT_ID AND a.APP = APP);
ALTER TABLE app_settings MODIFY (id NOT NULL);
ALTER TABLE app_settings DROP PRIMARY KEY;
ALTER TABLE app_settings ADD CONSTRAINT app_settings_pkey PRIMARY KEY (id);

--changeset baron:add__column__position__to__app_settings dbms:postgresql
ALTER TABLE app_settings ADD position ${int32};
ALTER TABLE app_settings ADD CONSTRAINT unique__tenant_id__position UNIQUE (tenant_id, position);

--changeset baron:drop__position__constraint__from__app_settings dbms:postgresql
ALTER TABLE app_settings DROP CONSTRAINT unique__tenant_id__position;

--changeset baron:add__column__position__to__app_settings dbms:oracle
ALTER TABLE app_settings ADD position ${int32};

--changeset baron:create__table__app_home_screen_position
CREATE TABLE app_home_screen_position(
    app_id ${varchar64} NOT NULL PRIMARY KEY,
    position ${int32} NOT NULL,
    tenant_id ${uuid} NOT NULL,
    CONSTRAINT fk__tenant__app_home_screen_position
        FOREIGN KEY(tenant_id)
        REFERENCES tenant(id)
        ON DELETE CASCADE,
    CONSTRAINT unique__tenant_id__position UNIQUE (tenant_id, position)
);

--changeset baron:alter__table__app_settings__drop__column__position
ALTER TABLE app_settings DROP COLUMN position

--changeset baron:alter__table__external_app_settings__drop__column__position
ALTER TABLE external_app_settings DROP CONSTRAINT unique__id__position;
ALTER TABLE external_app_settings DROP COLUMN position

--changeset baron:fill__table__app_home_screen_position__with__default__positions
insert into app_home_screen_position (app_id, position, tenant_id)
select id, 0, tenant_id from app_settings as2 where id like 'APP,news_%';
insert into app_home_screen_position (app_id, position, tenant_id)
select id, 1, tenant_id from app_settings as2 where id like 'APP,chat_%';
insert into app_home_screen_position (app_id, position, tenant_id)
select id, 2, tenant_id from app_settings as2 where id like 'APP,wiki_%';
insert into app_home_screen_position (app_id, position, tenant_id)
select id, 3, tenant_id from app_settings as2 where id like 'APP,drive_%';
insert into app_home_screen_position (app_id, position, tenant_id)
select id, 4, tenant_id from app_settings as2 where id like 'APP,people_%';
insert into app_home_screen_position (app_id, position, tenant_id)
select id, 5, tenant_id from app_settings as2 where id like 'APP,admin_%';
insert into app_home_screen_position (app_id, position, tenant_id)
select id, 6, tenant_id from app_settings as2 where id like 'APP,connect_%';
insert into app_home_screen_position (app_id, position, tenant_id)
select id, 7, tenant_id from app_settings as2 where id like 'APP,tasks_%';

--changeset baron:add__column__open_url_in__to__external_app_settings
ALTER TABLE external_app_settings ADD open_url_in ${varchar255} NOT NULL;

--changeset baron:add__column__active__to__app_settings__and__external_app_settings
ALTER TABLE app_settings ADD active ${bool} DEFAULT 't' NOT NULL;
ALTER TABLE external_app_settings ADD active ${bool} DEFAULT 't' NOT NULL;

--changeset baron:create__table__app_placement
CREATE TABLE app_placement(
    app_id ${varchar64} NOT NULL,
    user_id ${varchar64} NOT NULL,
    type ${varchar32} NOT NULL,
    tenant_id ${uuid} NOT NULL,
    CONSTRAINT fk__tenant__app_placement
        FOREIGN KEY(tenant_id)
        REFERENCES tenant(id)
        ON DELETE CASCADE,
    CONSTRAINT unique__app_id__user_id UNIQUE (app_id, user_id)
);

--changeset baron:add__column__icon_id__to__app_settings__and__external_app_settings
ALTER TABLE app_settings ADD icon_id ${varchar64};
ALTER TABLE external_app_settings ADD icon_id ${varchar64};

--changeset baron:add__column__name__to__app_settings dbms:postgresql
ALTER TABLE app_settings ADD name ${varchar1024};
UPDATE app_settings SET name = app;
ALTER TABLE app_settings alter column name SET NOT NULL;

--changeset baron:add__column__name__to__app_settings dbms:oracle
ALTER TABLE app_settings ADD name ${varchar1024};
UPDATE app_settings SET name = app;
ALTER TABLE app_settings MODIFY (name NOT NULL);

--changeset baron:add__columns__ctime__mtime__to__all__tables
${execute} ctime_mtime_columns('external_app_settings');
${execute} ctime_mtime_columns('app_placement');
${execute} ctime_mtime_columns('app_home_screen_position');
${execute} ctime_mtime_columns('profile');
${execute} ctime_mtime_columns('profile_tenant');

--changeset baron:remove__idp_client_target__mobile_connect
UPDATE idp_configuration SET targets = REPLACE(targets, 'MOBILE_CONNECT,', '');
UPDATE idp_configuration SET targets = REPLACE(targets, 'MOBILE_CONNECT', '');

--changeset baron:replace__empty__deepl__keys__with__null
UPDATE tenant_settings SET deepl_key = NULL WHERE deepl_key = '';

--changeset liivanurm:fix__trailing__comma
UPDATE idp_configuration SET targets = rtrim(targets, ',') where targets like '%,';

--changeset baron:add__column__clients__to__external_app_settings
ALTER TABLE external_app_settings ADD clients ${varchar1024};
UPDATE external_app_settings SET clients = 'DESKTOP,MOBILE';

--changeset baron:set__column__clients__not_null dbms:postgresql
ALTER TABLE external_app_settings alter column clients SET NOT NULL;

--changeset baron:set__column__clients__not_null dbms:oracle
ALTER TABLE external_app_settings MODIFY (clients NOT NULL);

--changeset baron:split__app__column__color
ALTER TABLE app_settings RENAME COLUMN color TO app_color;
ALTER TABLE external_app_settings RENAME COLUMN color TO home_screen_app_color;
ALTER TABLE app_settings ADD home_screen_app_color ${colorHex};
UPDATE app_settings SET home_screen_app_color = app_color;

--changeset baron:app_settings__set__column__home_screen_app_color__not_null dbms:postgresql
ALTER TABLE app_settings alter column home_screen_app_color SET NOT NULL;

--changeset baron:app_settings__set__column__home_screen_app_color__not_null dbms:oracle
ALTER TABLE app_settings MODIFY (home_screen_app_color NOT NULL);

--changeset baron:delete__invalid__idp_configs
DELETE FROM idp_configuration where targets = '';

--changeset christian.ewers:add__tracking_settings_table
CREATE TABLE tracking_settings(
    PRIMARY KEY (tenant_id),
    tenant_id ${uuid} NOT NULL,
    matomo_url ${varchar1024},
    matomo_site_id ${varchar255},
    matomo_auth_token ${varchar1024},
    cookie_banner_enabled ${bool} NOT NULL,
    CONSTRAINT fk__tenant__tracking_settings
        FOREIGN KEY(tenant_id)
        REFERENCES tenant(id)
        ON DELETE CASCADE
);

--changeset alexander.zottnick:add_tenant_column_to_profile dbms:postgresql
ALTER TABLE profile ADD column tenant_id ${uuid};

--changeset alexander.zottnick:add_tenant_column_to_profile dbms:oracle
ALTER TABLE profile ADD tenant_id ${uuid} NULL;

--changeset alexander.zottnick:copy_tenant_from_profile_tenant_to_profile_table
UPDATE profile SET tenant_id = (SELECT tenant_id FROM profile_tenant WHERE profile_tenant.profile_id = profile.id);

--changeset alexander.zottnick:delete_deprecated_profile_tenant_relations
DROP TABLE profile_tenant;

--changeset simon.ward:change_default_app_colors
UPDATE app_settings SET app_color = '#016571' where app_color = '#016567' AND app = 'ADMIN';
UPDATE app_settings SET home_screen_app_color = '#016571' where home_screen_app_color = '#016567' AND app = 'ADMIN';
UPDATE app_settings SET app_color = '#0B836D' where app_color = '#42b392' AND app = 'PEOPLE';
UPDATE app_settings SET home_screen_app_color = '#0B836D' where home_screen_app_color = '#42b392' AND app = 'PEOPLE';
UPDATE app_settings SET app_color = '#BF2993' where app_color = '#CD52B0' AND app = 'NEWS';
UPDATE app_settings SET home_screen_app_color = '#BF2993' where home_screen_app_color = '#CD52B0' AND app = 'NEWS';
UPDATE app_settings SET app_color = '#00819D' where app_color = '#169CC4' AND app = 'WIKI';
UPDATE app_settings SET home_screen_app_color = '#00819D' where home_screen_app_color = '#169CC4' AND app = 'WIKI';

--changeset maxse:add_scim_available_fields_table
CREATE TABLE scim_available_fields (
    name ${varchar1024} NOT NULL,
    tenant_id ${uuid} NOT NULL,
    CONSTRAINT fk__tenant__scim_available_fields
            FOREIGN KEY(tenant_id)
            REFERENCES tenant(id)
            ON DELETE CASCADE,
    CONSTRAINT unique__scim__name__tenant_id UNIQUE (name, tenant_id)
);

--changeset liivanurm:remove_newlines_from_meet_app_url
UPDATE external_app_settings SET url = REPLACE(url, chr(10), '') WHERE url LIKE 'https://meet.just.social/%';

--changeset liivanurm:add_missing_core_apps dbms:postgresql endDelimiter:/
DO
$do$
BEGIN
  IF EXISTS (SELECT count(id) FROM tenant HAVING count(id) = 1) THEN
    -- chat
    INSERT INTO app_settings (tenant_id,app,app_color,id,active,icon_id,name,home_screen_app_color)
    VALUES ((select id from tenant),'CHAT','#5587A2',(select concat('APP,chat_',id) from tenant),false,null,'Chat','#5587A2')
    on CONFLICT (id) DO nothing;
    -- drive
    INSERT INTO app_settings (tenant_id,app,app_color,id,active,icon_id,name,home_screen_app_color)
    VALUES ((select id from tenant),'DRIVE','#FFA000',(select concat('APP,drive_',id) from tenant),false,null,'Drive','#FFA000')
    on CONFLICT (id) DO nothing;
    -- people
    INSERT INTO app_settings (tenant_id,app,app_color,id,active,icon_id,name,home_screen_app_color)
    VALUES ((select id from tenant),'PEOPLE','#0B836D',(select concat('APP,people_',id) from tenant),false,null,'People','#0B836D')
    on CONFLICT (id) DO nothing;
    -- news
    INSERT INTO app_settings (tenant_id,app,app_color,id,active,icon_id,name,home_screen_app_color)
    VALUES ((select id from tenant),'NEWS','#BF2993',(select concat('APP,news_',id) from tenant),false,null,'News','#BF2993')
    on CONFLICT (id) DO nothing;
    -- wiki
    INSERT INTO app_settings (tenant_id,app,app_color,id,active,icon_id,name,home_screen_app_color)
    VALUES ((select id from tenant),'WIKI','#00819D',(select concat('APP,wiki_',id) from tenant),false,null,'Wiki','#00819D')
    on CONFLICT (id) DO nothing;
  END IF;
END
$do$/

--changeset liivanurm:trim_external_app_settings_urls
UPDATE external_app_settings SET url = TRIM(url) WHERE LENGTH(TRIM(url)) <> LENGTH(url);

--changeset baron:add_constraint_fk__tenant__app_permission
ALTER TABLE app_permission ADD CONSTRAINT fk__tenant__app_permission FOREIGN KEY(tenant_id) REFERENCES tenant(id) ON DELETE CASCADE;

--changeset liivanurm:add_home_screen_position_to_all_missing_core_apps dbms:postgresql
--validCheckSum: 9:9e5e7780d58b58c5cab084c66c79ba5d
INSERT INTO app_home_screen_position (app_id, position, tenant_id)
  SELECT a.id, (SELECT max(ahsp.position) FROM app_home_screen_position ahsp WHERE ahsp.tenant_id = a.tenant_id) + ROW_NUMBER () OVER ( ORDER BY a.id ), a.tenant_id
  FROM app_settings a ON CONFLICT (app_id) DO NOTHING;
