#!/usr/bin/env python3

"""
A python script to monitor our just-permission service
memory usage and send warnings if memory gets too low.
When our permission system is running out of memory, quite a few
wonky behaviours start to happen in our software and it's not always
directly apparent that just-permission is the culprit.
"""

import urllib.request as req
import json
import argparse

parser = argparse.ArgumentParser()
parser.add_argument("-w", "--warning", required=False, default=10, type=int,
                    help="Send a warning if free memory percentage falls below this threshold")
parser.add_argument("-c", "--critical", required=False, default=5, type=int,
                    help="Send a critical status if free memory percentage falls below this threshold")

args = parser.parse_args()


def get_memory(type):
    """
    :param str type, can be either 'max' or 'used'
    """
    url = f'http://localhost:7890/permission/internal/monitoring/metrics/jvm.memory.{type}?tag=area:heap'
    try:
        with req.urlopen(url) as res:
            decoded = res.read().decode('utf-8')
            res.close()
            parsed = json.loads(decoded)
            max_in_bytes = parsed['measurements'][0]['value']
            return round(max_in_bytes / 1024 / 1024, 2)

    except Exception as e:
        print(
            f'UNKNOWN - could not get memory data from actuator. Exception {e}')
        exit(3)


try:
    max_memory = get_memory('max')
    used_memory = get_memory('used')

    used_memory_percent = (used_memory / max_memory) * 100
    free_memory_percent = 100 - used_memory_percent
    critical_threshold = args.critical
    warning_treshold = args.warning

    if (free_memory_percent < critical_threshold):
        print(
            f'CRITICAL - less than {critical_threshold}% free memory. {used_memory}/{max_memory} MiB in use.')
        exit(2)

    if (free_memory_percent < warning_treshold):
        print(
            f'WARNING - less than {warning_treshold}% free memory. {used_memory}/{max_memory} MiB in use.')
        exit(1)

    print(
        f'OK - enough free memory available. {used_memory}/{max_memory} MiB in use.')
    exit(0)
except Exception as e:
    print(f'UNKNOWN - Something went wrong here. Exception {e}')
    exit(3)
