# -*- coding: utf-8 -*-
# Description: kafka qeue netdata python.d module
# Author: Serge Müller (schwatvogel)
# SPDX-License-Identifier: GPL-3.0+

from bases.FrameworkServices.UrlService import UrlService

import re
import sys
import string

# default module values (can be overridden per job in `config`)
update_every = 5
priority = 60000
retries = 60
timeout = 20
ORDER = [
    'jvm_threads',
    'jvm_mem',
    'log_size',
    'proc_cpu',
    'proc_file',
    'kafka_network_requestmetrics_requests_total',
    'kafka_log_size',

    'server_brokertopicmetrics_failedfetchrequests_total',
    'server_replicafetchermanager_minfetchrate',
    'network_requestmetrics_throttletimems',
    'server_brokertopicmetrics_bytesin_total',
    'server_brokertopicmetrics_bytesout_total',
    'network_requestmetrics_requests_total',
    'server_replicafetchermanager_maxlag',
    'network_requestmetrics_localtimems',
    'network_requestmetrics_responsesendtimems',
    'cluster_partition_replicascount',
    'server_brokertopicmetrics_messagesin_total',
    'network_requestmetrics_totaltimems',
    'log_logendoffset',
    'server_brokertopicmetrics_totalfetchrequests_total',
    'server_brokertopicmetrics_bytesrejected_total',
    'server_brokertopicmetrics_totalproducerequests_total',
    'network_requestmetrics_remotetimems',
    'network_requestmetrics_requestqueuetimems',
    'log_logstartoffset',
    'server_brokertopicmetrics_failedproducerequests_total',
    'network_requestmetrics_responsequeuetimems',
    'cluster_partition_underreplicated',
    'cluster_partition_insyncreplicascount',
    'cluster_partition_laststableoffsetlag',
    'log_numlogsegments']

CHARTS = {

    'proc_file': {
        'options': [None, 'Process file handles', 'handles', 'proc',
                    'proc.file', 'line'],
        'lines': [
            ['process_open_fds', 'File Descriptors'],
            ['process_max_fds', 'Limit'],
        ]
    },
    'proc_cpu': {
        'options': [None, 'Process Cpu usage', 'seconds', 'proc',
                    'proc.cpu', 'line'],
        'lines': [
            ['process_cpu_seconds_total', 'CPU Seconds', 'incremental'],
        ]
    },
    'jvm_threads': {
        'options': [None, 'Java Threads', 'threads', 'jvm', 'jvm.stats', 'line'],
        'lines': [
            ['jvm_threads_current', 'Current Threads'],
            ['jvm_threads_daemon', 'Deamon Threads'],
            ['jvm_threads_peak', 'Maximum Threads'],
            ['jvm_threads_started_total', 'Threads started/s', 'incremental'],
            ['jvm_threads_deadlocked', 'Deadlocks'],
            ['jvm_threads_deadlocked_monitor', 'Deadlock Monitors'],
        ]
    },
    'jvm_mem': {
        'options': [None, 'Java Memory', 'Bytes', 'jvm', 'jvm.stats', 'line'],
        'lines': [
            ['jvm_memory_bytes_used-area_heap', 'Bytes Used area Heap'],
            ['jvm_memory_bytes_used-area_nonheap', 'Bytes Used area Nonheap'],
            ['jvm_memory_bytes_committed-area_heap',
             'Bytes Committed area Heap'],
            ['jvm_memory_bytes_committed-area_nonheap',
             'Bytes Committed area Nonheap'],
            ['jvm_memory_bytes_max-area_heap', 'Bytes Max area Heap'],
            ['jvm_memory_bytes_max-area_nonheap', 'Bytes Max area Nonheap'],
            ['jvm_memory_pool_bytes_used-pool_Code',
             'Pool Bytes Used pool Code'],
            ['jvm_memory_pool_bytes_used-pool_Metaspace',
             'Pool Bytes Used pool Metaspace'],
            ['jvm_memory_pool_bytes_used-pool_Compressed',
             'Pool Bytes Used pool Compressed'],
            ['jvm_memory_pool_bytes_used-pool_G1', 'Pool Bytes Used pool G1'],
            ['jvm_memory_pool_bytes_used-pool_G1', 'Pool Bytes Used pool G1'],
            ['jvm_memory_pool_bytes_used-pool_G1', 'Pool Bytes Used pool G1'],
            ['jvm_memory_pool_bytes_committed-pool_Code',
             'Pool Bytes Committed pool Code'],
            ['jvm_memory_pool_bytes_committed-pool_Metaspace',
             'Pool Bytes Committed pool Metaspace'],
            ['jvm_memory_pool_bytes_committed-pool_Compressed',
             'Pool Bytes Committed pool Compressed'],
            ['jvm_memory_pool_bytes_committed-pool_G1',
             'Pool Bytes Committed pool G1'],
            ['jvm_memory_pool_bytes_committed-pool_G1',
             'Pool Bytes Committed pool G1'],
            ['jvm_memory_pool_bytes_committed-pool_G1',
             'Pool Bytes Committed pool G1'],
            ['jvm_memory_pool_bytes_max-pool_Code', 'Pool Bytes Max pool Code'],
            ['jvm_memory_pool_bytes_max-pool_Metaspace',
             'Pool Bytes Max pool Metaspace'],
            ['jvm_memory_pool_bytes_max-pool_Compressed',
             'Pool Bytes Max pool Compressed'],
            ['jvm_memory_pool_bytes_max-pool_G1', 'Pool Bytes Max pool G1'],
            ['jvm_memory_pool_bytes_max-pool_G1', 'Pool Bytes Max pool G1'],
            ['jvm_memory_pool_bytes_max-pool_G1', 'Pool Bytes Max pool G1']]
    },
    'kafka_network_requestmetrics_requests_total': {
        'options': [None, 'Kafka Network Requests', 'requests', 'kafka', 'kafka.net', 'line'],
        'lines': []
    },
    'kafka_log_size': {
        'options': [None, 'Kafka Log Size', 'bytes', 'kafka', 'kafka.logsize', 'stacked'],
        'lines': []
    }
}
LINES = {
    'kafka_network_requestmetrics_requests_total': {
        'type': 'incremental',
        'deletestr': 'request_'
    },
    'kafka_log_size': {
        'type': 'absolute',
        'deletestr': 'topic_'
    }
}

PYV = sys.version_info.major
REG = re.compile('^(?P<definition>\w+.*) (?P<value>\S+)')

if PYV >= 3:
    TRMAP = string.maketrans('{=, ', '-___')
else:
    TRMAP = {34: None, 44: 95, 61: 95, 123: 45, 125: None}


class Service(UrlService):
    def __init__(self, configuration=None, name=None):
        UrlService.__init__(self, configuration=configuration, name=name)
        self.order = ORDER
        self.definitions = CHARTS
        self.url = self.configuration.get(
            'url', 'http://localhost:7071/metrics')
        self.timeout = self.configuration.get('timeout', 10)
        self.linetypes = {}

    def check(self):
        self._manager = self._build_manager()
        if not self._manager:
            return False
        data = self._get_data()
        self.debug(data)
        if not data:
            return None
        for k in data.keys():
            if k.startswith('kafka_') and '-' in k:
                definition = k.split('-')[0]
                description = k.split('-')[1]
                if definition in self.definitions.keys():
                    self.definitions[definition]['lines'].append([k, description.replace(
                        LINES[definition]['deletestr'], ''), LINES[definition]['type']])
        return True

    def clean_keys(self, keys):
        ckeys = keys.replace('"', '').replace('.', '_').replace('__', '_')
        ckey = ""
        for els in ckeys.split(','):
            k, v = els.split('=')
            ckey = "{}-{}".format(v.strip('_'), ckey)
        return ckey.rstrip('-')

    def _get_data(self):
        """
        Format data received from http request
        :return: dict
        """
        raw_data = self._get_raw_data()

        data = dict()
        for line in raw_data.splitlines():
            match = REG.match(line)
            if match:
                if PYV >= 3 or type(match.group('definition')) == unicode:
                    definition = match.group(
                        'definition').translate(TRMAP).rstrip('_')
                else:
                    definition = match.group('definition').translate(
                        TRMAP, '"}').rstrip('_')
                data[definition] = float(match.group('value'))
        return data or None
