/*
 * Decompiled with CFR 0.152.
 */
package com.freiheit.simplegroup;

import com.freiheit.simplegroup.ChannelListener;
import com.google.common.base.MoreObjects;
import com.google.common.base.Strings;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.io.UncheckedIOException;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.InetAddress;
import java.net.SocketException;
import java.net.SocketTimeoutException;
import java.net.UnknownHostException;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Stream;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public final class Channel {
    private static final Log LOG = LogFactory.getLog(Channel.class);
    private static final int DEFAULT_PING_TIME = 42000;
    private static final InetAddress ANY_ADDRESS = Channel.addressByName("0.0.0.0");
    private static final InetAddress LOOPBACK_ADDRESS = Channel.addressByName("127.0.0.1");
    private static final int PEER_DEAD_FACTOR = 100;
    private final long _uid = System.currentTimeMillis();
    private final int _port;
    private final Map<Peer, PeerKnownStatus> _peers = new ConcurrentHashMap<Peer, PeerKnownStatus>();
    private final boolean _ignoreOwn;
    private final DatagramSocket _socket;
    private final String _channelName;
    private int _pingTime = 42000;
    private boolean _closed = false;
    private final List<ChannelListener> _listenerList = new LinkedList<ChannelListener>();

    private static InetAddress addressByName(String name) {
        try {
            return InetAddress.getByName(name);
        }
        catch (UnknownHostException e) {
            throw new UncheckedIOException(e);
        }
    }

    public Channel(int port) throws SocketException {
        this(port, ANY_ADDRESS);
    }

    public Channel(int port, InetAddress addr) throws SocketException {
        this(port, addr, "", true);
    }

    public Channel(int port, String channelName) throws SocketException {
        this(port, ANY_ADDRESS, channelName, true);
    }

    public Channel(int port, InetAddress addr, String channelName) throws SocketException {
        this(port, addr, channelName, true);
    }

    public Channel(int port, InetAddress addr, String channelName, boolean ignoreOwnMessages) throws SocketException {
        this._port = port;
        this._channelName = Strings.nullToEmpty((String)channelName);
        this._ignoreOwn = ignoreOwnMessages;
        this._socket = this.determineFreeSocket((InetAddress)MoreObjects.firstNonNull((Object)addr, (Object)ANY_ADDRESS), port);
        Thread t = new Thread(this::receiveLoop);
        t.setDaemon(true);
        t.start();
        if (this._socket.getLocalPort() != this._port) {
            LOG.info((Object)("using different port: " + this._port));
            this.putPeer(LOOPBACK_ADDRESS, this._port, true, false);
            this.sendPing();
        }
    }

    private DatagramSocket determineFreeSocket(InetAddress addr, int initPort) throws SocketException {
        int i = 0;
        while (true) {
            try {
                return new DatagramSocket(initPort + i, addr);
            }
            catch (SocketException e) {
                if (i >= 10) {
                    throw e;
                }
                ++i;
                continue;
            }
            break;
        }
    }

    void sendPing() {
        HashSet<Peer> knownPeers = new HashSet<Peer>();
        Iterator<Map.Entry<Peer, PeerKnownStatus>> it = this._peers.entrySet().iterator();
        while (it.hasNext()) {
            Map.Entry<Peer, PeerKnownStatus> entry = it.next();
            PeerKnownStatus status = entry.getValue();
            if (status.getEntryAge() > (long)(100 * this._pingTime) && status.mayRemove()) {
                it.remove();
                continue;
            }
            if (!status.isKnownPersonally()) continue;
            knownPeers.add(entry.getKey());
        }
        LOG.debug((Object)("-- " + this._channelName + " ping -- " + knownPeers));
        try {
            this.sendMessage(new Message(this._channelName, knownPeers, this._uid, true));
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    private void putPeer(InetAddress addr, int port, boolean isKnown, boolean mayRemove) {
        if (!addr.equals(this._socket.getLocalAddress()) || port != this._socket.getLocalPort()) {
            PeerKnownStatus status;
            Peer peer = new Peer(addr, port);
            if (!(isKnown || (status = this._peers.get(peer)) == null || !status.isKnownPersonally() && status.mayRemove())) {
                return;
            }
            this._peers.put(peer, new PeerKnownStatus(isKnown, mayRemove));
        }
    }

    public void setPingTime(int time) {
        this._pingTime = time;
    }

    public void addPeer(String hostname) {
        this.putPeer(Channel.addressByName(hostname), this._port, false, false);
    }

    public void addPeer(InetAddress addr) {
        this.putPeer(addr, this._port, false, false);
    }

    public void addPeers(InetAddress ... addrs) {
        Stream.of(addrs).forEach(this::addPeer);
    }

    public void addPeers(String ... addrs) {
        Stream.of(addrs).forEach(this::addPeer);
    }

    public void setChannelListener(ChannelListener listener) {
        this.addChannelListener(listener);
    }

    public void addChannelListener(ChannelListener listener) {
        this._listenerList.add(listener);
    }

    public void close() {
        this._closed = true;
    }

    private synchronized void sendMessage(Message msg) throws IOException {
        if (this._closed) {
            throw new IOException("Channel closed");
        }
        ByteArrayOutputStream stream = new ByteArrayOutputStream();
        ObjectOutputStream oout = new ObjectOutputStream(stream);
        oout.writeObject(msg);
        oout.close();
        byte[] buf = stream.toByteArray();
        DatagramPacket packet = new DatagramPacket(buf, buf.length);
        packet.setPort(this._port);
        for (Peer peer : this._peers.keySet()) {
            packet.setAddress(peer.getAddress());
            packet.setPort(peer.getPort());
            this._socket.send(packet);
        }
    }

    public void publish(Serializable msg) throws IOException {
        this.sendMessage(new Message(this._channelName, msg, this._uid));
    }

    public void receiveLoop() {
        long lastPing = -1L;
        byte[] buf = new byte[40000];
        while (!this._closed) {
            try {
                if (System.currentTimeMillis() - lastPing >= (long)this._pingTime) {
                    this.sendPing();
                    lastPing = System.currentTimeMillis();
                }
                DatagramPacket packet = new DatagramPacket(buf, buf.length);
                try {
                    this._socket.setSoTimeout(this._pingTime);
                    this._socket.receive(packet);
                }
                catch (SocketTimeoutException e) {
                    continue;
                }
                ByteArrayInputStream stream = new ByteArrayInputStream(packet.getData(), 0, packet.getLength());
                ObjectInputStream oin = new ObjectInputStream(stream);
                Message msg = (Message)oin.readObject();
                oin.close();
                if (this._ignoreOwn && msg.getSource() == this._uid) continue;
                if (msg.getSource() != this._uid) {
                    this.putPeer(packet.getAddress(), packet.getPort(), true, true);
                }
                if (msg.isPing()) {
                    Set otherKnown = (Set)((Object)msg.getPayload());
                    if (otherKnown == null) continue;
                    for (Peer peer : otherKnown) {
                        this.putPeer(peer.getAddress(), peer.getPort(), false, true);
                    }
                    continue;
                }
                if (!msg.getChannelName().equals(this._channelName) || this._listenerList == null) continue;
                LOG.debug((Object)"deliver message to listeners");
                for (ChannelListener listener : this._listenerList) {
                    try {
                        listener.messageReceived(msg.getPayload());
                    }
                    catch (Exception e) {
                        LOG.warn((Object)e);
                    }
                }
            }
            catch (IOException | ClassNotFoundException | RuntimeException e) {
                LOG.error((Object)e.getMessage(), (Throwable)e);
            }
        }
        LOG.info((Object)"closing channel ..");
    }

    private static final class PeerKnownStatus {
        private final boolean _msgReceived;
        private final boolean _mayRemove;
        private final long _created;

        public PeerKnownStatus(boolean received, boolean mayRemove) {
            this._msgReceived = received;
            this._mayRemove = mayRemove;
            this._created = System.currentTimeMillis();
        }

        public boolean mayRemove() {
            return this._mayRemove;
        }

        public boolean isKnownPersonally() {
            return this._msgReceived;
        }

        public long getEntryAge() {
            return System.currentTimeMillis() - this._created;
        }
    }

    private static final class Peer
    implements Serializable {
        private static final long serialVersionUID = 200408120710L;
        public final InetAddress _addr;
        public final int _port;

        public Peer(InetAddress addr, int port) {
            this._addr = addr;
            this._port = port;
        }

        public InetAddress getAddress() {
            return this._addr;
        }

        public int getPort() {
            return this._port;
        }

        public int hashCode() {
            return this._port ^ this._addr.hashCode();
        }

        public boolean equals(Object other) {
            if (!(other instanceof Peer)) {
                return false;
            }
            Peer op = (Peer)other;
            return op._addr.equals(this._addr) && op._port == this._port;
        }

        public String toString() {
            return this._addr + ":" + this._port;
        }
    }

    private static final class Message
    implements Serializable {
        private static final long serialVersionUID = 200408112044L;
        private final String _channelName;
        private final long _source;
        private final boolean _ping;
        private final Serializable _payload;

        private Message(String channelName, Serializable payload, long source, boolean ping) {
            this._channelName = channelName;
            this._payload = payload;
            this._source = source;
            this._ping = ping;
        }

        private Message(String channelName, Serializable payload, long source) {
            this(channelName, payload, source, false);
        }

        boolean isPing() {
            return this._ping;
        }

        long getSource() {
            return this._source;
        }

        Serializable getPayload() {
            return this._payload;
        }

        String getChannelName() {
            return this._channelName;
        }
    }
}

