/*
 * Decompiled with CFR 0.152.
 */
package de.justsoftware.gateway.security;

import de.justsoftware.gateway.ids.ExternalId;
import de.justsoftware.gateway.ids.ProfileId;
import de.justsoftware.gateway.ids.TenantId;
import de.justsoftware.gateway.security.EmailToProfileMapper;
import de.justsoftware.gateway.security.ErrorCode;
import de.justsoftware.gateway.security.InMemoryExternalIdMapper;
import de.justsoftware.gateway.security.JustAuthToken;
import de.justsoftware.gateway.security.JustAuthenticationException;
import de.justsoftware.gateway.security.NoJustUserForLoginException;
import de.justsoftware.gateway.security.UnknownClientRegistrationException;
import de.justsoftware.gateway.security.UserLoginService;
import de.justsoftware.gateway.security.UserNameAttributeNotFoundException;
import de.justsoftware.gateway.security.oauth2.JustInMemoryClientRegistrationRepository;
import de.justsoftware.gateway.security.oauth2.JustLoginHelper;
import java.util.ArrayList;
import java.util.Map;
import javax.annotation.CheckForNull;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.security.authentication.AbstractAuthenticationToken;
import org.springframework.security.authentication.ReactiveAuthenticationManager;
import org.springframework.security.core.Authentication;
import org.springframework.security.oauth2.client.authentication.OAuth2AuthenticationToken;
import org.springframework.security.oauth2.client.registration.ClientRegistration;
import org.springframework.security.oauth2.server.resource.authentication.BearerTokenAuthentication;
import org.springframework.security.oauth2.server.resource.authentication.JwtAuthenticationToken;
import org.springframework.stereotype.Service;
import reactor.core.publisher.Mono;

@ParametersAreNonnullByDefault
@Service
public class JustAuthenticationManager
implements ReactiveAuthenticationManager {
    private static final Logger LOG = LoggerFactory.getLogger(JustAuthenticationManager.class);
    private static final String AUDIENCE_CLAIM = "aud";
    private static final String ISSUER_CLAIM = "iss";
    private static final String TENANT_ID_KEY = "tenantId";
    private final JustInMemoryClientRegistrationRepository _clientRegistrationsRepo;
    private final InMemoryExternalIdMapper _inMemoryExternalIdMapper;
    private final EmailToProfileMapper _emailToProfileMapper;
    private final UserLoginService _userLoginService;

    public JustAuthenticationManager(JustInMemoryClientRegistrationRepository clientRegistrationsRepo, InMemoryExternalIdMapper inMemoryExternalIdMapper, EmailToProfileMapper emailToProfileMapper, UserLoginService userLoginService) {
        this._clientRegistrationsRepo = clientRegistrationsRepo;
        this._inMemoryExternalIdMapper = inMemoryExternalIdMapper;
        this._emailToProfileMapper = emailToProfileMapper;
        this._userLoginService = userLoginService;
    }

    public Mono<Authentication> authenticate(Authentication authentication) {
        if (authentication instanceof BearerTokenAuthentication) {
            return this.authenticateFromBearerToken((BearerTokenAuthentication)authentication);
        }
        if (authentication instanceof JwtAuthenticationToken) {
            return this.authenticateFromJwt(authentication);
        }
        if (authentication instanceof OAuth2AuthenticationToken) {
            return this.authenticateFromOAuthLoginToken((OAuth2AuthenticationToken)authentication);
        }
        return Mono.empty();
    }

    @Nonnull
    private Mono<Authentication> authenticateFromBearerToken(BearerTokenAuthentication justOpaqueToken) {
        Map tokenAttributes = justOpaqueToken.getTokenAttributes();
        String profileIdString = (String)tokenAttributes.get("profileId");
        if (profileIdString == null) {
            return Mono.just((Object)justOpaqueToken);
        }
        ProfileId profileId = ProfileId.parse((String)profileIdString);
        this.checkProfile(profileId);
        TenantId tenantId = TenantId.parse((String)((String)tokenAttributes.get(TENANT_ID_KEY)));
        JustAuthToken newToken = new JustAuthToken(profileId, tenantId, (AbstractAuthenticationToken)justOpaqueToken, null);
        return Mono.just((Object)newToken);
    }

    public void checkProfile(ProfileId profileId) {
        boolean canLogin = this._emailToProfileMapper.mayLogin(profileId);
        if (!canLogin) {
            LOG.warn("inactive, blocked or deleted user tried to access just {}", (Object)profileId);
            throw new JustAuthenticationException("user can not login", ErrorCode.USER_NOT_FOUND);
        }
    }

    @Nonnull
    public Mono<Authentication> authenticateFromOAuthLoginToken(OAuth2AuthenticationToken token) {
        ProfileId profileId;
        TenantId tenantId = this.getTenantId(token);
        if (JustLoginHelper.isJustIdp((String)token.getAuthorizedClientRegistrationId())) {
            profileId = ProfileId.parse((String)token.getName());
        } else {
            if (tenantId == null) {
                throw new IllegalArgumentException("TenantId may only be null when 'just' is the identity provider");
            }
            profileId = this.getProfileId(tenantId, token.getName());
        }
        if (profileId == null) {
            return Mono.error((Throwable)new NoJustUserForLoginException(tenantId, token.getName()));
        }
        return this._userLoginService.saveLogin(profileId, tenantId, token).map(login -> new JustAuthToken(profileId, tenantId, (AbstractAuthenticationToken)token, login.getId())).switchIfEmpty(Mono.just((Object)new JustAuthToken(profileId, tenantId, (AbstractAuthenticationToken)token, null))).cast(Authentication.class);
    }

    @Nonnull
    private Mono<Authentication> authenticateFromJwt(Authentication authentication) {
        JwtAuthenticationToken jwt = (JwtAuthenticationToken)authentication;
        Map tokenAttributes = jwt.getTokenAttributes();
        Mono registration = this.getClientRegistrationFromToken(tokenAttributes);
        return registration.map(clientRegistration -> {
            String tenantIdString = (String)clientRegistration.getProviderDetails().getConfigurationMetadata().get(TENANT_ID_KEY);
            if (tenantIdString != null) {
                TenantId tenantId = TenantId.parse((String)tenantIdString);
                String userNameAttributeName = clientRegistration.getProviderDetails().getUserInfoEndpoint().getUserNameAttributeName();
                String externalId = (String)tokenAttributes.get(userNameAttributeName);
                if (externalId == null) {
                    throw new UserNameAttributeNotFoundException(String.format("userNameAttributeName %s not found in token", userNameAttributeName));
                }
                ProfileId profileId = this.getProfileId(tenantId, externalId);
                if (profileId != null) {
                    return new JustAuthToken(profileId, tenantId, (AbstractAuthenticationToken)jwt, null);
                }
                throw new NoJustUserForLoginException(tenantId, externalId);
            }
            return null;
        }).cast(Authentication.class).switchIfEmpty(Mono.error((Throwable)new UnknownClientRegistrationException(tokenAttributes.toString())));
    }

    @Nonnull
    private Mono<ClientRegistration> getClientRegistrationFromToken(Map<String, Object> tokenAttributes) {
        String issuer = (String)tokenAttributes.get(ISSUER_CLAIM);
        return this.getClientIdFromTokenAttribute(tokenAttributes.get(AUDIENCE_CLAIM)).flatMap(clientId -> this._clientRegistrationsRepo.findByClientId(clientId, issuer));
    }

    @Nonnull
    private Mono<String> getClientIdFromTokenAttribute(@Nullable Object audienceAttribute) {
        if (audienceAttribute instanceof ArrayList) {
            ArrayList list = (ArrayList)audienceAttribute;
            if (list.size() > 0) {
                return Mono.just((Object)((String)list.get(0)));
            }
        } else if (audienceAttribute instanceof String) {
            return Mono.just((Object)((String)audienceAttribute));
        }
        return Mono.empty();
    }

    @CheckForNull
    public TenantId getTenantId(OAuth2AuthenticationToken principal) {
        if (JustLoginHelper.isJustIdp((String)principal.getAuthorizedClientRegistrationId())) {
            String tenantIdString = (String)principal.getPrincipal().getAttribute(TENANT_ID_KEY);
            return tenantIdString != null ? TenantId.parse((String)tenantIdString) : null;
        }
        ClientRegistration clientRegistration = (ClientRegistration)this._clientRegistrationsRepo.findByRegistrationId(principal.getAuthorizedClientRegistrationId()).block();
        if (clientRegistration == null) {
            throw new UnknownClientRegistrationException(String.format("unknown registrationId %s", principal.getAuthorizedClientRegistrationId()));
        }
        String tenantId = (String)clientRegistration.getProviderDetails().getConfigurationMetadata().get(TENANT_ID_KEY);
        return TenantId.parse((String)tenantId);
    }

    @CheckForNull
    private ProfileId getProfileId(TenantId tenantId, String externalId) {
        ProfileId byEmail = this._emailToProfileMapper.findByEmail(externalId);
        if (byEmail != null) {
            return byEmail;
        }
        ProfileId profileIdByExternalId = this._inMemoryExternalIdMapper.lookupInternalId(tenantId, ExternalId.parse((String)externalId));
        if (profileIdByExternalId == null) {
            LOG.warn(String.format("no internalId found for externalId: %s and tenantId: %s", externalId, tenantId));
        }
        return profileIdByExternalId;
    }
}

