--liquibase formatted sql

--changeset christian.ewers:add_timestamps dbms:postgresql runAlways:true splitStatements:false
CREATE OR REPLACE FUNCTION update_mtime()
RETURNS TRIGGER AS $$
BEGIN
   NEW.mtime = clock_timestamp();
   RETURN NEW;
END;
$$ language 'plpgsql';

CREATE OR REPLACE FUNCTION ctime_mtime_columns(tname TEXT)
RETURNS VOID AS $$
BEGIN
    EXECUTE 'ALTER TABLE ' || tname || ' ADD COLUMN ctime TIMESTAMP WITHOUT TIME ZONE NOT NULL DEFAULT clock_timestamp();';
    EXECUTE 'ALTER TABLE ' || tname || ' ADD COLUMN mtime TIMESTAMP WITHOUT TIME ZONE NOT NULL DEFAULT clock_timestamp();';
    EXECUTE 'CREATE TRIGGER ' || tname || '_mtrg BEFORE UPDATE ON ' || tname || ' FOR EACH ROW EXECUTE PROCEDURE update_mtime();';
END;
$$ language 'plpgsql';

--changeset christian.ewers:add_now_function dbms:oracle runAlways:true splitStatements:false
CREATE OR REPLACE FUNCTION now
    RETURN DATE
AS
BEGIN
    RETURN SYSDATE;
END now;

--changeset christian.ewers:add_clock_timestamp_function dbms:oracle runAlways:true splitStatements:false
CREATE OR REPLACE FUNCTION clock_timestamp
    RETURN TIMESTAMP WITH TIME ZONE
AS
BEGIN
    RETURN SYSTIMESTAMP;
END clock_timestamp;

--changeset christian.ewers:add_timestamps dbms:oracle runAlways:true splitStatements:false
CREATE OR REPLACE PROCEDURE ctime_mtime_columns(tname VARCHAR2)
IS
BEGIN
    EXECUTE IMMEDIATE 'ALTER TABLE ' || tname || ' ADD (ctime TIMESTAMP DEFAULT SYSDATE NOT NULL,mtime TIMESTAMP DEFAULT SYSTIMESTAMP NOT NULL )';
    EXECUTE IMMEDIATE 'CREATE TRIGGER ' || tname || '_mtrg BEFORE UPDATE ON ' || tname || ' FOR EACH ROW BEGIN :new.mtime := SYSTIMESTAMP; END;';
END ctime_mtime_columns;

--changeset baron:create_initial_tables
CREATE TABLE profile (
    id ${varchar64} NOT NULL PRIMARY KEY,
    first_name ${text} NOT NULL,
    last_name ${text} NOT NULL,
    email ${text} NOT NULL,
    image_url ${text} NOT NULL,
    status ${text} NOT NULL
);
${execute} ctime_mtime_columns('profile');

CREATE TABLE channel (
    id ${varchar64} NOT NULL PRIMARY KEY,
    title ${text} NOT NULL,
    tenant_id ${varchar64} NOT NULL,
    deleted ${bool} NOT NULL,
    deleted_at ${timestamp} DEFAULT NULL,
    deleted_by ${varchar64} DEFAULT NULL,
    create_date ${timestamp} NOT NULL,
    modify_date ${timestamp} NOT NULL
);
${execute} ctime_mtime_columns('channel');

CREATE TABLE personal_subscription (
    id ${varchar64} NOT NULL PRIMARY KEY,
    channel_id ${varchar64} NOT NULL,
    profile_id ${varchar64} NOT NULL,
    notifications_enabled ${bool} NOT NULL,
    user_has_unsubscribed ${bool} NOT NULL,
    CONSTRAINT unique__channel__profile
        UNIQUE(channel_id, profile_id),
    CONSTRAINT fk__channel__personal_subscription
        FOREIGN KEY(channel_id)
        REFERENCES channel(id)
        ON DELETE CASCADE,
    CONSTRAINT fk__profile__subscription
        FOREIGN KEY(profile_id)
        REFERENCES profile(id)
        ON DELETE CASCADE
);
${execute} ctime_mtime_columns('personal_subscription');

CREATE TABLE managed_subscription (
    id ${varchar64} NOT NULL PRIMARY KEY,
    channel_id ${varchar64} NOT NULL,
    type ${varchar64} NOT NULL,
    user_id ${varchar64} NOT NULL,
    CONSTRAINT unique_managed_subscription__channel__user
        UNIQUE (channel_id, user_id),
    CONSTRAINT fk__channel__managed_subscription
        FOREIGN KEY(channel_id)
        REFERENCES channel(id)
        ON DELETE CASCADE
);
${execute} ctime_mtime_columns('managed_subscription');

CREATE TABLE user_group (
    id ${varchar64} NOT NULL PRIMARY KEY,
    name ${text} NOT NULL,
    tenant_id ${varchar64} NOT NULL
);
${execute} ctime_mtime_columns('user_group');

CREATE TABLE post (
    id ${varchar64} NOT NULL PRIMARY KEY,
    author_id ${varchar64} NOT NULL,
    channel_id ${varchar64} NOT NULL,
    create_date ${timestamp} NOT NULL,
    modify_date ${timestamp} NOT NULL,
    title ${text} NOT NULL,
    text ${text} NOT NULL,
    json_content ${text} NOT NULL,
    cover_element_id ${varchar64},
    pinned ${bool} DEFAULT ${false} NOT NULL,
    pinned_at ${timestamp},
    publish_date ${timestamp} NOT NULL,
    published ${bool} NOT NULL,
    deleted ${bool} NOT NULL,
    deleted_at ${timestamp} DEFAULT NULL,
    deleted_by ${varchar64} DEFAULT NULL,
    CONSTRAINT fk__author__post
        FOREIGN KEY(author_id)
        REFERENCES profile(id),
    CONSTRAINT fk__channel__post
        FOREIGN KEY(channel_id)
        REFERENCES channel(id)
        ON DELETE CASCADE
);
${execute} ctime_mtime_columns('post');

CREATE TABLE attachment (
    id ${varchar64} NOT NULL PRIMARY KEY,
    post_id ${varchar64} NOT NULL,
    mime_type ${text} NOT NULL,
    name ${text} NOT NULL,
    file_size ${int64} DEFAULT 0 NOT NULL,
    CONSTRAINT fk__post__attachment
        FOREIGN KEY(post_id)
        REFERENCES post(id)
        ON DELETE CASCADE
);
${execute} ctime_mtime_columns('attachment');

CREATE TABLE poll (
    id ${varchar64} NOT NULL PRIMARY KEY,
    question ${text} NOT NULL,
    allow_multiple_answers ${bool} NOT NULL,
    post_id ${varchar64} NOT NULL,
    CONSTRAINT fk__post__poll
        FOREIGN KEY(post_id)
        REFERENCES post(id)
        ON DELETE CASCADE
);
${execute} ctime_mtime_columns('poll');

CREATE TABLE poll_answer (
    id ${varchar64} NOT NULL PRIMARY KEY,
    text ${text} NOT NULL,
    position ${int32} NOT NULL,
    poll_id ${varchar64} NOT NULL,
    CONSTRAINT fk__poll__poll_answer
        FOREIGN KEY(poll_id)
        REFERENCES poll(id)
        ON DELETE CASCADE
);
${execute} ctime_mtime_columns('poll_answer');

CREATE TABLE poll_vote (
    id ${varchar64} NOT NULL PRIMARY KEY,
    poll_id ${varchar64} NOT NULL,
    answer_id ${varchar64} NOT NULL,
    profile_id ${varchar64} NOT NULL,
    CONSTRAINT unique__answer__profile
        UNIQUE (answer_id, profile_id),
    CONSTRAINT fk__poll_answer__poll_vote
        FOREIGN KEY(answer_id)
        REFERENCES poll_answer(id)
        ON DELETE CASCADE
);
${execute} ctime_mtime_columns('poll_vote');

CREATE TABLE post_comment (
    id ${varchar64} NOT NULL PRIMARY KEY,
    post_id ${varchar64} NOT NULL,
    parent_id ${varchar64} NOT NULL,
    reference_id ${varchar64} NOT NULL,
    author_id ${varchar64} NOT NULL,
    created_at ${timestamp} NOT NULL,
    updated_at ${timestamp} NOT NULL,
    last_updated_by ${varchar64},
    json_content ${text} NOT NULL,
    text ${text} NOT NULL,
    deleted ${bool} DEFAULT ${false} NOT NULL,
    deleted_at ${timestamp} DEFAULT NULL,
    deleted_by ${varchar64} DEFAULT NULL,
    CONSTRAINT fk__author__post_comment
        FOREIGN KEY(author_id)
        REFERENCES profile(id),
    CONSTRAINT fk__post__post_comment
        FOREIGN KEY(post_id)
        REFERENCES post(id)
        ON DELETE CASCADE
);
${execute} ctime_mtime_columns('post_comment');

CREATE TABLE reaction (
    id ${varchar64} NOT NULL PRIMARY KEY,
    emoji ${varchar1024} NOT NULL,
    item_id ${varchar64} NOT NULL,
    profile_id ${varchar64} NOT NULL
);
${execute} ctime_mtime_columns('reaction');

CREATE TABLE permission (
    id ${varchar64} NOT NULL PRIMARY KEY,
    item_id ${varchar64} NOT NULL,
    grantee_id ${varchar64} NOT NULL,
    role ${varchar1024} NOT NULL,
    CONSTRAINT unique__item_id__grantee_id__role
        UNIQUE (item_id, grantee_id, role)
);
${execute} ctime_mtime_columns('permission');

CREATE TABLE tenant (
    id ${varchar64} NOT NULL PRIMARY KEY,
    deepl_key ${varchar64}
);
${execute} ctime_mtime_columns('tenant');

--changeset baron:create_table__user_group_member
CREATE TABLE user_group_member (
    id ${varchar1024} NOT NULL PRIMARY KEY,
    user_group_id ${varchar64} NOT NULL,
    profile_id ${varchar64} NOT NULL,
    CONSTRAINT unique__user_group_id__profile_id
        UNIQUE (user_group_id, profile_id)
);
${execute} ctime_mtime_columns('user_group_member');

--changeset baron:create_table_news_cover
CREATE TABLE news_cover (
    tenant_id ${varchar64} NOT NULL PRIMARY KEY,
    open_on_start ${bool} NOT NULL,
    cycle_delay ${int32} NOT NULL,
    CONSTRAINT fk__tenant__news_cover
        FOREIGN KEY(tenant_id)
        REFERENCES tenant(id)
        ON DELETE CASCADE
);
${execute} ctime_mtime_columns('news_cover');

CREATE TABLE news_cover_section (
    id ${varchar64} NOT NULL PRIMARY KEY,
    title ${text} NOT NULL,
    pinned_only ${bool} NOT NULL,
    position ${int32} NOT NULL,
    tenant_id ${varchar64} NOT NULL,
    CONSTRAINT fk__news_cover__news_cover_section
        FOREIGN KEY(tenant_id)
        REFERENCES news_cover(tenant_id)
        ON DELETE CASCADE
);
${execute} ctime_mtime_columns('news_cover_section');

CREATE TABLE news_cover_section_channel (
    section_id ${varchar64} NOT NULL,
    channel_id ${varchar64} NOT NULL,
    CONSTRAINT fk__news_cover_section__news_cover_section_channel
        FOREIGN KEY(section_id)
        REFERENCES news_cover_section(id)
        ON DELETE CASCADE,
    CONSTRAINT unique__news_cover_section__channel
        UNIQUE (section_id, channel_id)
);
${execute} ctime_mtime_columns('news_cover_section_channel');

--changeset baron:add_post_read_mark
CREATE TABLE post_read_mark(
    id ${varchar64} NOT NULL PRIMARY KEY,
    post_id ${varchar64} NOT NULL,
    profile_id ${varchar64} NOT NULL,
    post_published_date ${timestamp} NOT NULL,
    CONSTRAINT fk__post__post_read_mark
        FOREIGN KEY(post_id)
        REFERENCES post(id)
        ON DELETE CASCADE,
    CONSTRAINT fk__profile__post_read_mark
        FOREIGN KEY(profile_id)
        REFERENCES profile(id)
        ON DELETE CASCADE,
    CONSTRAINT unique__post_id__profile_id
        UNIQUE (post_id, profile_id)
);
${execute} ctime_mtime_columns('post_read_mark');

--changeset baron:permission__alter__column__item_id
ALTER TABLE permission RENAME COLUMN item_id to channel_id;
ALTER TABLE permission ADD CONSTRAINT fk__channel__permission
    FOREIGN KEY(channel_id)
    REFERENCES channel(id)
    ON DELETE CASCADE;

--changeset baron:permission__rename_table
ALTER TABLE permission DROP CONSTRAINT unique__item_id__grantee_id__role;
ALTER TABLE permission ADD CONSTRAINT unique__channel_id__grantee_id
    UNIQUE (channel_id, grantee_id);
ALTER TABLE permission RENAME to channel_permission;
