--liquibase formatted sql
--preconditions dbms:postgresql

--changeset tsubklewe:postprocessing_check_db_version runAlways:true
select check_db_version(90600);

--changeset tsubklewe:postprocessing_update_shadow_triggers runAlways:true endDelimiter:@@
BEGIN;

CREATE OR REPLACE FUNCTION simple_join(s1 TEXT, sep TEXT, s2 TEXT) RETURNS TEXT AS $u$
BEGIN
    IF (length(s1) > 0 AND length(s2) > 0) THEN
        RETURN s1 || sep || s2;
    ELSE
        RETURN s1 || s2;
    END IF;
END
$u$ language plpgsql;@@

CREATE OR REPLACE FUNCTION compare_tables(table1 TEXT, table2 TEXT, filter_col TEXT = '') RETURNS TEXT AS $u$
DECLARE
    obsolete_columns TEXT := '';
    obs_col RECORD;
BEGIN
    FOR obs_col IN (
        select table1 || '.' || a.attname as obs_col_name
        from pg_catalog.pg_attribute a
                 join pg_catalog.pg_class c on c.oid = a.attrelid
                 join pg_catalog.pg_namespace n on n.oid = c.relnamespace
        where a.attnum > 0
          and not a.attisdropped
          and c.relname = table1
          and n.nspname = 'public'
            except
        select table1 || '.' || a.attname as obs_col_name
        from pg_catalog.pg_attribute a
                 join pg_catalog.pg_class c on c.oid = a.attrelid
                 join pg_catalog.pg_namespace n on n.oid = c.relnamespace
        where a.attnum > 0
          and not a.attisdropped
          and c.relname = table2
          and n.nspname = 'public'
    )
        LOOP
            IF (length(filter_col) = 0 OR obs_col.obs_col_name <> (table1 || '.' || filter_col)) THEN
                obsolete_columns := simple_join(obsolete_columns, ', ', obs_col.obs_col_name);
            END IF;
        END LOOP;

    RETURN obsolete_columns;
END
$u$ language plpgsql;@@

CREATE OR REPLACE FUNCTION check_for_obsolete_columns() returns void AS $u$
DECLARE
    obsolete_columns TEXT := '';
    sh_rec RECORD;
BEGIN
    FOR sh_rec IN
        (SELECT table_name, shadow_table_name FROM shadow_table)
        LOOP
            obsolete_columns := simple_join(obsolete_columns, ', ', compare_tables(sh_rec.table_name, sh_rec.shadow_table_name));
            obsolete_columns := simple_join(obsolete_columns, ', ', compare_tables(sh_rec.shadow_table_name, sh_rec.table_name, 'delete_action_id'));
        END LOOP;

    IF length(obsolete_columns) > 0 THEN
        RAISE EXCEPTION 'Unexpected columns found: %', obsolete_columns;
    END IF;
END
$u$ language plpgsql;@@

CREATE OR REPLACE FUNCTION update_shadow_triggers() returns void AS $u$
DECLARE
    sh_rec RECORD;
    col_rec RECORD;
    tr_name TEXT;
    tr_fn_name TEXT;
    stmt TEXT;
    insert_cols TEXT;
    insert_vals TEXT;
BEGIN
    FOR sh_rec IN
        (SELECT table_name, shadow_table_name, delete_action_mandatory FROM shadow_table)
        LOOP
            tr_name := sh_rec.table_name || '_ad';
            tr_fn_name := sh_rec.table_name || '_ad_fn()';
            stmt := 'CREATE OR REPLACE FUNCTION ' || tr_fn_name || E' RETURNS trigger AS $$\n';
            stmt := stmt || E'DECLARE del_ac_id BIGINT;\n';
            stmt := stmt || E'BEGIN\n';
            stmt := stmt || E'  SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = ''t'';\n';

            IF sh_rec.delete_action_mandatory = 't' THEN
                stmt := stmt || E'  IF del_ac_id IS NULL THEN\n';
                stmt := stmt ||  '    RAISE EXCEPTION ''No delete action for current transaction found while trying to delete from ' || sh_rec.table_name || E''';\n';
                stmt := stmt || E'  END IF;\n';
            END IF;

            stmt := stmt || E'  IF del_ac_id IS NOT NULL THEN\n';
            stmt := stmt ||  '    INSERT INTO ' || sh_rec.shadow_table_name || '(';

            insert_cols := '';
            insert_vals := '';
            FOR col_rec IN (
                select a.attname as column_name
                from pg_catalog.pg_attribute a
                         join pg_catalog.pg_class c on c.oid = a.attrelid
                         join pg_catalog.pg_namespace n on n.oid = c.relnamespace
                where a.attnum > 0
                  and not a.attisdropped
                  and c.relname = sh_rec.table_name
                  and n.nspname = 'public'
                order by a.attnum
            )
                LOOP
                    insert_cols := insert_cols || '"' || col_rec.column_name || '", ';
                    insert_vals := insert_vals || 'OLD."' || col_rec.column_name || '", ';
                END LOOP;
            stmt := stmt || insert_cols || E'delete_action_id)\n';
            stmt := stmt || '    values (' || insert_vals || E'del_ac_id);\n';
            stmt := stmt || E'  END IF;\n';
            stmt := stmt || E'  RETURN NEW;\n';
            stmt := stmt || E'END\n';
            stmt := stmt || '$$ language plpgsql;';

            EXECUTE stmt;
            EXECUTE 'DROP TRIGGER IF EXISTS ' || tr_name || ' ON ' || sh_rec.table_name;
            EXECUTE 'CREATE TRIGGER ' || tr_name || ' AFTER DELETE ON ' || sh_rec.table_name || ' FOR EACH ROW EXECUTE procedure ' || tr_fn_name || ';';
        END LOOP;
END
$u$ language plpgsql;@@

SELECT check_for_obsolete_columns();
SELECT update_shadow_triggers();