--liquibase formatted sql
--preconditions dbms:postgresql

--changeset tsubklewe:initialize_database_from_dump endDelimiter:@@
--preconditions onFail:MARK_RAN onError:HALT
--precondition-sql-check expectedResult:0 SELECT count(*) FROM information_schema.tables where table_name = 'juco_version';
CREATE FUNCTION public._(character varying, character varying) RETURNS character varying
    LANGUAGE plpgsql STABLE
AS $_$
DECLARE
    _original ALIAS FOR $1;
    _language_id ALIAS FOR $2;
    _translation VARCHAR(255);
BEGIN
    IF _original IS NULL OR _language_id IS NULL THEN
        RETURN _original;
    END IF;
    SELECT translation INTO _translation
    FROM translation
    WHERE translation.language_id = _language_id
      AND translation.native = _original;
    IF NOT FOUND OR _translation IS NULL THEN
        RETURN _original;
    ELSE
        RETURN _translation;
    END IF;
END;
$_$;@@

CREATE FUNCTION public.check_db_version(min_version integer) RETURNS void
    LANGUAGE plpgsql
AS $$
DECLARE
    db_version INTEGER;
BEGIN
    SELECT setting INTO db_version from pg_settings where name='server_version_num';
    IF db_version < min_version THEN
        RAISE EXCEPTION 'DB version is obsolete, minimum version requirement not statisfied.';
    END IF;
END
$$;@@

CREATE FUNCTION public.check_for_obsolete_columns() RETURNS void
    LANGUAGE plpgsql
AS $$
DECLARE
    obsolete_columns TEXT := '';
    sh_rec RECORD;
BEGIN
    FOR sh_rec IN
        (SELECT table_name, shadow_table_name FROM shadow_table)
        LOOP
            obsolete_columns := simple_join(obsolete_columns, ', ', compare_tables(sh_rec.table_name, sh_rec.shadow_table_name));
            obsolete_columns := simple_join(obsolete_columns, ', ', compare_tables(sh_rec.shadow_table_name, sh_rec.table_name, 'delete_action_id'));
        END LOOP;

    IF length(obsolete_columns) > 0 THEN
        RAISE EXCEPTION 'Unexpected columns found: %', obsolete_columns;
    END IF;
END
$$;@@

CREATE FUNCTION public.compare_tables(table1 text, table2 text, filter_col text DEFAULT ''::text) RETURNS text
    LANGUAGE plpgsql
AS $$
DECLARE
    obsolete_columns TEXT := '';
    obs_col RECORD;
BEGIN
    FOR obs_col IN (
        select table1 || '.' || a.attname as obs_col_name
        from pg_catalog.pg_attribute a
                 join pg_catalog.pg_class c on c.oid = a.attrelid
                 join pg_catalog.pg_namespace n on n.oid = c.relnamespace
        where a.attnum > 0
          and not a.attisdropped
          and c.relname = table1
          and n.nspname = 'public'
            except
        select table1 || '.' || a.attname as obs_col_name
        from pg_catalog.pg_attribute a
                 join pg_catalog.pg_class c on c.oid = a.attrelid
                 join pg_catalog.pg_namespace n on n.oid = c.relnamespace
        where a.attnum > 0
          and not a.attisdropped
          and c.relname = table2
          and n.nspname = 'public'
    )
        LOOP
            IF (length(filter_col) = 0 OR obs_col.obs_col_name <> (table1 || '.' || filter_col)) THEN
                obsolete_columns := simple_join(obsolete_columns, ', ', obs_col.obs_col_name);
            END IF;
        END LOOP;

    RETURN obsolete_columns;
END
$$;@@

CREATE FUNCTION public.conversation_ad_fn() RETURNS trigger
    LANGUAGE plpgsql
AS $$
DECLARE del_ac_id BIGINT;
BEGIN
    SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = 't';
    IF del_ac_id IS NOT NULL THEN
        INSERT INTO sh_conversation("id", "type", "create_date", "title", "last_message_date", "creation_id", "last_message_update", delete_action_id)
        values (OLD."id", OLD."type", OLD."create_date", OLD."title", OLD."last_message_date", OLD."creation_id", OLD."last_message_update", del_ac_id);
    END IF;
    RETURN NEW;
END
$$;@@

CREATE FUNCTION public.conversation_attachment_ad_fn() RETURNS trigger
    LANGUAGE plpgsql
AS $$
DECLARE del_ac_id BIGINT;
BEGIN
    SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = 't';
    IF del_ac_id IS NOT NULL THEN
        INSERT INTO sh_conversation_attachment("id", "message_id", "document_id", "document_version_id", "filename", "file_size", "content_type", "version", "attachment_type", "image_height", "image_width", "create_date", delete_action_id)
        values (OLD."id", OLD."message_id", OLD."document_id", OLD."document_version_id", OLD."filename", OLD."file_size", OLD."content_type", OLD."version", OLD."attachment_type", OLD."image_height", OLD."image_width", OLD."create_date", del_ac_id);
    END IF;
    RETURN NEW;
END
$$;@@

CREATE FUNCTION public.conversation_message_ad_fn() RETURNS trigger
    LANGUAGE plpgsql
AS $$
DECLARE del_ac_id BIGINT;
BEGIN
    SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = 't';
    IF del_ac_id IS NOT NULL THEN
        INSERT INTO sh_conversation_message("id", "conversation_id", "from_person_id", "create_date", "message", "message_type", "creation_id", "modify_date", "delete_date", delete_action_id)
        values (OLD."id", OLD."conversation_id", OLD."from_person_id", OLD."create_date", OLD."message", OLD."message_type", OLD."creation_id", OLD."modify_date", OLD."delete_date", del_ac_id);
    END IF;
    RETURN NEW;
END
$$;@@

CREATE FUNCTION public.conversation_participant_ad_fn() RETURNS trigger
    LANGUAGE plpgsql
AS $$
DECLARE del_ac_id BIGINT;
BEGIN
    SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = 't';
    IF del_ac_id IS NOT NULL THEN
        INSERT INTO sh_conversation_participant("conversation_id", "participant_id", "create_date", "modify_date", "read_date", "deleted", "last_mail_sent", "deleted_with_conversation", delete_action_id)
        values (OLD."conversation_id", OLD."participant_id", OLD."create_date", OLD."modify_date", OLD."read_date", OLD."deleted", OLD."last_mail_sent", OLD."deleted_with_conversation", del_ac_id);
    END IF;
    RETURN NEW;
END
$$;@@

CREATE FUNCTION public.conversation_settings_ad_fn() RETURNS trigger
    LANGUAGE plpgsql
AS $$
DECLARE del_ac_id BIGINT;
BEGIN
    SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = 't';
    IF del_ac_id IS NOT NULL THEN
        INSERT INTO sh_conversation_settings("conversation_id", "participant_id", "muted", "create_date", "modify_date", "hidden", delete_action_id)
        values (OLD."conversation_id", OLD."participant_id", OLD."muted", OLD."create_date", OLD."modify_date", OLD."hidden", del_ac_id);
    END IF;
    RETURN NEW;
END
$$;@@

CREATE FUNCTION public.ctime_mtime_columns(tname text) RETURNS void
    LANGUAGE plpgsql
AS $$
BEGIN
    EXECUTE 'ALTER TABLE ' || tname || ' ADD COLUMN ctime TIMESTAMP WITHOUT TIME ZONE NOT NULL DEFAULT clock_timestamp();';
    EXECUTE 'ALTER TABLE ' || tname || ' ADD COLUMN mtime TIMESTAMP WITHOUT TIME ZONE NOT NULL DEFAULT clock_timestamp();';
    EXECUTE 'CREATE TRIGGER ' || tname || '_t BEFORE UPDATE ON ' || tname || ' FOR EACH ROW EXECUTE PROCEDURE update_mtime();';
END;
$$;@@

CREATE FUNCTION public.default_album_ad_fn() RETURNS trigger
    LANGUAGE plpgsql
AS $$
DECLARE del_ac_id BIGINT;
BEGIN
    SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = 't';
    IF del_ac_id IS NOT NULL THEN
        INSERT INTO sh_default_album("item_id", "item_type", "album_id", delete_action_id)
        values (OLD."item_id", OLD."item_type", OLD."album_id", del_ac_id);
    END IF;
    RETURN NEW;
END
$$;@@

CREATE FUNCTION public.drop_contraint_if_exists(t_name text, c_name text) RETURNS void
    LANGUAGE plpgsql
AS $$
BEGIN
    IF EXISTS (SELECT constraint_name FROM information_schema.constraint_column_usage WHERE table_name = t_name  AND constraint_name = c_name) THEN
        EXECUTE 'ALTER TABLE ' || quote_ident(t_name) || ' DROP CONSTRAINT ' || quote_ident(c_name) || ';';
    END IF;
END;
$$;@@

CREATE FUNCTION public.dropconstraint(tblschema character varying, tblname character varying, cstname character varying) RETURNS void
    LANGUAGE plpgsql
AS $$
DECLARE
    exec_string TEXT;
BEGIN
    exec_string := 'ALTER TABLE ';
    IF tblSchema != NULL THEN
        exec_string := exec_string || quote_ident(tblSchema) || '.';
    END IF;
    exec_string := exec_string || quote_ident(tblName)
                       || ' DROP CONSTRAINT '
        || quote_ident(cstName);
    EXECUTE exec_string;
EXCEPTION
    WHEN OTHERS THEN
        NULL;
END;
$$;@@

CREATE FUNCTION public.email_notification_setting_ad_fn() RETURNS trigger
    LANGUAGE plpgsql
AS $$
DECLARE del_ac_id BIGINT;
BEGIN
    SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = 't';
    IF del_ac_id IS NOT NULL THEN
        INSERT INTO sh_email_notification_setting("person_id", "create_date", "modify_date", "notifications_enabled", delete_action_id)
        values (OLD."person_id", OLD."create_date", OLD."modify_date", OLD."notifications_enabled", del_ac_id);
    END IF;
    RETURN NEW;
END
$$;@@

CREATE FUNCTION public.entity_ad_fn() RETURNS trigger
    LANGUAGE plpgsql
AS $$
DECLARE del_ac_id BIGINT;
BEGIN
    SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = 't';
    IF del_ac_id IS NOT NULL THEN
        INSERT INTO sh_entity("id", "name", "start_at", "end_at", "closing_date", "timezone", "description", "location", "tags", "subdomain", "subdomain_deactivated", "external_domain", "logo", "icon", "create_date", "modify_date", "use_parent_admin_as_coadmin", "entity_type", "slideshow_image", "participant_limit", "assignee_id", "default_entity_tab", "default_navigation", "last_drive_change", delete_action_id)
        values (OLD."id", OLD."name", OLD."start_at", OLD."end_at", OLD."closing_date", OLD."timezone", OLD."description", OLD."location", OLD."tags", OLD."subdomain", OLD."subdomain_deactivated", OLD."external_domain", OLD."logo", OLD."icon", OLD."create_date", OLD."modify_date", OLD."use_parent_admin_as_coadmin", OLD."entity_type", OLD."slideshow_image", OLD."participant_limit", OLD."assignee_id", OLD."default_entity_tab", OLD."default_navigation", OLD."last_drive_change", del_ac_id);
    END IF;
    RETURN NEW;
END
$$;@@

CREATE FUNCTION public.entity_component_ad_fn() RETURNS trigger
    LANGUAGE plpgsql
AS $$
DECLARE del_ac_id BIGINT;
BEGIN
    SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = 't';
    IF del_ac_id IS NOT NULL THEN
        INSERT INTO sh_entity_component("entity_id", "component_type", "position", "order", "visible", "open", "customized_title", delete_action_id)
        values (OLD."entity_id", OLD."component_type", OLD."position", OLD."order", OLD."visible", OLD."open", OLD."customized_title", del_ac_id);
    END IF;
    RETURN NEW;
END
$$;@@

CREATE FUNCTION public.entity_component_open_for_ad_fn() RETURNS trigger
    LANGUAGE plpgsql
AS $$
DECLARE del_ac_id BIGINT;
BEGIN
    SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = 't';
    IF del_ac_id IS NOT NULL THEN
        INSERT INTO sh_entity_component_open_for("entity_id", "component_type", "authority_name", delete_action_id)
        values (OLD."entity_id", OLD."component_type", OLD."authority_name", del_ac_id);
    END IF;
    RETURN NEW;
END
$$;@@

CREATE FUNCTION public.entity_dynamic_attribute_ad_fn() RETURNS trigger
    LANGUAGE plpgsql
AS $$
DECLARE del_ac_id BIGINT;
BEGIN
    SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = 't';
    IF del_ac_id IS NOT NULL THEN
        INSERT INTO sh_entity_dynamic_attribute("entity_id", "attribute_id", "attribute_value", "position", delete_action_id)
        values (OLD."entity_id", OLD."attribute_id", OLD."attribute_value", OLD."position", del_ac_id);
    END IF;
    RETURN NEW;
END
$$;@@

CREATE FUNCTION public.entity_internal_invitation_ad_fn() RETURNS trigger
    LANGUAGE plpgsql
AS $$
DECLARE del_ac_id BIGINT;
BEGIN
    SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = 't';
    IF del_ac_id IS NOT NULL THEN
        INSERT INTO sh_entity_internal_invitation("person_id", "entity_id", "inviter_id", "invite_message", "create_date", delete_action_id)
        values (OLD."person_id", OLD."entity_id", OLD."inviter_id", OLD."invite_message", OLD."create_date", del_ac_id);
    END IF;
    RETURN NEW;
END
$$;@@

CREATE FUNCTION public.entity_link_ad_fn() RETURNS trigger
    LANGUAGE plpgsql
AS $$
DECLARE del_ac_id BIGINT;
BEGIN
    SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = 't';
    IF del_ac_id IS NOT NULL THEN
        INSERT INTO sh_entity_link("entity_id", "linked_entity_id", "create_date", "modify_date", "inherit_admin", "inherit_member", delete_action_id)
        values (OLD."entity_id", OLD."linked_entity_id", OLD."create_date", OLD."modify_date", OLD."inherit_admin", OLD."inherit_member", del_ac_id);
    END IF;
    RETURN NEW;
END
$$;@@

CREATE FUNCTION public.entity_member_role_ad_fn() RETURNS trigger
    LANGUAGE plpgsql
AS $$
DECLARE del_ac_id BIGINT;
BEGIN
    SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = 't';
    IF del_ac_id IS NOT NULL THEN
        INSERT INTO sh_entity_member_role("person_id", "entity_id", "role", "create_date", "workflow", delete_action_id)
        values (OLD."person_id", OLD."entity_id", OLD."role", OLD."create_date", OLD."workflow", del_ac_id);
    END IF;
    RETURN NEW;
END
$$;@@

CREATE FUNCTION public.entity_nav_link_ad_fn() RETURNS trigger
    LANGUAGE plpgsql
AS $$
DECLARE del_ac_id BIGINT;
BEGIN
    SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = 't';
    IF del_ac_id IS NOT NULL THEN
        INSERT INTO sh_entity_nav_link("id", "entity_id", "text", "target", "list_index", "create_date", "modify_date", delete_action_id)
        values (OLD."id", OLD."entity_id", OLD."text", OLD."target", OLD."list_index", OLD."create_date", OLD."modify_date", del_ac_id);
    END IF;
    RETURN NEW;
END
$$;@@

CREATE FUNCTION public.entity_nav_sublink_ad_fn() RETURNS trigger
    LANGUAGE plpgsql
AS $$
DECLARE del_ac_id BIGINT;
BEGIN
    SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = 't';
    IF del_ac_id IS NOT NULL THEN
        INSERT INTO sh_entity_nav_sublink("id", "parent_id", "text", "target", "list_index", "create_date", "edit_date", delete_action_id)
        values (OLD."id", OLD."parent_id", OLD."text", OLD."target", OLD."list_index", OLD."create_date", OLD."edit_date", del_ac_id);
    END IF;
    RETURN NEW;
END
$$;@@

CREATE FUNCTION public.entity_privacy_ad_fn() RETURNS trigger
    LANGUAGE plpgsql
AS $$
DECLARE del_ac_id BIGINT;
BEGIN
    SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = 't';
    IF del_ac_id IS NOT NULL THEN
        INSERT INTO sh_entity_privacy("entity_id", "privacy_type", "authority_name", "create_date", "modify_date", delete_action_id)
        values (OLD."entity_id", OLD."privacy_type", OLD."authority_name", OLD."create_date", OLD."modify_date", del_ac_id);
    END IF;
    RETURN NEW;
END
$$;@@

CREATE FUNCTION public.entity_recommendation_ad_fn() RETURNS trigger
    LANGUAGE plpgsql
AS $$
DECLARE del_ac_id BIGINT;
BEGIN
    SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = 't';
    IF del_ac_id IS NOT NULL THEN
        INSERT INTO sh_entity_recommendation("person_id", "entity_id", "recommender_id", "recommend_message", "create_date", delete_action_id)
        values (OLD."person_id", OLD."entity_id", OLD."recommender_id", OLD."recommend_message", OLD."create_date", del_ac_id);
    END IF;
    RETURN NEW;
END
$$;@@

CREATE FUNCTION public.entity_to_entity_type_privacy_ad_fn() RETURNS trigger
    LANGUAGE plpgsql
AS $$
DECLARE del_ac_id BIGINT;
BEGIN
    SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = 't';
    IF del_ac_id IS NOT NULL THEN
        INSERT INTO sh_entity_to_entity_type_priva("entity_id", "entity_type", "authority_name", "create_date", "modify_date", delete_action_id)
        values (OLD."entity_id", OLD."entity_type", OLD."authority_name", OLD."create_date", OLD."modify_date", del_ac_id);
    END IF;
    RETURN NEW;
END
$$;@@

CREATE FUNCTION public.entity_version_ad_fn() RETURNS trigger
    LANGUAGE plpgsql
AS $$
DECLARE del_ac_id BIGINT;
BEGIN
    SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = 't';
    IF del_ac_id IS NOT NULL THEN
        INSERT INTO sh_entity_version("id", "entity_id", "person_id", "major_version", "minor_version", "draft_version", "version_type", "target_type", "version_status", "release_date", "expiration_date", "change_message", "comment_for_reviewers", "read_date", "decline_message", "create_date", "modify_date", "entity_status_id", delete_action_id)
        values (OLD."id", OLD."entity_id", OLD."person_id", OLD."major_version", OLD."minor_version", OLD."draft_version", OLD."version_type", OLD."target_type", OLD."version_status", OLD."release_date", OLD."expiration_date", OLD."change_message", OLD."comment_for_reviewers", OLD."read_date", OLD."decline_message", OLD."create_date", OLD."modify_date", OLD."entity_status_id", del_ac_id);
    END IF;
    RETURN NEW;
END
$$;@@

CREATE FUNCTION public.entity_version_components_ad_fn() RETURNS trigger
    LANGUAGE plpgsql
AS $$
DECLARE del_ac_id BIGINT;
BEGIN
    SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = 't';
    IF del_ac_id IS NOT NULL THEN
        INSERT INTO sh_entity_version_components("entity_version_id", "multi_wiki_version_id", delete_action_id)
        values (OLD."entity_version_id", OLD."multi_wiki_version_id", del_ac_id);
    END IF;
    RETURN NEW;
END
$$;@@

CREATE FUNCTION public.entity_version_review_ad_fn() RETURNS trigger
    LANGUAGE plpgsql
AS $$
DECLARE del_ac_id BIGINT;
BEGIN
    SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = 't';
    IF del_ac_id IS NOT NULL THEN
        INSERT INTO sh_entity_version_review("entity_version_id", "reviewer_id", "inviter_id", "review_status", "review_order", delete_action_id)
        values (OLD."entity_version_id", OLD."reviewer_id", OLD."inviter_id", OLD."review_status", OLD."review_order", del_ac_id);
    END IF;
    RETURN NEW;
END
$$;@@

CREATE FUNCTION public.entity_waitlist_ad_fn() RETURNS trigger
    LANGUAGE plpgsql
AS $$
DECLARE del_ac_id BIGINT;
BEGIN
    SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = 't';
    IF del_ac_id IS NOT NULL THEN
        INSERT INTO sh_entity_waitlist("entity_id", "person_id", "waitlist_order", delete_action_id)
        values (OLD."entity_id", OLD."person_id", OLD."waitlist_order", del_ac_id);
    END IF;
    RETURN NEW;
END
$$;@@

CREATE FUNCTION public.external_invitation_ad_fn() RETURNS trigger
    LANGUAGE plpgsql
AS $$
DECLARE del_ac_id BIGINT;
BEGIN
    SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = 't';
    IF del_ac_id IS NOT NULL THEN
        INSERT INTO sh_external_invitation("inviter_id", "email", "hash", "entity_id", delete_action_id)
        values (OLD."inviter_id", OLD."email", OLD."hash", OLD."entity_id", del_ac_id);
    END IF;
    RETURN NEW;
END
$$;@@

CREATE FUNCTION public.insert_translation(character varying, character, character varying) RETURNS void
    LANGUAGE plpgsql
AS $_$
DECLARE
    _native ALIAS FOR $1;
    _language ALIAS FOR $2;
    _translation ALIAS FOR $3;
    _dummy VARCHAR(255);
    _native_id BIGINT;
BEGIN
    SELECT id INTO _native_id FROM native WHERE native = _native;
    IF NOT FOUND THEN
        SELECT nextval('native_seq') INTO _native_id;
        INSERT INTO native (id, native) VALUES (_native_id, _native);
    END IF;
    SELECT translation INTO _dummy FROM translation
    WHERE translation.language_id = _language AND translation.native_id = _native_id;
    IF NOT FOUND THEN
        INSERT INTO translation (native_id, language_id, translation) VALUES (_native_id, _language, _translation);
    END IF;
    RETURN;
END;
$_$;@@

CREATE FUNCTION public.like_table_ad_fn() RETURNS trigger
    LANGUAGE plpgsql
AS $$
DECLARE del_ac_id BIGINT;
BEGIN
    SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = 't';
    IF del_ac_id IS NOT NULL THEN
        INSERT INTO sh_like_table("liker_id", "created_at", "modified_at", "recent", "item_id", delete_action_id)
        values (OLD."liker_id", OLD."created_at", OLD."modified_at", OLD."recent", OLD."item_id", del_ac_id);
    END IF;
    RETURN NEW;
END
$$;@@

CREATE FUNCTION public.map_person_person_ad_fn() RETURNS trigger
    LANGUAGE plpgsql
AS $$
DECLARE del_ac_id BIGINT;
BEGIN
    SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = 't';
    IF del_ac_id IS NOT NULL THEN
        INSERT INTO sh_map_person_person("person_id", "friend_id", delete_action_id)
        values (OLD."person_id", OLD."friend_id", del_ac_id);
    END IF;
    RETURN NEW;
END
$$;@@

CREATE FUNCTION public.map_person_person_invite_ad_fn() RETURNS trigger
    LANGUAGE plpgsql
AS $$
DECLARE del_ac_id BIGINT;
BEGIN
    SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = 't';
    IF del_ac_id IS NOT NULL THEN
        INSERT INTO sh_map_person_person_invite("person_id", "friend_id", "message", delete_action_id)
        values (OLD."person_id", OLD."friend_id", OLD."message", del_ac_id);
    END IF;
    RETURN NEW;
END
$$;@@

CREATE FUNCTION public.mdl_album_ad_fn() RETURNS trigger
    LANGUAGE plpgsql
AS $$
DECLARE del_ac_id BIGINT;
BEGIN
    SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = 't';
    IF del_ac_id IS NOT NULL THEN
        INSERT INTO sh_mdl_album("id", "parent_item_id", "parent_item_type", "module", "owner_id", "create_date", "edit_date", "title", "description", "location", "friends_only", "old_album_id", delete_action_id)
        values (OLD."id", OLD."parent_item_id", OLD."parent_item_type", OLD."module", OLD."owner_id", OLD."create_date", OLD."edit_date", OLD."title", OLD."description", OLD."location", OLD."friends_only", OLD."old_album_id", del_ac_id);
    END IF;
    RETURN NEW;
END
$$;@@

CREATE FUNCTION public.mdl_album_image_ad_fn() RETURNS trigger
    LANGUAGE plpgsql
AS $$
DECLARE del_ac_id BIGINT;
BEGIN
    SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = 't';
    IF del_ac_id IS NOT NULL THEN
        INSERT INTO sh_mdl_album_image("id", "title", "album_id", "filename", "owner_id", "create_date", "edit_date", "old_album_image_id", delete_action_id)
        values (OLD."id", OLD."title", OLD."album_id", OLD."filename", OLD."owner_id", OLD."create_date", OLD."edit_date", OLD."old_album_image_id", del_ac_id);
    END IF;
    RETURN NEW;
END
$$;@@

CREATE FUNCTION public.mdl_comment_ad_fn() RETURNS trigger
    LANGUAGE plpgsql
AS $$
DECLARE del_ac_id BIGINT;
BEGIN
    SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = 't';
    IF del_ac_id IS NOT NULL THEN
        INSERT INTO sh_mdl_comment("id", "author_id", "text", "posted_at", "custom_info", "editor_id", "edited_at", "parent_item_id", "editor_model", delete_action_id)
        values (OLD."id", OLD."author_id", OLD."text", OLD."posted_at", OLD."custom_info", OLD."editor_id", OLD."edited_at", OLD."parent_item_id", OLD."editor_model", del_ac_id);
    END IF;
    RETURN NEW;
END
$$;@@

CREATE FUNCTION public.mdl_document_ad_fn() RETURNS trigger
    LANGUAGE plpgsql
AS $$
DECLARE del_ac_id BIGINT;
BEGIN
    SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = 't';
    IF del_ac_id IS NOT NULL THEN
        INSERT INTO sh_mdl_document("id", "parent_item_id", "parent_item_type", "description", "keywords", "download_count", "checked_out_by", "checked_out_at", "status", "status_changed_by", "status_changed_at", "modified_at", delete_action_id)
        values (OLD."id", OLD."parent_item_id", OLD."parent_item_type", OLD."description", OLD."keywords", OLD."download_count", OLD."checked_out_by", OLD."checked_out_at", OLD."status", OLD."status_changed_by", OLD."status_changed_at", OLD."modified_at", del_ac_id);
    END IF;
    RETURN NEW;
END
$$;@@

CREATE FUNCTION public.mdl_document_version_ad_fn() RETURNS trigger
    LANGUAGE plpgsql
AS $$
DECLARE del_ac_id BIGINT;
BEGIN
    SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = 't';
    IF del_ac_id IS NOT NULL THEN
        INSERT INTO sh_mdl_document_version("id", "version", "filename", "document_id", "url", "uploaded_by", "uploaded_at", "size", "file_type_id", "restored_from", "notified_others", delete_action_id)
        values (OLD."id", OLD."version", OLD."filename", OLD."document_id", OLD."url", OLD."uploaded_by", OLD."uploaded_at", OLD."size", OLD."file_type_id", OLD."restored_from", OLD."notified_others", del_ac_id);
    END IF;
    RETURN NEW;
END
$$;@@

CREATE FUNCTION public.mdl_video_link_ad_fn() RETURNS trigger
    LANGUAGE plpgsql
AS $$
DECLARE del_ac_id BIGINT;
BEGIN
    SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = 't';
    IF del_ac_id IS NOT NULL THEN
        INSERT INTO sh_mdl_video_link("id", "parent_item_id", "parent_item_type", "title", "url", "edit_date", "create_date", "author_id", "public", "video_file", delete_action_id)
        values (OLD."id", OLD."parent_item_id", OLD."parent_item_type", OLD."title", OLD."url", OLD."edit_date", OLD."create_date", OLD."author_id", OLD."public", OLD."video_file", del_ac_id);
    END IF;
    RETURN NEW;
END
$$;@@

CREATE FUNCTION public.multi_wiki_ad_fn() RETURNS trigger
    LANGUAGE plpgsql
AS $$
DECLARE del_ac_id BIGINT;
BEGIN
    SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = 't';
    IF del_ac_id IS NOT NULL THEN
        INSERT INTO sh_multi_wiki("id", "entity_id", "create_date", "modify_date", delete_action_id)
        values (OLD."id", OLD."entity_id", OLD."create_date", OLD."modify_date", del_ac_id);
    END IF;
    RETURN NEW;
END
$$;@@

CREATE FUNCTION public.multi_wiki_section_visibility_ad_fn() RETURNS trigger
    LANGUAGE plpgsql
AS $$
DECLARE del_ac_id BIGINT;
BEGIN
    SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = 't';
    IF del_ac_id IS NOT NULL THEN
        INSERT INTO sh_multi_wiki_section_visibili("version_section_id", "authority_name", "create_date", "modify_date", delete_action_id)
        values (OLD."version_section_id", OLD."authority_name", OLD."create_date", OLD."modify_date", del_ac_id);
    END IF;
    RETURN NEW;
END
$$;@@

CREATE FUNCTION public.multi_wiki_version_ad_fn() RETURNS trigger
    LANGUAGE plpgsql
AS $$
DECLARE del_ac_id BIGINT;
BEGIN
    SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = 't';
    IF del_ac_id IS NOT NULL THEN
        INSERT INTO sh_multi_wiki_version("id", "wiki_id", "create_date", "modify_date", delete_action_id)
        values (OLD."id", OLD."wiki_id", OLD."create_date", OLD."modify_date", del_ac_id);
    END IF;
    RETURN NEW;
END
$$;@@

CREATE FUNCTION public.multi_wiki_version_section_ad_fn() RETURNS trigger
    LANGUAGE plpgsql
AS $$
DECLARE del_ac_id BIGINT;
BEGIN
    SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = 't';
    IF del_ac_id IS NOT NULL THEN
        INSERT INTO sh_multi_wiki_version_section("id", "version_id", "headline", "html", "create_date", "modify_date", "section_id", "section_order", delete_action_id)
        values (OLD."id", OLD."version_id", OLD."headline", OLD."html", OLD."create_date", OLD."modify_date", OLD."section_id", OLD."section_order", del_ac_id);
    END IF;
    RETURN NEW;
END
$$;@@

CREATE FUNCTION public.news_blog_ad_fn() RETURNS trigger
    LANGUAGE plpgsql
AS $$
DECLARE del_ac_id BIGINT;
BEGIN
    SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = 't';
    IF del_ac_id IS NOT NULL THEN
        INSERT INTO sh_news_blog("id", "parent_item_id", "parent_item_type", "headline", "create_date", "edit_date", "author_id", "source_text", "source_href", "keywords", "text", "image_id", "draft", delete_action_id)
        values (OLD."id", OLD."parent_item_id", OLD."parent_item_type", OLD."headline", OLD."create_date", OLD."edit_date", OLD."author_id", OLD."source_text", OLD."source_href", OLD."keywords", OLD."text", OLD."image_id", OLD."draft", del_ac_id);
    END IF;
    RETURN NEW;
END
$$;@@

CREATE FUNCTION public.person_ad_fn() RETURNS trigger
    LANGUAGE plpgsql
AS $$
DECLARE del_ac_id BIGINT;
BEGIN
    SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = 't';
    IF del_ac_id IS NOT NULL THEN
        INSERT INTO sh_person("id", "last_login", "show_tempstartpage", "account_active", "title", "firstname", "surname", "email", "newmail", "birthday", "password_hash", "keywords", "language_id", "password_question", "password_answer", "registerdate", "terms_accepted", "image", "blocked", "default_timezone", "modified_at", "created_at", "birthday_visibility", "hashed_id", "additional_title", "deleted", delete_action_id)
        values (OLD."id", OLD."last_login", OLD."show_tempstartpage", OLD."account_active", OLD."title", OLD."firstname", OLD."surname", OLD."email", OLD."newmail", OLD."birthday", OLD."password_hash", OLD."keywords", OLD."language_id", OLD."password_question", OLD."password_answer", OLD."registerdate", OLD."terms_accepted", OLD."image", OLD."blocked", OLD."default_timezone", OLD."modified_at", OLD."created_at", OLD."birthday_visibility", OLD."hashed_id", OLD."additional_title", OLD."deleted", del_ac_id);
    END IF;
    RETURN NEW;
END
$$;@@

CREATE FUNCTION public.person_external_ids_ad_fn() RETURNS trigger
    LANGUAGE plpgsql
AS $$
DECLARE del_ac_id BIGINT;
BEGIN
    SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = 't';
    IF del_ac_id IS NOT NULL THEN
        INSERT INTO sh_person_external_ids("person_id", "external_person_id", "external_system_id", delete_action_id)
        values (OLD."person_id", OLD."external_person_id", OLD."external_system_id", del_ac_id);
    END IF;
    RETURN NEW;
END
$$;@@

CREATE FUNCTION public.person_to_role_ad_fn() RETURNS trigger
    LANGUAGE plpgsql
AS $$
DECLARE del_ac_id BIGINT;
BEGIN
    SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = 't';
    IF del_ac_id IS NOT NULL THEN
        INSERT INTO sh_person_to_role("person_id", "role_name", "create_date", "modify_date", delete_action_id)
        values (OLD."person_id", OLD."role_name", OLD."create_date", OLD."modify_date", del_ac_id);
    END IF;
    RETURN NEW;
END
$$;@@

CREATE FUNCTION public.profile_attribute_ad_fn() RETURNS trigger
    LANGUAGE plpgsql
AS $$
DECLARE del_ac_id BIGINT;
BEGIN
    SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = 't';
    IF del_ac_id IS NOT NULL THEN
        INSERT INTO sh_profile_attribute("id", "person_id", "name", "value", "value_order", "block_order", "create_date", delete_action_id)
        values (OLD."id", OLD."person_id", OLD."name", OLD."value", OLD."value_order", OLD."block_order", OLD."create_date", del_ac_id);
    END IF;
    RETURN NEW;
END
$$;@@

CREATE FUNCTION public.profile_component_ad_fn() RETURNS trigger
    LANGUAGE plpgsql
AS $$
DECLARE del_ac_id BIGINT;
BEGIN
    SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = 't';
    IF del_ac_id IS NOT NULL THEN
        INSERT INTO sh_profile_component("person_id", "component_type", "position", "order", "create_date", "modify_date", delete_action_id)
        values (OLD."person_id", OLD."component_type", OLD."position", OLD."order", OLD."create_date", OLD."modify_date", del_ac_id);
    END IF;
    RETURN NEW;
END
$$;@@

CREATE FUNCTION public.profile_visibility_for_role_ad_fn() RETURNS trigger
    LANGUAGE plpgsql
AS $$
DECLARE del_ac_id BIGINT;
BEGIN
    SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = 't';
    IF del_ac_id IS NOT NULL THEN
        INSERT INTO sh_profile_visibility_for_role("person_id", "role_name_may_see", "modify_date", delete_action_id)
        values (OLD."person_id", OLD."role_name_may_see", OLD."modify_date", del_ac_id);
    END IF;
    RETURN NEW;
END
$$;@@

CREATE FUNCTION public.raweditor_version_ad_fn() RETURNS trigger
    LANGUAGE plpgsql
AS $$
DECLARE del_ac_id BIGINT;
BEGIN
    SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = 't';
    IF del_ac_id IS NOT NULL THEN
        INSERT INTO sh_raweditor_version("id", "raweditor_id", "parent_item_id", "parent_item_type", "editor_type", "version", "text", "create_date", "author_id", delete_action_id)
        values (OLD."id", OLD."raweditor_id", OLD."parent_item_id", OLD."parent_item_type", OLD."editor_type", OLD."version", OLD."text", OLD."create_date", OLD."author_id", del_ac_id);
    END IF;
    RETURN NEW;
END
$$;@@

CREATE FUNCTION public.simple_join(s1 text, sep text, s2 text) RETURNS text
    LANGUAGE plpgsql
AS $$
BEGIN
    IF (length(s1) > 0 AND length(s2) > 0) THEN
        RETURN s1 || sep || s2;
    ELSE
        RETURN s1 || s2;
    END IF;
END
$$;@@

CREATE FUNCTION public.subscriptions_ad_fn() RETURNS trigger
    LANGUAGE plpgsql
AS $$
DECLARE del_ac_id BIGINT;
BEGIN
    SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = 't';
    IF del_ac_id IS NOT NULL THEN
        INSERT INTO sh_subscriptions("subscriber_id", "subscription_type", "subscription_state", "created_at", "modified_at", "item_id", delete_action_id)
        values (OLD."subscriber_id", OLD."subscription_type", OLD."subscription_state", OLD."created_at", OLD."modified_at", OLD."item_id", del_ac_id);
    END IF;
    RETURN NEW;
END
$$;@@

CREATE FUNCTION public.survey_ad_fn() RETURNS trigger
    LANGUAGE plpgsql
AS $$
DECLARE del_ac_id BIGINT;
BEGIN
    SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = 't';
    IF del_ac_id IS NOT NULL THEN
        INSERT INTO sh_survey("id", "parent_item_id", "parent_item_type", "title", "expired_date", "owner_id", "result_visibilty", "anonym", "participant_may_modify", "auto_delete_answers", "create_date", "modify_date", "redirect_on_new_member", delete_action_id)
        values (OLD."id", OLD."parent_item_id", OLD."parent_item_type", OLD."title", OLD."expired_date", OLD."owner_id", OLD."result_visibilty", OLD."anonym", OLD."participant_may_modify", OLD."auto_delete_answers", OLD."create_date", OLD."modify_date", OLD."redirect_on_new_member", del_ac_id);
    END IF;
    RETURN NEW;
END
$$;@@

CREATE FUNCTION public.survey_participant_answer_ad_fn() RETURNS trigger
    LANGUAGE plpgsql
AS $$
DECLARE del_ac_id BIGINT;
BEGIN
    SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = 't';
    IF del_ac_id IS NOT NULL THEN
        INSERT INTO sh_survey_participant_answer("participant_id", "survey_question_id", "survey_predefined_answer_id", "freetext_answer", "create_date", "modify_date", delete_action_id)
        values (OLD."participant_id", OLD."survey_question_id", OLD."survey_predefined_answer_id", OLD."freetext_answer", OLD."create_date", OLD."modify_date", del_ac_id);
    END IF;
    RETURN NEW;
END
$$;@@

CREATE FUNCTION public.survey_predefined_answer_ad_fn() RETURNS trigger
    LANGUAGE plpgsql
AS $$
DECLARE del_ac_id BIGINT;
BEGIN
    SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = 't';
    IF del_ac_id IS NOT NULL THEN
        INSERT INTO sh_survey_predefined_answer("id", "survey_question_id", "answer", "order", "create_date", "modify_date", delete_action_id)
        values (OLD."id", OLD."survey_question_id", OLD."answer", OLD."order", OLD."create_date", OLD."modify_date", del_ac_id);
    END IF;
    RETURN NEW;
END
$$;@@

CREATE FUNCTION public.survey_question_ad_fn() RETURNS trigger
    LANGUAGE plpgsql
AS $$
DECLARE del_ac_id BIGINT;
BEGIN
    SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = 't';
    IF del_ac_id IS NOT NULL THEN
        INSERT INTO sh_survey_question("id", "survey_id", "question", "order", "freetext_answer", "min_answer", "max_answer", "create_date", "modify_date", delete_action_id)
        values (OLD."id", OLD."survey_id", OLD."question", OLD."order", OLD."freetext_answer", OLD."min_answer", OLD."max_answer", OLD."create_date", OLD."modify_date", del_ac_id);
    END IF;
    RETURN NEW;
END
$$;@@

CREATE FUNCTION public.task_list_ad_fn() RETURNS trigger
    LANGUAGE plpgsql
AS $$
DECLARE del_ac_id BIGINT;
BEGIN
    SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = 't';
    IF del_ac_id IS NOT NULL THEN
        INSERT INTO sh_task_list("id", "name", "create_date", "modify_date", delete_action_id)
        values (OLD."id", OLD."name", OLD."create_date", OLD."modify_date", del_ac_id);
    END IF;
    RETURN NEW;
END
$$;@@

CREATE FUNCTION public.task_list_permissions_ad_fn() RETURNS trigger
    LANGUAGE plpgsql
AS $$
DECLARE del_ac_id BIGINT;
BEGIN
    SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = 't';
    IF del_ac_id IS NOT NULL THEN
        INSERT INTO sh_task_list_permissions("task_list_id", "role", "task_right", "ctime", "mtime", delete_action_id)
        values (OLD."task_list_id", OLD."role", OLD."task_right", OLD."ctime", OLD."mtime", del_ac_id);
    END IF;
    RETURN NEW;
END
$$;@@

CREATE FUNCTION public.tasks_ad_fn() RETURNS trigger
    LANGUAGE plpgsql
AS $$
DECLARE del_ac_id BIGINT;
BEGIN
    SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = 't';
    IF del_ac_id IS NOT NULL THEN
        INSERT INTO sh_tasks("id", "name", "description", "owner_id", "assignee_id", "status", "create_date", "modify_date", "parent_task_id", "attachment_name", "attachment_url", "task_list_id", "due_date", delete_action_id)
        values (OLD."id", OLD."name", OLD."description", OLD."owner_id", OLD."assignee_id", OLD."status", OLD."create_date", OLD."modify_date", OLD."parent_task_id", OLD."attachment_name", OLD."attachment_url", OLD."task_list_id", OLD."due_date", del_ac_id);
    END IF;
    RETURN NEW;
END
$$;@@

CREATE FUNCTION public.update_mtime() RETURNS trigger
    LANGUAGE plpgsql
AS $$
BEGIN
    NEW.mtime = clock_timestamp();
    RETURN NEW;
END;
$$;@@

CREATE FUNCTION public.update_shadow_triggers() RETURNS void
    LANGUAGE plpgsql
AS $_$
DECLARE
    sh_rec RECORD;
    col_rec RECORD;
    tr_name TEXT;
    tr_fn_name TEXT;
    stmt TEXT;
    insert_cols TEXT;
    insert_vals TEXT;
BEGIN
    FOR sh_rec IN
        (SELECT table_name, shadow_table_name, delete_action_mandatory FROM shadow_table)
        LOOP
            tr_name := sh_rec.table_name || '_ad';
            tr_fn_name := sh_rec.table_name || '_ad_fn()';
            stmt := 'CREATE OR REPLACE FUNCTION ' || tr_fn_name || E' RETURNS trigger AS $$\n';
            stmt := stmt || E'DECLARE del_ac_id BIGINT;\n';
            stmt := stmt || E'BEGIN\n';
            stmt := stmt || E'  SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = ''t'';\n';

            IF sh_rec.delete_action_mandatory = 't' THEN
                stmt := stmt || E'  IF del_ac_id IS NULL THEN\n';
                stmt := stmt ||  '    RAISE EXCEPTION ''No delete action for current transaction found while trying to delete from ' || sh_rec.table_name || E''';\n';
                stmt := stmt || E'  END IF;\n';
            END IF;

            stmt := stmt || E'  IF del_ac_id IS NOT NULL THEN\n';
            stmt := stmt ||  '    INSERT INTO ' || sh_rec.shadow_table_name || '(';

            insert_cols := '';
            insert_vals := '';
            FOR col_rec IN (
                select a.attname as column_name
                from pg_catalog.pg_attribute a
                         join pg_catalog.pg_class c on c.oid = a.attrelid
                         join pg_catalog.pg_namespace n on n.oid = c.relnamespace
                where a.attnum > 0
                  and not a.attisdropped
                  and c.relname = sh_rec.table_name
                  and n.nspname = 'public'
                order by a.attnum
            )
                LOOP
                    insert_cols := insert_cols || '"' || col_rec.column_name || '", ';
                    insert_vals := insert_vals || 'OLD."' || col_rec.column_name || '", ';
                END LOOP;
            stmt := stmt || insert_cols || E'delete_action_id)\n';
            stmt := stmt || '    values (' || insert_vals || E'del_ac_id);\n';
            stmt := stmt || E'  END IF;\n';
            stmt := stmt || E'  RETURN NEW;\n';
            stmt := stmt || E'END\n';
            stmt := stmt || '$$ language plpgsql;';

            EXECUTE stmt;
            EXECUTE 'DROP TRIGGER IF EXISTS ' || tr_name || ' ON ' || sh_rec.table_name;
            EXECUTE 'CREATE TRIGGER ' || tr_name || ' AFTER DELETE ON ' || sh_rec.table_name || ' FOR EACH ROW EXECUTE procedure ' || tr_fn_name || ';';
        END LOOP;
END
$_$;@@

CREATE FUNCTION public.user_setting_ad_fn() RETURNS trigger
    LANGUAGE plpgsql
AS $$
DECLARE del_ac_id BIGINT;
BEGIN
    SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = 't';
    IF del_ac_id IS NOT NULL THEN
        INSERT INTO sh_user_setting("person_id", "key", "value", "create_date", "modify_date", delete_action_id)
        values (OLD."person_id", OLD."key", OLD."value", OLD."create_date", OLD."modify_date", del_ac_id);
    END IF;
    RETURN NEW;
END
$$;@@

CREATE FUNCTION public.wiki_editor_ad_fn() RETURNS trigger
    LANGUAGE plpgsql
AS $$
DECLARE del_ac_id BIGINT;
BEGIN
    SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = 't';
    IF del_ac_id IS NOT NULL THEN
        INSERT INTO sh_wiki_editor("id", "parent_item_id", "parent_item_type", "old_group_id", delete_action_id)
        values (OLD."id", OLD."parent_item_id", OLD."parent_item_type", OLD."old_group_id", del_ac_id);
    END IF;
    RETURN NEW;
END
$$;@@

CREATE FUNCTION public.wiki_editor_backup_ad_fn() RETURNS trigger
    LANGUAGE plpgsql
AS $$
DECLARE del_ac_id BIGINT;
BEGIN
    SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = 't';
    IF del_ac_id IS NOT NULL THEN
        INSERT INTO sh_wiki_editor_backup("wiki_editor_id", "wiki_text", "author_id", "create_date", delete_action_id)
        values (OLD."wiki_editor_id", OLD."wiki_text", OLD."author_id", OLD."create_date", del_ac_id);
    END IF;
    RETURN NEW;
END
$$;@@

CREATE FUNCTION public.wiki_editor_version_ad_fn() RETURNS trigger
    LANGUAGE plpgsql
AS $$
DECLARE del_ac_id BIGINT;
BEGIN
    SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = 't';
    IF del_ac_id IS NOT NULL THEN
        INSERT INTO sh_wiki_editor_version("id", "wiki_editor_id", "wiki_text", "version", "author_id", "create_date", "old_page_id", "restored_version", delete_action_id)
        values (OLD."id", OLD."wiki_editor_id", OLD."wiki_text", OLD."version", OLD."author_id", OLD."create_date", OLD."old_page_id", OLD."restored_version", del_ac_id);
    END IF;
    RETURN NEW;
END
$$;@@

CREATE FUNCTION public.workstream_attachment_ad_fn() RETURNS trigger
    LANGUAGE plpgsql
AS $$
DECLARE del_ac_id BIGINT;
BEGIN
    SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = 't';
    IF del_ac_id IS NOT NULL THEN
        INSERT INTO sh_workstream_attachment("entry_id", "position", "item_id", delete_action_id)
        values (OLD."entry_id", OLD."position", OLD."item_id", del_ac_id);
    END IF;
    RETURN NEW;
END
$$;@@

CREATE FUNCTION public.workstream_entry_ad_fn() RETURNS trigger
    LANGUAGE plpgsql
AS $$
DECLARE del_ac_id BIGINT;
BEGIN
    SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = 't';
    IF del_ac_id IS NOT NULL THEN
        INSERT INTO sh_workstream_entry("id", "type", "created_at", "modified_at", "origin_item_id", "subject_item_id", "author_id", "pinned", "pin_author", "pin_date", "published", "published_at", delete_action_id)
        values (OLD."id", OLD."type", OLD."created_at", OLD."modified_at", OLD."origin_item_id", OLD."subject_item_id", OLD."author_id", OLD."pinned", OLD."pin_author", OLD."pin_date", OLD."published", OLD."published_at", del_ac_id);
    END IF;
    RETURN NEW;
END
$$;@@

CREATE FUNCTION public.workstream_entry_attribute_ad_fn() RETURNS trigger
    LANGUAGE plpgsql
AS $$
DECLARE del_ac_id BIGINT;
BEGIN
    SELECT id INTO del_ac_id FROM delete_action WHERE transaction_id = txid_current() AND transaction_id IS NOT NULL AND archive_data = 't';
    IF del_ac_id IS NOT NULL THEN
        INSERT INTO sh_workstream_entry_attribute("entry_id", "key", "value", delete_action_id)
        values (OLD."entry_id", OLD."key", OLD."value", del_ac_id);
    END IF;
    RETURN NEW;
END
$$;@@

SET default_tablespace = '';

CREATE TABLE public.account_failed_logins (
                                              person_id bigint NOT NULL,
                                              counted_since timestamp without time zone NOT NULL,
                                              failed_count integer NOT NULL,
                                              ctime timestamp without time zone DEFAULT clock_timestamp() NOT NULL,
                                              mtime timestamp without time zone DEFAULT clock_timestamp() NOT NULL
);

CREATE TABLE public.account_lock (
                                     person_id bigint NOT NULL,
                                     locked_until timestamp without time zone NOT NULL,
                                     ctime timestamp without time zone DEFAULT clock_timestamp() NOT NULL,
                                     mtime timestamp without time zone DEFAULT clock_timestamp() NOT NULL
);

CREATE TABLE public.additional_app_info (
                                            app_id character varying(32) NOT NULL,
                                            url character varying(4000) NOT NULL,
                                            icon_url character varying(4000),
                                            color character varying(16) NOT NULL,
                                            is_iframe_app character(1) NOT NULL,
                                            ctime timestamp without time zone DEFAULT now() NOT NULL,
                                            mtime timestamp without time zone DEFAULT now() NOT NULL
);

CREATE SEQUENCE public.album_seq
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE TABLE public.another_juco_version_table (
    id uuid NOT NULL
);

CREATE TABLE public.app (
                            id character varying(32) NOT NULL,
                            name character varying(100) NOT NULL,
                            is_core_app character(1) NOT NULL,
                            default_order integer NOT NULL,
                            ctime timestamp without time zone DEFAULT now() NOT NULL,
                            mtime timestamp without time zone DEFAULT now() NOT NULL
);

CREATE SEQUENCE public.appointment_seq
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE TABLE public.audit_log (
                                  id bigint NOT NULL,
                                  person_id bigint,
                                  change_date timestamp without time zone NOT NULL,
                                  change_call character varying(255) NOT NULL,
                                  change_result text NOT NULL
);

CREATE TABLE public.audit_log_param (
                                        audit_id bigint NOT NULL,
                                        "position" integer NOT NULL,
                                        name character varying(255) NOT NULL,
                                        value text NOT NULL
);

CREATE SEQUENCE public.audit_log_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

CREATE TABLE public.base_property (
                                      id bigint NOT NULL,
                                      property_name character varying(255) NOT NULL,
                                      converter_class character varying(255),
                                      converter_config character varying(255),
                                      description character varying(1000) NOT NULL,
                                      default_value text,
                                      inheritance_type character varying(255) NOT NULL,
                                      container_name character varying(255) NOT NULL,
                                      last_changed timestamp without time zone DEFAULT now() NOT NULL,
                                      change_user character varying(255),
                                      create_at timestamp without time zone DEFAULT now() NOT NULL,
                                      created_version character varying(255) DEFAULT 'unknown'::character varying NOT NULL,
                                      last_used_version character varying(255) DEFAULT 'unknown'::character varying NOT NULL
);

CREATE TABLE public.chat_search_queue (
                                          id bigint NOT NULL,
                                          item_id bigint NOT NULL,
                                          item_type character varying(32) NOT NULL,
                                          created_at timestamp without time zone DEFAULT clock_timestamp() NOT NULL,
                                          modified_at timestamp without time zone DEFAULT clock_timestamp() NOT NULL,
                                          in_progress character(1) DEFAULT 'f'::bpchar NOT NULL,
                                          lock_id bigint
);

CREATE SEQUENCE public.chat_search_queue_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

CREATE TABLE public.conversation (
                                     id bigint NOT NULL,
                                     type character varying(255) NOT NULL,
                                     create_date timestamp without time zone DEFAULT now() NOT NULL,
                                     title character varying(1024),
                                     last_message_date timestamp without time zone DEFAULT now() NOT NULL,
                                     creation_id character varying(100),
                                     last_message_update timestamp without time zone DEFAULT now() NOT NULL
);

CREATE TABLE public.conversation_attachment (
                                                id bigint NOT NULL,
                                                message_id bigint NOT NULL,
                                                document_id character varying(255) NOT NULL,
                                                document_version_id character varying(255) NOT NULL,
                                                filename character varying(255) NOT NULL,
                                                file_size bigint DEFAULT 0 NOT NULL,
                                                content_type character varying(255) NOT NULL,
                                                version character varying(255) NOT NULL,
                                                attachment_type character varying(255) NOT NULL,
                                                image_height integer,
                                                image_width integer,
                                                create_date timestamp without time zone DEFAULT now() NOT NULL
);

CREATE SEQUENCE public.conversation_attachment_seq
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE TABLE public.conversation_change (
                                            id bigint NOT NULL,
                                            conversation_id bigint NOT NULL,
                                            participant_id bigint NOT NULL,
                                            change_type character varying(255) NOT NULL,
                                            create_date timestamp without time zone DEFAULT now() NOT NULL
);

CREATE SEQUENCE public.conversation_change_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

CREATE TABLE public.conversation_message (
                                             id bigint NOT NULL,
                                             conversation_id bigint NOT NULL,
                                             from_person_id bigint NOT NULL,
                                             create_date timestamp without time zone DEFAULT now() NOT NULL,
                                             message text,
                                             message_type character varying(32) DEFAULT 'CHAT_MESSAGE'::character varying NOT NULL,
                                             creation_id character varying(100),
                                             modify_date timestamp without time zone DEFAULT now() NOT NULL,
                                             delete_date timestamp without time zone
);

CREATE SEQUENCE public.conversation_message_seq
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE TABLE public.conversation_participant (
                                                 conversation_id bigint NOT NULL,
                                                 participant_id bigint NOT NULL,
                                                 create_date timestamp without time zone DEFAULT now() NOT NULL,
                                                 modify_date timestamp without time zone DEFAULT now() NOT NULL,
                                                 read_date timestamp without time zone DEFAULT now() NOT NULL,
                                                 deleted character(1) DEFAULT 'f'::bpchar NOT NULL,
                                                 last_mail_sent timestamp without time zone,
                                                 deleted_with_conversation character(1) DEFAULT 'f'::bpchar NOT NULL
);

CREATE TABLE public.conversation_search_queue (
                                                  id bigint NOT NULL,
                                                  item_id bigint NOT NULL,
                                                  item_type character varying(32) NOT NULL,
                                                  conversation_index_type character varying(32) NOT NULL,
                                                  created_at timestamp without time zone DEFAULT clock_timestamp() NOT NULL,
                                                  modified_at timestamp without time zone DEFAULT clock_timestamp() NOT NULL,
                                                  in_progress character(1) DEFAULT 'f'::bpchar NOT NULL,
                                                  lock_id bigint
);

CREATE SEQUENCE public.conversation_search_queue_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

CREATE SEQUENCE public.conversation_seq
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE TABLE public.conversation_settings (
                                              conversation_id bigint NOT NULL,
                                              participant_id bigint NOT NULL,
                                              muted character(1) DEFAULT 'f'::bpchar NOT NULL,
                                              create_date timestamp without time zone DEFAULT now() NOT NULL,
                                              modify_date timestamp without time zone DEFAULT now() NOT NULL,
                                              hidden character(1) DEFAULT 'f'::bpchar NOT NULL
);

CREATE TABLE public.customer_category (
                                          customertype character varying(50) NOT NULL,
                                          created_at timestamp without time zone NOT NULL,
                                          modified_at timestamp without time zone NOT NULL
);

CREATE TABLE public.customer_domain (
    customer_domain_name character varying(255) NOT NULL
);

CREATE TABLE public.db_patch_version (
                                         version character varying(255) NOT NULL,
                                         ctime timestamp without time zone DEFAULT now() NOT NULL
);

CREATE TABLE public.dbmaintain_scripts (
                                           file_name character varying(150),
                                           file_last_modified_at bigint,
                                           checksum character varying(50),
                                           executed_at character varying(20),
                                           succeeded bigint
);

CREATE TABLE public.default_album (
                                      item_id bigint NOT NULL,
                                      item_type character varying(32) NOT NULL,
                                      album_id bigint NOT NULL
);

CREATE TABLE public.default_app_visibility (
                                               app_id character varying(32) NOT NULL,
                                               visible_for_role character varying(100) NOT NULL,
                                               visibility character varying(64) NOT NULL,
                                               ctime timestamp without time zone DEFAULT now() NOT NULL,
                                               mtime timestamp without time zone DEFAULT now() NOT NULL
);

CREATE TABLE public.default_layout_entity (
                                              person_id bigint NOT NULL,
                                              entity_id bigint NOT NULL,
                                              create_date timestamp without time zone DEFAULT now() NOT NULL
);

CREATE TABLE public.delete_action (
                                      id bigint NOT NULL,
                                      person_id bigint NOT NULL,
                                      deleted_at timestamp without time zone DEFAULT clock_timestamp(),
                                      transaction_id bigint DEFAULT txid_current(),
                                      descriptor character varying(255) NOT NULL,
                                      archive_data character(1) DEFAULT 't'::bpchar NOT NULL,
                                      item_id character varying(255) NOT NULL
);

CREATE SEQUENCE public.delete_action_seq
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE TABLE public.document_access_token (
                                              token character varying(64) NOT NULL,
                                              token_expiry timestamp without time zone NOT NULL,
                                              document_id character varying(64)
);

CREATE SEQUENCE public.document_files_seq
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE TABLE public.document_version_judo_id (
                                                 version_id bigint NOT NULL,
                                                 judo_id character varying(64) NOT NULL,
                                                 created_at timestamp without time zone DEFAULT clock_timestamp() NOT NULL
);

CREATE SEQUENCE public.document_versions_seq
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE TABLE public.domain_property (
                                        id bigint NOT NULL,
                                        base_property bigint NOT NULL,
                                        domain character varying(255) NOT NULL,
                                        overridden_value text,
                                        last_changed timestamp without time zone DEFAULT now() NOT NULL,
                                        change_user character varying(255),
                                        create_at timestamp without time zone DEFAULT now() NOT NULL
);

CREATE TABLE public.drive_document_item (
                                            id character varying(255) NOT NULL,
                                            name character varying(255) NOT NULL,
                                            parent character varying(255) NOT NULL,
                                            create_date timestamp without time zone DEFAULT now() NOT NULL,
                                            mime_type character varying(255) DEFAULT 'application/octet-stream'::character varying NOT NULL,
                                            file_size bigint DEFAULT 0 NOT NULL,
                                            change_date timestamp without time zone DEFAULT now() NOT NULL,
                                            publicity character varying(255) NOT NULL
);

CREATE TABLE public.email_notification_setting (
                                                   person_id bigint NOT NULL,
                                                   create_date timestamp without time zone DEFAULT clock_timestamp(),
                                                   modify_date timestamp without time zone DEFAULT clock_timestamp(),
                                                   notifications_enabled character(1) DEFAULT 't'::bpchar NOT NULL
);

CREATE SEQUENCE public.entity_seq
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE TABLE public.entity (
                               id bigint DEFAULT nextval('public.entity_seq'::regclass) NOT NULL,
                               name character varying(255) NOT NULL,
                               start_at timestamp without time zone,
                               end_at timestamp without time zone,
                               closing_date timestamp without time zone,
                               timezone character varying(255),
                               description text NOT NULL,
                               location character varying(255),
                               tags text,
                               subdomain character varying(255),
                               subdomain_deactivated character(1) DEFAULT 'f'::bpchar NOT NULL,
                               external_domain character varying(255),
                               logo character varying(255),
                               icon character varying(255),
                               create_date timestamp without time zone DEFAULT now(),
                               modify_date timestamp without time zone DEFAULT now(),
                               use_parent_admin_as_coadmin character(1) DEFAULT 'f'::bpchar NOT NULL,
                               entity_type character varying(255) NOT NULL,
                               slideshow_image character varying(255),
                               participant_limit bigint DEFAULT '-1'::integer NOT NULL,
                               assignee_id bigint,
                               default_entity_tab character varying(256),
                               default_navigation character(1) DEFAULT 't'::bpchar NOT NULL,
                               last_drive_change integer
);

CREATE TABLE public.entity_anchor (
                                      entity_id bigint NOT NULL,
                                      component character varying(255) NOT NULL,
                                      anchor character varying(2048) NOT NULL
);

CREATE TABLE public.entity_component (
                                         entity_id bigint NOT NULL,
                                         component_type character varying(255) NOT NULL,
                                         "position" bigint NOT NULL,
                                         "order" integer NOT NULL,
                                         visible character(1) DEFAULT 't'::bpchar NOT NULL,
                                         open character(1) DEFAULT 't'::bpchar NOT NULL,
                                         customized_title character varying
);

CREATE TABLE public.entity_component_open_for (
                                                  entity_id bigint NOT NULL,
                                                  component_type character varying(255) NOT NULL,
                                                  authority_name character varying(50) NOT NULL
);

CREATE SEQUENCE public.entity_default_design_seq
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE TABLE public.entity_dynamic_attribute (
                                                 entity_id bigint NOT NULL,
                                                 attribute_id character varying(255) NOT NULL,
                                                 attribute_value text,
                                                 "position" integer NOT NULL
);

CREATE TABLE public.entity_internal_invitation (
                                                   person_id bigint NOT NULL,
                                                   entity_id bigint NOT NULL,
                                                   inviter_id bigint NOT NULL,
                                                   invite_message text NOT NULL,
                                                   create_date timestamp without time zone DEFAULT now() NOT NULL
);

CREATE TABLE public.entity_link (
                                    entity_id bigint NOT NULL,
                                    linked_entity_id bigint NOT NULL,
                                    create_date timestamp without time zone DEFAULT now(),
                                    modify_date timestamp without time zone DEFAULT now(),
                                    inherit_admin character(1) DEFAULT 'f'::bpchar NOT NULL,
                                    inherit_member character(1) DEFAULT 'f'::bpchar NOT NULL
);

CREATE TABLE public.entity_member_role (
                                           person_id bigint NOT NULL,
                                           entity_id bigint NOT NULL,
                                           role character varying(50) NOT NULL,
                                           create_date timestamp without time zone DEFAULT now() NOT NULL,
                                           workflow character varying(16) NOT NULL
);

CREATE TABLE public.entity_member_role_name (
                                                name character varying(50) NOT NULL,
                                                create_date timestamp without time zone DEFAULT now() NOT NULL,
                                                workflow character varying(16) NOT NULL
);

CREATE SEQUENCE public.entity_nav_link_seq
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE SEQUENCE public.entity_nav_sublink_seq
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE TABLE public.entity_parents (
                                       created_at timestamp without time zone DEFAULT clock_timestamp(),
                                       entity_id character varying(255) NOT NULL,
                                       parent_id character varying(255) NOT NULL
);

CREATE TABLE public.entity_privacy (
                                       entity_id bigint NOT NULL,
                                       privacy_type character varying(50) NOT NULL,
                                       authority_name character varying(50) NOT NULL,
                                       create_date timestamp without time zone DEFAULT now(),
                                       modify_date timestamp without time zone DEFAULT now()
);

CREATE TABLE public.entity_recommendation (
                                              person_id bigint NOT NULL,
                                              entity_id bigint NOT NULL,
                                              recommender_id bigint NOT NULL,
                                              recommend_message character varying(4000) NOT NULL,
                                              create_date timestamp without time zone DEFAULT now() NOT NULL
);

CREATE TABLE public.entity_status (
                                      id bigint NOT NULL,
                                      entity_type character varying NOT NULL,
                                      status_name character varying NOT NULL,
                                      default_status character(1) DEFAULT 'f'::bpchar NOT NULL,
                                      create_date timestamp without time zone DEFAULT now() NOT NULL,
                                      modify_date timestamp without time zone DEFAULT now(),
                                      published_status character(1) DEFAULT 'f'::bpchar NOT NULL,
                                      archived_status character(1) DEFAULT 'f'::bpchar NOT NULL,
                                      icon_type character varying NOT NULL,
                                      status_order bigint DEFAULT 0 NOT NULL
);

CREATE SEQUENCE public.entity_status_seq
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE TABLE public.entity_tenant (
                                      entity_id bigint NOT NULL,
                                      tenant_id uuid NOT NULL,
                                      ctime timestamp without time zone DEFAULT now() NOT NULL,
                                      mtime timestamp without time zone DEFAULT now() NOT NULL
);

CREATE TABLE public.entity_theme (
                                     theme_id bigint NOT NULL,
                                     entity_id bigint NOT NULL
);

CREATE TABLE public.entity_to_entity_type_privacy (
                                                      entity_id bigint NOT NULL,
                                                      entity_type character varying(255) NOT NULL,
                                                      authority_name character varying(50) NOT NULL,
                                                      create_date timestamp without time zone DEFAULT now(),
                                                      modify_date timestamp without time zone DEFAULT now()
);

CREATE TABLE public.entity_type_app_mapping (
                                                entity_type character varying(32) NOT NULL,
                                                app_id character varying(100) NOT NULL,
                                                ctime timestamp without time zone DEFAULT now() NOT NULL,
                                                mtime timestamp without time zone DEFAULT now() NOT NULL
);

CREATE TABLE public.entity_type_name (
    name character varying(255) NOT NULL
);

CREATE SEQUENCE public.entity_version_seq
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE TABLE public.entity_version (
                                       id bigint DEFAULT nextval('public.entity_version_seq'::regclass) NOT NULL,
                                       entity_id bigint NOT NULL,
                                       person_id bigint NOT NULL,
                                       major_version integer DEFAULT 0 NOT NULL,
                                       minor_version integer DEFAULT 0 NOT NULL,
                                       draft_version integer DEFAULT 0 NOT NULL,
                                       version_type character varying(255) NOT NULL,
                                       target_type character varying(255),
                                       version_status character varying(255) NOT NULL,
                                       release_date timestamp without time zone,
                                       expiration_date timestamp without time zone,
                                       change_message character varying(512),
                                       comment_for_reviewers character varying(255),
                                       read_date timestamp without time zone,
                                       decline_message character varying(255),
                                       create_date timestamp without time zone DEFAULT clock_timestamp() NOT NULL,
                                       modify_date timestamp without time zone DEFAULT clock_timestamp() NOT NULL,
                                       entity_status_id bigint NOT NULL
);

CREATE TABLE public.entity_version_archive_notice (
                                                      id bigint NOT NULL,
                                                      create_date timestamp without time zone DEFAULT now() NOT NULL
);

CREATE TABLE public.entity_version_components (
                                                  entity_version_id bigint NOT NULL,
                                                  multi_wiki_version_id bigint NOT NULL
);

CREATE TABLE public.entity_version_drive_change (
                                                    entity_version_id bigint NOT NULL,
                                                    drive_change_id character varying(64) NOT NULL
);

CREATE TABLE public.entity_version_review (
                                              entity_version_id bigint NOT NULL,
                                              reviewer_id bigint NOT NULL,
                                              inviter_id bigint NOT NULL,
                                              review_status character varying(255) NOT NULL,
                                              review_order bigint NOT NULL
);

CREATE TABLE public.entity_waitlist (
                                        entity_id bigint NOT NULL,
                                        person_id bigint NOT NULL,
                                        waitlist_order bigint NOT NULL
);

CREATE SEQUENCE public.entity_waitlist_order_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

CREATE SEQUENCE public.event_seq
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE SEQUENCE public.event_subtype_seq
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE SEQUENCE public.event_type_seq
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE SEQUENCE public.export_item_event_queue_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

CREATE TABLE public.export_item_event_queue (
                                                id bigint DEFAULT nextval('public.export_item_event_queue_seq'::regclass) NOT NULL,
                                                type character varying(32) NOT NULL,
                                                state character varying(32) NOT NULL,
                                                created_at timestamp without time zone DEFAULT clock_timestamp() NOT NULL,
                                                item_id character varying(255) NOT NULL
);

CREATE TABLE public.external_invitation (
                                            inviter_id numeric(10,0) NOT NULL,
                                            email character varying(100) NOT NULL,
                                            hash character varying(100) NOT NULL,
                                            entity_id bigint
);

CREATE TABLE public.file_type (
                                  id bigint NOT NULL,
                                  extension character varying(32) NOT NULL,
                                  mime_type character varying(128) DEFAULT 'application/octet-stream'::character varying NOT NULL,
                                  description character varying(1024) NOT NULL,
                                  icon_url character varying(1024) NOT NULL
);

CREATE SEQUENCE public.forum_seq
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE SEQUENCE public.forum_topic_seq
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE SEQUENCE public.google_maps_location_seq
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE TABLE public.greeting (
                                 sender_id numeric(10,0) NOT NULL,
                                 recipient_id numeric(10,0) NOT NULL,
                                 create_date timestamp without time zone DEFAULT now() NOT NULL
);

CREATE SEQUENCE public.group_seq
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE SEQUENCE public.group_subtype_seq
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE SEQUENCE public.group_type_seq
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE TABLE public.home_screen_apps (
                                         person_id bigint NOT NULL,
                                         app_id character varying(255) NOT NULL,
                                         app_order integer NOT NULL,
                                         create_date timestamp without time zone DEFAULT now() NOT NULL,
                                         page_number integer DEFAULT 0 NOT NULL
);

CREATE TABLE public.ical_access_token (
                                          token character varying(64) NOT NULL,
                                          person_id bigint NOT NULL,
                                          active character(1) DEFAULT 'f'::bpchar NOT NULL
);

CREATE SEQUENCE public.image_comment_seq
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE SEQUENCE public.image_seq
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE TABLE public.importedentity (
                                       externalid character varying(100) NOT NULL,
                                       internalid bigint
);

CREATE TABLE public.importedperson (
                                       externalid character varying(100) NOT NULL,
                                       internalid bigint,
                                       import_date timestamp without time zone DEFAULT now() NOT NULL
);

CREATE SEQUENCE public.item_event_queue_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

CREATE TABLE public.item_event_queue (
                                         id bigint DEFAULT nextval('public.item_event_queue_seq'::regclass) NOT NULL,
                                         type character varying(32) NOT NULL,
                                         state character varying(32) NOT NULL,
                                         created_at timestamp without time zone DEFAULT clock_timestamp() NOT NULL,
                                         item_id character varying(255) NOT NULL
);

CREATE TABLE public.item_visibility_queue (
                                              id bigint NOT NULL,
                                              item_id bigint NOT NULL,
                                              item_type character varying(32) NOT NULL,
                                              created_at timestamp without time zone DEFAULT clock_timestamp() NOT NULL,
                                              in_progress character(1) DEFAULT 'f'::bpchar NOT NULL,
                                              modified_at timestamp without time zone DEFAULT clock_timestamp() NOT NULL,
                                              lock_id bigint
);

CREATE TABLE public.item_visibility_queue_event (
                                                    item_visibility_queue_id bigint NOT NULL,
                                                    event_type character varying(50) NOT NULL
);

CREATE SEQUENCE public.item_visibility_queue_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

CREATE TABLE public.item_visibility_role (
                                             role_name character varying(15) NOT NULL,
                                             created_at timestamp without time zone DEFAULT clock_timestamp(),
                                             origin_type character varying(255) NOT NULL,
                                             id bigint NOT NULL,
                                             item_id character varying(255) NOT NULL
);

CREATE SEQUENCE public.item_visibility_role_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

CREATE TABLE public.jabber_synced (
                                      person_id bigint NOT NULL,
                                      in_progress character(1) DEFAULT NULL::bpchar NOT NULL
);

CREATE TABLE public.jabber_system_user_password (
                                                    username character varying(255) NOT NULL,
                                                    server character varying(255) NOT NULL,
                                                    password character varying(255) NOT NULL,
                                                    modify_at timestamp without time zone DEFAULT now() NOT NULL,
                                                    valid_until timestamp without time zone
);

CREATE TABLE public.jabber_temp_password (
                                             person_id bigint NOT NULL,
                                             type character varying NOT NULL,
                                             password character varying NOT NULL,
                                             valid_until timestamp without time zone NOT NULL
);

CREATE TABLE public.juco_version (
                                     id bigint NOT NULL,
                                     version character varying(256) NOT NULL,
                                     create_at timestamp without time zone DEFAULT now(),
                                     hostname character varying(256)
);

CREATE SEQUENCE public.juco_version_seq
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE TABLE public.juco_version_update_process (
                                                    version_id bigint NOT NULL,
                                                    process character varying(100) NOT NULL,
                                                    status character varying(50) NOT NULL,
                                                    create_at timestamp without time zone DEFAULT now(),
                                                    modify_at timestamp without time zone DEFAULT now(),
                                                    processor_version integer DEFAULT 0 NOT NULL
);

CREATE TABLE public.language (
                                 id character(2) NOT NULL,
                                 name character varying(50),
                                 translated character(1) DEFAULT 'f'::bpchar
);

CREATE TABLE public.like_table (
                                   liker_id bigint NOT NULL,
                                   created_at timestamp without time zone DEFAULT clock_timestamp(),
                                   modified_at timestamp without time zone DEFAULT clock_timestamp(),
                                   recent character(1) DEFAULT 'f'::bpchar NOT NULL,
                                   item_id character varying(255) NOT NULL
);

CREATE TABLE public.link (
                             id bigint NOT NULL,
                             url character varying NOT NULL,
                             title character varying,
                             create_date timestamp without time zone DEFAULT clock_timestamp() NOT NULL,
                             modify_date timestamp without time zone DEFAULT clock_timestamp()
);

CREATE SEQUENCE public.link_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

CREATE SEQUENCE public.map_document_files_appointments_seq
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE SEQUENCE public.map_document_files_groups_seq
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE SEQUENCE public.map_document_files_networks_seq
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE SEQUENCE public.map_document_files_wiki_groups_seq
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE TABLE public.map_person_person (
                                          person_id numeric(10,0) NOT NULL,
                                          friend_id numeric(10,0) NOT NULL
);

CREATE TABLE public.map_person_person_invite (
                                                 person_id numeric(10,0) NOT NULL,
                                                 friend_id numeric(10,0) NOT NULL,
                                                 message text
);

CREATE SEQUENCE public.mdl_album_seq
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE TABLE public.mdl_album (
                                  id bigint DEFAULT nextval('public.mdl_album_seq'::regclass) NOT NULL,
                                  parent_item_id bigint NOT NULL,
                                  parent_item_type character varying(32) NOT NULL,
                                  module character(1) DEFAULT 't'::bpchar NOT NULL,
                                  owner_id bigint DEFAULT '-1'::integer,
                                  create_date timestamp without time zone DEFAULT now() NOT NULL,
                                  edit_date timestamp without time zone DEFAULT now() NOT NULL,
                                  title character varying(255) NOT NULL,
                                  description text DEFAULT ''::text NOT NULL,
                                  location character varying(80) DEFAULT ''::character varying NOT NULL,
                                  friends_only character(1) DEFAULT 't'::bpchar NOT NULL,
                                  old_album_id bigint
);

CREATE SEQUENCE public.mdl_album_image_seq
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE TABLE public.mdl_album_image (
                                        id bigint DEFAULT nextval('public.mdl_album_image_seq'::regclass) NOT NULL,
                                        title character varying(255) NOT NULL,
                                        album_id bigint NOT NULL,
                                        filename character varying(255) NOT NULL,
                                        owner_id bigint DEFAULT '-1'::integer NOT NULL,
                                        create_date timestamp without time zone DEFAULT now() NOT NULL,
                                        edit_date timestamp without time zone DEFAULT now() NOT NULL,
                                        old_album_image_id bigint
);

CREATE SEQUENCE public.mdl_comments_seq
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE TABLE public.mdl_comment (
                                    id bigint DEFAULT nextval('public.mdl_comments_seq'::regclass) NOT NULL,
                                    author_id bigint DEFAULT '-1'::integer,
                                    text text NOT NULL,
                                    posted_at timestamp without time zone DEFAULT now() NOT NULL,
                                    custom_info character varying(255),
                                    editor_id bigint,
                                    edited_at timestamp without time zone,
                                    parent_item_id character varying(255) NOT NULL,
                                    editor_model text
);

CREATE SEQUENCE public.mdl_documents_seq
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE TABLE public.mdl_document (
                                     id bigint DEFAULT nextval('public.mdl_documents_seq'::regclass) NOT NULL,
                                     parent_item_id bigint NOT NULL,
                                     parent_item_type character varying(32) NOT NULL,
                                     description character varying(1024),
                                     keywords character varying(4096),
                                     download_count bigint DEFAULT 0 NOT NULL,
                                     checked_out_by bigint DEFAULT '-1'::integer,
                                     checked_out_at timestamp without time zone,
                                     status integer DEFAULT 0 NOT NULL,
                                     status_changed_by bigint DEFAULT '-1'::integer,
                                     status_changed_at timestamp without time zone,
                                     modified_at timestamp without time zone NOT NULL,
                                     CONSTRAINT mdl_document_files_positive_download_count CHECK ((download_count >= 0))
);

CREATE SEQUENCE public.mdl_document_versions_seq
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE TABLE public.mdl_document_version (
                                             id bigint DEFAULT nextval('public.mdl_document_versions_seq'::regclass) NOT NULL,
                                             version integer DEFAULT 0 NOT NULL,
                                             filename character varying(256) NOT NULL,
                                             document_id bigint NOT NULL,
                                             url character varying(1024) NOT NULL,
                                             uploaded_by bigint DEFAULT '-1'::integer NOT NULL,
                                             uploaded_at timestamp without time zone DEFAULT now() NOT NULL,
                                             size bigint NOT NULL,
                                             file_type_id bigint NOT NULL,
                                             restored_from integer,
                                             notified_others character(1) DEFAULT 't'::bpchar NOT NULL,
                                             CONSTRAINT mdl_document_version_pos_size CHECK ((size >= 0))
);

CREATE SEQUENCE public.mdl_videolink_seq
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE TABLE public.mdl_video_link (
                                       id bigint DEFAULT nextval('public.mdl_videolink_seq'::regclass) NOT NULL,
                                       parent_item_id bigint NOT NULL,
                                       parent_item_type character varying(32) NOT NULL,
                                       title character varying(255) NOT NULL,
                                       url character varying(255) NOT NULL,
                                       edit_date timestamp without time zone DEFAULT now() NOT NULL,
                                       create_date timestamp without time zone DEFAULT now() NOT NULL,
                                       author_id bigint DEFAULT '-1'::integer NOT NULL,
                                       public character(1) DEFAULT 't'::bpchar NOT NULL,
                                       video_file character varying(256)
);

CREATE SEQUENCE public.messageid_seq
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE TABLE public.migrated_item (
                                      origin_id character varying(255) NOT NULL,
                                      destination_id character varying(255) NOT NULL,
                                      cdate timestamp without time zone DEFAULT now() NOT NULL
);

CREATE TABLE public.migration_mapping (
                                          migration_name character varying(4000) NOT NULL,
                                          migration_key character varying(4000) NOT NULL,
                                          migration_value character varying(4000) NOT NULL
);

CREATE TABLE public.multi_wiki (
                                   id bigint NOT NULL,
                                   entity_id bigint NOT NULL,
                                   create_date timestamp without time zone DEFAULT clock_timestamp() NOT NULL,
                                   modify_date timestamp without time zone DEFAULT clock_timestamp() NOT NULL
);

CREATE SEQUENCE public.multi_wiki_section_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

CREATE TABLE public.multi_wiki_section_visibility (
                                                      version_section_id bigint NOT NULL,
                                                      authority_name character varying(50) NOT NULL,
                                                      create_date timestamp without time zone DEFAULT clock_timestamp() NOT NULL,
                                                      modify_date timestamp without time zone DEFAULT clock_timestamp() NOT NULL
);

CREATE SEQUENCE public.multi_wiki_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

CREATE TABLE public.multi_wiki_version (
                                           id bigint NOT NULL,
                                           wiki_id bigint NOT NULL,
                                           create_date timestamp without time zone DEFAULT clock_timestamp() NOT NULL,
                                           modify_date timestamp without time zone DEFAULT clock_timestamp() NOT NULL
);

CREATE TABLE public.multi_wiki_version_section (
                                                   id bigint NOT NULL,
                                                   version_id bigint NOT NULL,
                                                   headline character varying NOT NULL,
                                                   html text NOT NULL,
                                                   create_date timestamp without time zone DEFAULT clock_timestamp() NOT NULL,
                                                   modify_date timestamp without time zone DEFAULT clock_timestamp() NOT NULL,
                                                   section_id bigint DEFAULT 0 NOT NULL,
                                                   section_order integer DEFAULT 0 NOT NULL
);

CREATE SEQUENCE public.multi_wiki_version_section_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

CREATE SEQUENCE public.multi_wiki_version_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

CREATE TABLE public.native (
                               id bigint NOT NULL,
                               native character varying(255)
);

CREATE SEQUENCE public.native_seq
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE SEQUENCE public.network_navigation_links_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

CREATE SEQUENCE public.network_news_seq
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE SEQUENCE public.newsblog_seq
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE TABLE public.news_blog (
                                  id bigint DEFAULT nextval('public.newsblog_seq'::regclass) NOT NULL,
                                  parent_item_id bigint NOT NULL,
                                  parent_item_type character varying(32) NOT NULL,
                                  headline character varying(255) NOT NULL,
                                  create_date timestamp without time zone DEFAULT now() NOT NULL,
                                  edit_date timestamp without time zone DEFAULT now() NOT NULL,
                                  author_id bigint DEFAULT '-1'::integer NOT NULL,
                                  source_text character varying(255) DEFAULT ''::character varying NOT NULL,
                                  source_href character varying(255) DEFAULT ''::character varying NOT NULL,
                                  keywords character varying(255) DEFAULT ''::character varying NOT NULL,
                                  text text NOT NULL,
                                  image_id bigint,
                                  draft character(1) DEFAULT 'f'::bpchar NOT NULL
);

CREATE TABLE public.newszoom_config (
                                        tenant_id uuid NOT NULL,
                                        cycle_automatically character(1) DEFAULT 't'::bpchar NOT NULL,
                                        open_on character varying(255) DEFAULT 'LOGIN'::character varying NOT NULL,
                                        ctime timestamp without time zone DEFAULT now() NOT NULL,
                                        mtime timestamp without time zone DEFAULT now() NOT NULL
);

CREATE TABLE public.newszoom_section (
                                         id uuid NOT NULL,
                                         tenant_id uuid NOT NULL,
                                         name character varying(100) NOT NULL,
                                         order_index integer NOT NULL,
                                         show_only_pinned character(1) DEFAULT 'f'::bpchar NOT NULL,
                                         ctime timestamp without time zone DEFAULT now() NOT NULL,
                                         mtime timestamp without time zone DEFAULT now() NOT NULL
);

CREATE TABLE public.newszoom_section_channel (
                                                 section_id uuid NOT NULL,
                                                 channel_id character varying(255) NOT NULL,
                                                 ctime timestamp without time zone DEFAULT now() NOT NULL,
                                                 mtime timestamp without time zone DEFAULT now() NOT NULL
);

CREATE TABLE public.notification (
                                     id bigint NOT NULL,
                                     notification_type character varying(255) NOT NULL,
                                     state character varying(255) NOT NULL,
                                     originator_id bigint NOT NULL,
                                     receiver_id bigint NOT NULL,
                                     entity_id bigint,
                                     create_date timestamp without time zone DEFAULT now() NOT NULL,
                                     modify_date timestamp without time zone DEFAULT now(),
                                     entity_type character varying(255) DEFAULT NULL::character varying,
                                     subject_id character varying(255)
);

CREATE TABLE public.notification_attribute (
                                               notification_id bigint NOT NULL,
                                               attribute_key character varying(24) NOT NULL,
                                               attribute_value character varying(5000) NOT NULL
);

CREATE TABLE public.notification_migration_status (
                                                      running character(1) NOT NULL,
                                                      migration_start_at timestamp without time zone DEFAULT now() NOT NULL,
                                                      last_migrated_notification_id bigint
);

CREATE SEQUENCE public.notification_seq
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE TABLE public.oauth_access_token (
                                           token_id character varying(256) NOT NULL,
                                           token text,
                                           authentication_id character varying(256),
                                           user_name character varying(256),
                                           client_id character varying(256),
                                           authentication text,
                                           refresh_token character varying(256)
);

CREATE TABLE public.oauth_client_details (
                                             client_id character varying(256) NOT NULL,
                                             resource_ids character varying(256),
                                             client_secret character varying(4000),
                                             scope character varying(256),
                                             authorized_grant_types character varying(256),
                                             web_server_redirect_uri character varying(256),
                                             authorities character varying(256),
                                             access_token_validity integer,
                                             refresh_token_validity integer,
                                             additional_information character varying(4000),
                                             autoapprove character varying(256)
);

CREATE TABLE public.oauth_code (
                                   code character varying(256),
                                   authentication bytea
);

CREATE TABLE public.oauth_refresh_token (
                                            token_id character varying(256) NOT NULL,
                                            token text,
                                            authentication text
);

CREATE TABLE public.oauth_token (
                                    access_token character varying(256) NOT NULL,
                                    access_token_expiry timestamp without time zone NOT NULL,
                                    refresh_token character varying(256),
                                    refresh_token_expiry timestamp without time zone,
                                    client_id character varying(256) NOT NULL,
                                    scope character varying(256) NOT NULL,
                                    grant_type character varying(256) NOT NULL,
                                    person_id bigint,
                                    create_date timestamp without time zone DEFAULT now() NOT NULL
);

CREATE TABLE public.open_social_app_data (
                                             id numeric(10,0) NOT NULL,
                                             key character varying(100) NOT NULL,
                                             value character varying(10000) NOT NULL,
                                             kadget_id bigint,
                                             person_id bigint NOT NULL
);

CREATE TABLE public.person (
                               id numeric(10,0) NOT NULL,
                               last_login timestamp without time zone DEFAULT now() NOT NULL,
                               show_tempstartpage character(1) DEFAULT 't'::bpchar NOT NULL,
                               account_active character(1) DEFAULT NULL::bpchar NOT NULL,
                               title character varying(255),
                               firstname character varying(50) NOT NULL,
                               surname character varying(100) NOT NULL,
                               email character varying(100),
                               newmail character varying(100),
                               birthday date,
                               password_hash character varying(4000),
                               keywords character varying(255),
                               language_id character(2),
                               password_question character varying(255),
                               password_answer character varying(255),
                               registerdate timestamp without time zone DEFAULT now(),
                               terms_accepted character(1) DEFAULT NULL::bpchar,
                               image character varying(64),
                               blocked character(1) DEFAULT 'f'::bpchar NOT NULL,
                               default_timezone character varying(100),
                               modified_at timestamp without time zone DEFAULT now() NOT NULL,
                               created_at timestamp without time zone DEFAULT now() NOT NULL,
                               birthday_visibility character varying(64) DEFAULT 'NONE'::character varying,
                               hashed_id character varying(32) NOT NULL,
                               additional_title character varying(255),
                               deleted character(1) DEFAULT 'f'::bpchar NOT NULL
);

CREATE TABLE public.person_external_ids (
                                            person_id bigint NOT NULL,
                                            external_person_id character varying(4000) NOT NULL,
                                            external_system_id character varying(50) NOT NULL
);

CREATE TABLE public.person_pending_password_reset (
                                                      person_id bigint NOT NULL,
                                                      expiry_time timestamp without time zone NOT NULL,
                                                      ctime timestamp without time zone DEFAULT clock_timestamp() NOT NULL,
                                                      mtime timestamp without time zone DEFAULT clock_timestamp() NOT NULL
);

CREATE TABLE public.person_role (
                                    role_name character varying(15) NOT NULL,
                                    create_date timestamp without time zone DEFAULT now()
);

CREATE TABLE public.person_tenant (
                                      person_id bigint NOT NULL,
                                      tenant_id uuid NOT NULL,
                                      ctime timestamp without time zone DEFAULT now() NOT NULL,
                                      mtime timestamp without time zone DEFAULT now() NOT NULL
);

CREATE TABLE public.person_to_role (
                                       person_id bigint NOT NULL,
                                       role_name character varying(15) NOT NULL,
                                       create_date timestamp without time zone DEFAULT now(),
                                       modify_date timestamp without time zone DEFAULT now()
);

CREATE SEQUENCE public.personid_seq
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE TABLE public.profile_allowed_component (
                                                  component_type character varying(100) NOT NULL,
                                                  "position" bigint NOT NULL,
                                                  default_order integer NOT NULL,
                                                  default_visibility integer NOT NULL,
                                                  may_modify_visibility character(1) DEFAULT NULL::bpchar NOT NULL,
                                                  create_date timestamp without time zone DEFAULT now(),
                                                  modify_date timestamp without time zone DEFAULT now()
);

CREATE TABLE public.profile_attribute (
                                          id bigint NOT NULL,
                                          person_id bigint NOT NULL,
                                          name character varying(255) NOT NULL,
                                          value text NOT NULL,
                                          value_order integer NOT NULL,
                                          block_order integer NOT NULL,
                                          create_date timestamp without time zone DEFAULT now() NOT NULL
);

CREATE TABLE public.profile_attribute_config (
                                                 config text NOT NULL,
                                                 create_date timestamp without time zone DEFAULT now() NOT NULL
);

CREATE SEQUENCE public.profile_attribute_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

CREATE TABLE public.profile_component (
                                          person_id bigint NOT NULL,
                                          component_type character varying(255) NOT NULL,
                                          "position" bigint NOT NULL,
                                          "order" integer NOT NULL,
                                          create_date timestamp without time zone DEFAULT now(),
                                          modify_date timestamp without time zone DEFAULT now()
);

CREATE TABLE public.properties_tmp (
    property_name character varying(255) NOT NULL
);

CREATE TABLE public.push_target_app (
                                        id bigint NOT NULL,
                                        token character varying(256) NOT NULL,
                                        protocol character varying(100) NOT NULL,
                                        person_id bigint NOT NULL,
                                        create_at timestamp without time zone DEFAULT now(),
                                        modify_at timestamp without time zone DEFAULT now(),
                                        app_name character varying(100) NOT NULL
);

CREATE SEQUENCE public.push_target_app_seq
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE SEQUENCE public.raweditor_seq
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE SEQUENCE public.raweditor_version_seq
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE TABLE public.raweditor_version (
                                          id bigint DEFAULT nextval('public.raweditor_version_seq'::regclass) NOT NULL,
                                          raweditor_id bigint NOT NULL,
                                          parent_item_id bigint NOT NULL,
                                          parent_item_type character varying(32) NOT NULL,
                                          editor_type character varying(255) NOT NULL,
                                          version integer NOT NULL,
                                          text text NOT NULL,
                                          create_date timestamp without time zone DEFAULT now() NOT NULL,
                                          author_id bigint DEFAULT '-1'::integer NOT NULL
);

CREATE TABLE public.scheduled_workstream_entry (
                                                   workstream_entry_id bigint NOT NULL,
                                                   publish_date timestamp without time zone NOT NULL,
                                                   ctime timestamp without time zone DEFAULT clock_timestamp() NOT NULL,
                                                   mtime timestamp without time zone DEFAULT clock_timestamp() NOT NULL
);

CREATE TABLE public.search_filter (
                                      id bigint NOT NULL,
                                      person_id bigint NOT NULL,
                                      name character varying(255) NOT NULL,
                                      filter character varying(2000) NOT NULL,
                                      create_date timestamp without time zone NOT NULL
);

CREATE SEQUENCE public.search_filter_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

CREATE SEQUENCE public.search_index_queue_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

CREATE TABLE public.search_index_queue (
                                           id bigint DEFAULT nextval('public.search_index_queue_seq'::regclass) NOT NULL,
                                           item_type character varying(32) NOT NULL,
                                           created_at timestamp without time zone DEFAULT clock_timestamp() NOT NULL,
                                           in_progress character(1) DEFAULT 'f'::bpchar NOT NULL,
                                           modified_at timestamp without time zone DEFAULT now(),
                                           index_type character varying(100) DEFAULT 'ALL'::character varying NOT NULL,
                                           item_id character varying(255) NOT NULL
);

CREATE TABLE public.secret (
                               secret_key character varying NOT NULL,
                               secret_value character varying NOT NULL
);

CREATE SEQUENCE public.seq_base_property
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE SEQUENCE public.seq_cities
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE SEQUENCE public.seq_domain_property
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE SEQUENCE public.seq_forum_article
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE SEQUENCE public.seq_kadgets
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE SEQUENCE public.seq_map_kadgets_networks
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE SEQUENCE public.seq_network
    START WITH 1000
    INCREMENT BY 1
    MINVALUE 1000
    NO MAXVALUE
    CACHE 1;

CREATE SEQUENCE public.seq_open_social_app_data
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

CREATE SEQUENCE public.seq_status
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE SEQUENCE public.seq_version
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE TABLE public.sh_conversation (
                                        id bigint NOT NULL,
                                        type character varying(255),
                                        create_date timestamp without time zone,
                                        title character varying(1024),
                                        last_message_date timestamp without time zone,
                                        delete_action_id bigint,
                                        creation_id character varying(100),
                                        last_message_update timestamp without time zone
);

CREATE TABLE public.sh_conversation_attachment (
                                                   id bigint NOT NULL,
                                                   message_id bigint,
                                                   document_id character varying(255),
                                                   document_version_id character varying(255),
                                                   filename character varying(255),
                                                   file_size bigint,
                                                   content_type character varying(255),
                                                   version character varying(255),
                                                   attachment_type character varying(255),
                                                   image_height integer,
                                                   image_width integer,
                                                   create_date timestamp without time zone,
                                                   delete_action_id bigint
);

CREATE TABLE public.sh_conversation_message (
                                                id bigint NOT NULL,
                                                conversation_id bigint,
                                                from_person_id bigint,
                                                create_date timestamp without time zone,
                                                message text,
                                                message_type character varying(32),
                                                delete_action_id bigint,
                                                creation_id character varying(100),
                                                modify_date timestamp without time zone,
                                                delete_date timestamp without time zone
);

CREATE TABLE public.sh_conversation_participant (
                                                    conversation_id bigint,
                                                    participant_id bigint,
                                                    create_date timestamp without time zone,
                                                    modify_date timestamp without time zone,
                                                    read_date timestamp without time zone,
                                                    deleted character(1),
                                                    last_mail_sent timestamp without time zone,
                                                    deleted_with_conversation character(1),
                                                    delete_action_id bigint
);

CREATE TABLE public.sh_conversation_settings (
                                                 conversation_id bigint NOT NULL,
                                                 participant_id bigint NOT NULL,
                                                 muted character(1),
                                                 create_date timestamp without time zone,
                                                 modify_date timestamp without time zone,
                                                 hidden character(1),
                                                 delete_action_id bigint
);

CREATE TABLE public.sh_default_album (
                                         item_id bigint,
                                         item_type character varying(32),
                                         album_id bigint,
                                         delete_action_id bigint
);

CREATE TABLE public.sh_email_notification_setting (
                                                      person_id bigint NOT NULL,
                                                      create_date timestamp without time zone,
                                                      modify_date timestamp without time zone,
                                                      notifications_enabled character(1),
                                                      delete_action_id bigint
);

CREATE TABLE public.sh_entity (
                                  id bigint NOT NULL,
                                  name character varying(255),
                                  start_at timestamp without time zone,
                                  end_at timestamp without time zone,
                                  closing_date timestamp without time zone,
                                  timezone character varying(255),
                                  description text,
                                  location character varying(255),
                                  tags text,
                                  subdomain character varying(255),
                                  subdomain_deactivated character(1),
                                  external_domain character varying(255),
                                  logo character varying(255),
                                  icon character varying(255),
                                  create_date timestamp without time zone,
                                  modify_date timestamp without time zone,
                                  use_parent_admin_as_coadmin character(1),
                                  entity_type character varying(255),
                                  delete_action_id bigint,
                                  slideshow_image character varying(255),
                                  participant_limit bigint,
                                  assignee_id bigint,
                                  default_entity_tab character varying(256),
                                  default_navigation character(1),
                                  last_drive_change integer
);

CREATE TABLE public.sh_entity_component (
                                            entity_id bigint NOT NULL,
                                            component_type character varying(255) NOT NULL,
                                            "position" bigint,
                                            "order" integer,
                                            visible character(1),
                                            open character(1),
                                            customized_title character varying,
                                            delete_action_id bigint
);

CREATE TABLE public.sh_entity_component_open_for (
                                                     entity_id bigint NOT NULL,
                                                     component_type character varying(255) NOT NULL,
                                                     authority_name character varying(50) NOT NULL,
                                                     delete_action_id bigint
);

CREATE TABLE public.sh_entity_dynamic_attribute (
                                                    entity_id bigint,
                                                    attribute_id character varying(255),
                                                    attribute_value text,
                                                    "position" integer,
                                                    delete_action_id bigint
);

CREATE TABLE public.sh_entity_internal_invitation (
                                                      person_id bigint NOT NULL,
                                                      entity_id bigint NOT NULL,
                                                      inviter_id bigint,
                                                      invite_message text,
                                                      create_date timestamp without time zone,
                                                      delete_action_id bigint NOT NULL
);

CREATE TABLE public.sh_entity_link (
                                       entity_id bigint NOT NULL,
                                       linked_entity_id bigint NOT NULL,
                                       create_date timestamp without time zone,
                                       modify_date timestamp without time zone,
                                       delete_action_id bigint,
                                       inherit_admin character(1),
                                       inherit_member character(1)
);

CREATE TABLE public.sh_entity_member_role (
                                              person_id bigint NOT NULL,
                                              entity_id bigint NOT NULL,
                                              role character varying(50) NOT NULL,
                                              create_date timestamp without time zone,
                                              delete_action_id bigint NOT NULL,
                                              workflow character varying(16)
);

CREATE TABLE public.sh_entity_privacy (
                                          entity_id bigint,
                                          privacy_type character varying(50),
                                          authority_name character varying(50),
                                          create_date timestamp without time zone,
                                          modify_date timestamp without time zone,
                                          delete_action_id bigint
);

CREATE TABLE public.sh_entity_recommendation (
                                                 person_id bigint NOT NULL,
                                                 entity_id bigint NOT NULL,
                                                 recommender_id bigint,
                                                 recommend_message character varying(4000),
                                                 create_date timestamp without time zone,
                                                 delete_action_id bigint
);

CREATE TABLE public.sh_entity_tenant (
                                         entity_id bigint NOT NULL,
                                         tenant_id uuid NOT NULL,
                                         ctime timestamp without time zone DEFAULT now() NOT NULL,
                                         mtime timestamp without time zone DEFAULT now() NOT NULL
);

CREATE TABLE public.sh_entity_to_entity_type_priva (
                                                       entity_id bigint NOT NULL,
                                                       entity_type character varying(255) NOT NULL,
                                                       authority_name character varying(50) NOT NULL,
                                                       create_date timestamp without time zone,
                                                       modify_date timestamp without time zone,
                                                       delete_action_id bigint
);

CREATE TABLE public.sh_entity_version (
                                          id bigint NOT NULL,
                                          entity_id bigint,
                                          person_id bigint,
                                          major_version integer,
                                          minor_version integer,
                                          version_type character varying(255),
                                          version_status character varying(255),
                                          release_date timestamp without time zone,
                                          expiration_date timestamp without time zone,
                                          draft_version bigint,
                                          change_message character varying(512),
                                          comment_for_reviewers character varying(255),
                                          target_type character varying(255),
                                          read_date timestamp without time zone,
                                          decline_message character varying(255),
                                          create_date timestamp without time zone,
                                          modify_date timestamp without time zone,
                                          delete_action_id bigint,
                                          entity_status_id bigint
);

CREATE TABLE public.sh_entity_version_components (
                                                     entity_version_id bigint,
                                                     multi_wiki_version_id bigint,
                                                     delete_action_id bigint
);

CREATE TABLE public.sh_entity_version_review (
                                                 entity_version_id bigint NOT NULL,
                                                 reviewer_id bigint NOT NULL,
                                                 inviter_id bigint,
                                                 review_status character varying(255),
                                                 review_order bigint,
                                                 delete_action_id bigint
);

CREATE TABLE public.sh_entity_waitlist (
                                           entity_id bigint,
                                           person_id bigint,
                                           waitlist_order bigint,
                                           delete_action_id bigint
);

CREATE TABLE public.sh_external_invitation (
                                               inviter_id numeric(10,0),
                                               email character varying(100),
                                               hash character varying(100) NOT NULL,
                                               entity_id bigint,
                                               delete_action_id bigint
);

CREATE TABLE public.sh_like_table (
                                      liker_id bigint NOT NULL,
                                      created_at timestamp without time zone,
                                      modified_at timestamp without time zone,
                                      recent character(1),
                                      delete_action_id bigint,
                                      item_id character varying(255)
);

CREATE TABLE public.sh_map_person_person (
                                             person_id numeric(10,0) NOT NULL,
                                             friend_id numeric(10,0) NOT NULL,
                                             delete_action_id bigint
);

CREATE TABLE public.sh_map_person_person_invite (
                                                    person_id numeric(10,0) NOT NULL,
                                                    friend_id numeric(10,0) NOT NULL,
                                                    message text,
                                                    delete_action_id bigint
);

CREATE TABLE public.sh_mdl_album (
                                     id bigint NOT NULL,
                                     parent_item_id bigint,
                                     parent_item_type character varying(32),
                                     module character(1),
                                     owner_id bigint,
                                     create_date timestamp without time zone,
                                     edit_date timestamp without time zone,
                                     title character varying(255),
                                     description text,
                                     location character varying(80),
                                     friends_only character(1),
                                     old_album_id bigint,
                                     delete_action_id bigint
);

CREATE TABLE public.sh_mdl_album_image (
                                           id bigint NOT NULL,
                                           title character varying(255),
                                           album_id bigint,
                                           filename character varying(255),
                                           owner_id bigint,
                                           create_date timestamp without time zone,
                                           edit_date timestamp without time zone,
                                           old_album_image_id bigint,
                                           delete_action_id bigint
);

CREATE TABLE public.sh_mdl_comment (
                                       id bigint NOT NULL,
                                       author_id bigint,
                                       text text,
                                       posted_at timestamp without time zone,
                                       custom_info character varying(255),
                                       delete_action_id bigint,
                                       editor_id bigint,
                                       edited_at timestamp without time zone,
                                       parent_item_id character varying(255),
                                       editor_model text
);

CREATE TABLE public.sh_mdl_document (
                                        id bigint NOT NULL,
                                        parent_item_id bigint,
                                        parent_item_type character varying(32),
                                        description character varying(1024),
                                        keywords character varying(4096),
                                        download_count bigint,
                                        checked_out_by bigint,
                                        checked_out_at timestamp without time zone,
                                        status integer,
                                        status_changed_by bigint,
                                        status_changed_at timestamp without time zone,
                                        delete_action_id bigint,
                                        modified_at timestamp without time zone
);

CREATE TABLE public.sh_mdl_document_version (
                                                id bigint NOT NULL,
                                                version integer,
                                                filename character varying(256),
                                                document_id bigint,
                                                url character varying(1024),
                                                uploaded_by bigint,
                                                uploaded_at timestamp without time zone,
                                                size bigint,
                                                file_type_id bigint,
                                                restored_from integer,
                                                delete_action_id bigint,
                                                notified_others character(1) DEFAULT 't'::bpchar NOT NULL
);

CREATE TABLE public.sh_mdl_video_link (
                                          id bigint NOT NULL,
                                          parent_item_id bigint,
                                          parent_item_type character varying(32),
                                          title character varying(255),
                                          url character varying(255),
                                          edit_date timestamp without time zone,
                                          create_date timestamp without time zone,
                                          author_id bigint,
                                          public character(1),
                                          video_file character varying(256),
                                          delete_action_id bigint
);

CREATE TABLE public.sh_multi_wiki (
                                      id bigint NOT NULL,
                                      entity_id bigint,
                                      create_date timestamp without time zone,
                                      modify_date timestamp without time zone,
                                      delete_action_id bigint
);

CREATE TABLE public.sh_multi_wiki_section_visibili (
                                                       version_section_id bigint NOT NULL,
                                                       authority_name character varying(50) NOT NULL,
                                                       create_date timestamp without time zone,
                                                       modify_date timestamp without time zone,
                                                       delete_action_id bigint
);

CREATE TABLE public.sh_multi_wiki_version (
                                              id bigint NOT NULL,
                                              wiki_id bigint,
                                              create_date timestamp without time zone,
                                              modify_date timestamp without time zone,
                                              delete_action_id bigint
);

CREATE TABLE public.sh_multi_wiki_version_section (
                                                      id bigint NOT NULL,
                                                      version_id bigint,
                                                      headline character varying,
                                                      html text,
                                                      create_date timestamp without time zone,
                                                      modify_date timestamp without time zone,
                                                      section_id bigint,
                                                      section_order integer,
                                                      delete_action_id bigint
);

CREATE TABLE public.sh_news_blog (
                                     id bigint NOT NULL,
                                     parent_item_id bigint,
                                     parent_item_type character varying(32),
                                     headline character varying(255),
                                     create_date timestamp without time zone,
                                     edit_date timestamp without time zone,
                                     author_id bigint,
                                     source_text character varying(255),
                                     source_href character varying(255),
                                     keywords character varying(255),
                                     text text,
                                     image_id bigint,
                                     draft character(1),
                                     delete_action_id bigint
);

CREATE TABLE public.sh_person (
                                  id numeric(10,0) NOT NULL,
                                  last_login timestamp without time zone,
                                  show_tempstartpage character(1),
                                  account_active character(1),
                                  title character varying(255),
                                  firstname character varying(50),
                                  surname character varying(100),
                                  email character varying(100),
                                  newmail character varying(100),
                                  birthday date,
                                  password_hash character varying(4000),
                                  keywords character varying(255),
                                  language_id character(2),
                                  password_question character varying(255),
                                  password_answer character varying(255),
                                  registerdate timestamp without time zone,
                                  terms_accepted character(1),
                                  image character varying(64),
                                  blocked character(1),
                                  default_timezone character varying(100),
                                  modified_at timestamp without time zone,
                                  created_at timestamp without time zone,
                                  birthday_visibility character varying(64),
                                  hashed_id character varying(32),
                                  additional_title character varying(255),
                                  deleted character(1),
                                  delete_action_id bigint
);

CREATE TABLE public.sh_person_external_ids (
                                               person_id bigint NOT NULL,
                                               external_person_id character varying(4000),
                                               external_system_id character varying(50) NOT NULL,
                                               delete_action_id bigint
);

CREATE TABLE public.sh_person_tenant (
                                         person_id bigint NOT NULL,
                                         tenant_id uuid NOT NULL,
                                         ctime timestamp without time zone DEFAULT now() NOT NULL,
                                         mtime timestamp without time zone DEFAULT now() NOT NULL
);

CREATE TABLE public.sh_person_to_role (
                                          person_id bigint NOT NULL,
                                          role_name character varying(15) NOT NULL,
                                          create_date timestamp without time zone,
                                          modify_date timestamp without time zone,
                                          delete_action_id bigint
);

CREATE TABLE public.sh_profile_attribute (
                                             id bigint,
                                             person_id bigint,
                                             name character varying(255),
                                             value text,
                                             value_order integer,
                                             block_order integer,
                                             delete_action_id bigint,
                                             create_date timestamp without time zone
);

CREATE TABLE public.sh_profile_component (
                                             person_id bigint NOT NULL,
                                             component_type character varying(255) NOT NULL,
                                             "position" bigint,
                                             "order" integer,
                                             create_date timestamp without time zone,
                                             modify_date timestamp without time zone,
                                             delete_action_id bigint
);

CREATE TABLE public.sh_raweditor_version (
                                             id bigint NOT NULL,
                                             raweditor_id bigint,
                                             parent_item_id bigint,
                                             parent_item_type character varying(32),
                                             editor_type character varying(255),
                                             version integer,
                                             text text,
                                             create_date timestamp without time zone,
                                             author_id bigint,
                                             delete_action_id bigint
);

CREATE TABLE public.sh_subscriptions (
                                         subscriber_id bigint,
                                         subscription_type character varying(32),
                                         subscription_state character varying(32),
                                         created_at timestamp without time zone,
                                         modified_at timestamp without time zone,
                                         delete_action_id bigint,
                                         item_id character varying(255)
);

CREATE TABLE public.sh_survey (
                                  id bigint NOT NULL,
                                  parent_item_id bigint,
                                  parent_item_type character varying(32),
                                  title character varying(255),
                                  expired_date timestamp without time zone,
                                  owner_id bigint,
                                  result_visibilty integer,
                                  anonym character(1),
                                  participant_may_modify character(1),
                                  auto_delete_answers character(1),
                                  create_date timestamp without time zone,
                                  modify_date timestamp without time zone,
                                  delete_action_id bigint,
                                  redirect_on_new_member character(1)
);

CREATE TABLE public.sh_survey_participant_answer (
                                                     participant_id bigint,
                                                     survey_question_id bigint,
                                                     survey_predefined_answer_id bigint,
                                                     freetext_answer text,
                                                     create_date timestamp without time zone,
                                                     modify_date timestamp without time zone,
                                                     delete_action_id bigint
);

CREATE TABLE public.sh_survey_predefined_answer (
                                                    id bigint NOT NULL,
                                                    survey_question_id bigint,
                                                    answer text,
                                                    "order" integer,
                                                    create_date timestamp without time zone,
                                                    modify_date timestamp without time zone,
                                                    delete_action_id bigint
);

CREATE TABLE public.sh_survey_question (
                                           id bigint NOT NULL,
                                           survey_id bigint,
                                           question text,
                                           "order" integer,
                                           freetext_answer character(1),
                                           min_answer integer,
                                           max_answer integer,
                                           create_date timestamp without time zone,
                                           modify_date timestamp without time zone,
                                           delete_action_id bigint
);

CREATE TABLE public.sh_task_list (
                                     id uuid,
                                     name character varying(255),
                                     create_date timestamp without time zone,
                                     modify_date timestamp without time zone,
                                     delete_action_id bigint
);

CREATE TABLE public.sh_task_list_permissions (
                                                 task_list_id uuid,
                                                 role character varying(64),
                                                 task_right character varying(64),
                                                 ctime timestamp without time zone,
                                                 mtime timestamp without time zone,
                                                 delete_action_id bigint
);

CREATE TABLE public.sh_tasks (
                                 id uuid,
                                 name character varying(2048),
                                 description text,
                                 owner_id bigint,
                                 assignee_id bigint,
                                 status character varying(255),
                                 create_date timestamp without time zone,
                                 modify_date timestamp without time zone,
                                 parent_task_id uuid,
                                 delete_action_id bigint,
                                 attachment_name character varying(255),
                                 attachment_url character varying(255),
                                 task_list_id uuid,
                                 due_date timestamp without time zone
);

CREATE TABLE public.sh_user_setting (
                                        person_id numeric(10,0) NOT NULL,
                                        key character varying(30) NOT NULL,
                                        value character varying(4000),
                                        create_date timestamp without time zone,
                                        modify_date timestamp without time zone,
                                        delete_action_id bigint
);

CREATE TABLE public.sh_wiki_editor (
                                       id bigint NOT NULL,
                                       parent_item_id bigint,
                                       parent_item_type character varying(32),
                                       old_group_id bigint,
                                       delete_action_id bigint
);

CREATE TABLE public.sh_wiki_editor_backup (
                                              wiki_editor_id bigint NOT NULL,
                                              wiki_text text,
                                              author_id bigint,
                                              create_date timestamp without time zone,
                                              delete_action_id bigint
);

CREATE TABLE public.sh_wiki_editor_version (
                                               id bigint NOT NULL,
                                               wiki_editor_id bigint,
                                               wiki_text text,
                                               version integer,
                                               author_id bigint,
                                               create_date timestamp without time zone,
                                               old_page_id bigint,
                                               delete_action_id bigint,
                                               restored_version integer
);

CREATE TABLE public.sh_workstream_attachment (
                                                 entry_id bigint,
                                                 delete_action_id bigint,
                                                 "position" integer,
                                                 item_id character varying(255) NOT NULL
);

CREATE TABLE public.sh_workstream_entry (
                                            id bigint NOT NULL,
                                            type character varying(32),
                                            created_at timestamp without time zone,
                                            modified_at timestamp without time zone,
                                            delete_action_id bigint,
                                            origin_item_id character varying(255),
                                            subject_item_id character varying(255),
                                            author_id character varying(255),
                                            pinned character(1),
                                            pin_author bigint,
                                            pin_date timestamp without time zone,
                                            published character(1) NOT NULL,
                                            published_at timestamp without time zone
);

CREATE TABLE public.sh_workstream_entry_attribute (
                                                      entry_id bigint,
                                                      key character varying(24),
                                                      value text,
                                                      delete_action_id bigint
);

CREATE TABLE public.shadow_table (
                                     table_name character varying NOT NULL,
                                     shadow_table_name character varying NOT NULL,
                                     delete_action_mandatory character(1) NOT NULL
);

CREATE TABLE public.subscriptions (
                                      subscriber_id bigint NOT NULL,
                                      subscription_type character varying(32) NOT NULL,
                                      subscription_state character varying(32) NOT NULL,
                                      created_at timestamp without time zone NOT NULL,
                                      modified_at timestamp without time zone NOT NULL,
                                      item_id character varying(255) NOT NULL
);

CREATE SEQUENCE public.survey_seq
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE TABLE public.survey (
                               id bigint DEFAULT nextval('public.survey_seq'::regclass) NOT NULL,
                               parent_item_id bigint NOT NULL,
                               parent_item_type character varying(32) NOT NULL,
                               title character varying(255) NOT NULL,
                               expired_date timestamp without time zone,
                               owner_id bigint DEFAULT '-1'::integer NOT NULL,
                               result_visibilty integer NOT NULL,
                               anonym character(1) DEFAULT NULL::bpchar NOT NULL,
                               participant_may_modify character(1) DEFAULT NULL::bpchar NOT NULL,
                               auto_delete_answers character(1) DEFAULT NULL::bpchar NOT NULL,
                               create_date timestamp without time zone DEFAULT now(),
                               modify_date timestamp without time zone DEFAULT now(),
                               redirect_on_new_member character(1) DEFAULT 'f'::bpchar NOT NULL
);

CREATE TABLE public.survey_participant_answer (
                                                  participant_id bigint DEFAULT '-1'::integer,
                                                  survey_question_id bigint NOT NULL,
                                                  survey_predefined_answer_id bigint,
                                                  freetext_answer text,
                                                  create_date timestamp without time zone DEFAULT now(),
                                                  modify_date timestamp without time zone DEFAULT now()
);

CREATE SEQUENCE public.survey_predefined_multiple_choice_answer_seq
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE TABLE public.survey_predefined_answer (
                                                 id bigint DEFAULT nextval('public.survey_predefined_multiple_choice_answer_seq'::regclass) NOT NULL,
                                                 survey_question_id bigint NOT NULL,
                                                 answer text NOT NULL,
                                                 "order" integer,
                                                 create_date timestamp without time zone DEFAULT now(),
                                                 modify_date timestamp without time zone DEFAULT now()
);

CREATE SEQUENCE public.survey_question_seq
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE TABLE public.survey_question (
                                        id bigint DEFAULT nextval('public.survey_question_seq'::regclass) NOT NULL,
                                        survey_id bigint NOT NULL,
                                        question text NOT NULL,
                                        "order" integer,
                                        freetext_answer character(1) DEFAULT NULL::bpchar NOT NULL,
                                        min_answer integer,
                                        max_answer integer,
                                        create_date timestamp without time zone DEFAULT now(),
                                        modify_date timestamp without time zone DEFAULT now()
);

CREATE TABLE public.sync_app (
                                 id character varying(36) NOT NULL,
                                 token character varying(36) NOT NULL,
                                 person_id bigint,
                                 client_version character varying(32),
                                 client_user character varying(255) NOT NULL,
                                 client_system character varying(255) NOT NULL,
                                 create_date timestamp without time zone NOT NULL,
                                 modify_date timestamp without time zone NOT NULL
);

CREATE SEQUENCE public.systemmessageid_seq
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE TABLE public.task_filter_order (
                                          filter_id character varying(255) NOT NULL,
                                          task_id uuid NOT NULL,
                                          "position" bigint NOT NULL
);

CREATE TABLE public.task_list (
                                  id uuid NOT NULL,
                                  name character varying(255) NOT NULL,
                                  create_date timestamp without time zone DEFAULT clock_timestamp() NOT NULL,
                                  modify_date timestamp without time zone DEFAULT clock_timestamp() NOT NULL
);

CREATE TABLE public.task_list_permissions (
                                              task_list_id uuid NOT NULL,
                                              role character varying(64) NOT NULL,
                                              task_right character varying(64) NOT NULL,
                                              ctime timestamp without time zone DEFAULT clock_timestamp() NOT NULL,
                                              mtime timestamp without time zone DEFAULT clock_timestamp() NOT NULL
);

CREATE TABLE public.tasks (
                              id uuid NOT NULL,
                              name character varying(2048) NOT NULL,
                              description text NOT NULL,
                              owner_id bigint NOT NULL,
                              assignee_id bigint,
                              status character varying(255) NOT NULL,
                              create_date timestamp without time zone NOT NULL,
                              modify_date timestamp without time zone NOT NULL,
                              parent_task_id uuid,
                              attachment_name character varying(255) DEFAULT ''::character varying NOT NULL,
                              attachment_url character varying(255) DEFAULT ''::character varying NOT NULL,
                              task_list_id uuid,
                              due_date timestamp without time zone,
                              CONSTRAINT tasks_unique_parent CHECK (((task_list_id IS NULL) OR (parent_task_id IS NULL)))
);

CREATE TABLE public.tenant (
                               id uuid NOT NULL,
                               default_tenant character(1) DEFAULT 'f'::bpchar NOT NULL,
                               ctime timestamp without time zone DEFAULT now() NOT NULL,
                               mtime timestamp without time zone DEFAULT now() NOT NULL,
                               name character varying(1024),
                               logo_id character varying(1024)
);

CREATE TABLE public.tenant_app_visibility (
                                              app_id character varying(32) NOT NULL,
                                              tenant_id uuid NOT NULL,
                                              visible_for_role character varying(100) NOT NULL,
                                              visibility character varying(64) NOT NULL,
                                              ctime timestamp without time zone DEFAULT now() NOT NULL,
                                              mtime timestamp without time zone DEFAULT now() NOT NULL
);

CREATE TABLE public.tenant_theme (
                                     tenant_id uuid NOT NULL,
                                     theme_attribute character varying(255) NOT NULL,
                                     theme_value character varying(7) NOT NULL
);

CREATE TABLE public.theme (
                              id bigint NOT NULL,
                              name character varying,
                              shared character(1) DEFAULT 'f'::bpchar NOT NULL,
                              default_theme character(1) DEFAULT 'f'::bpchar NOT NULL,
                              create_date timestamp without time zone DEFAULT now() NOT NULL,
                              modify_date timestamp without time zone DEFAULT now()
);

CREATE TABLE public.theme_attribute (
                                        theme_id bigint NOT NULL,
                                        attribute_type character varying NOT NULL,
                                        attribute_value character varying
);

CREATE SEQUENCE public.theme_seq
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE TABLE public.translation (
                                    language_id character(2) NOT NULL,
                                    native_id bigint NOT NULL,
                                    translation character varying(255)
);

CREATE TABLE public.user_setting (
                                     person_id numeric(10,0) NOT NULL,
                                     key character varying(30) NOT NULL,
                                     value character varying(4000) NOT NULL,
                                     create_date timestamp without time zone DEFAULT now(),
                                     modify_date timestamp without time zone DEFAULT now()
);

CREATE TABLE public.version (
                                id bigint NOT NULL,
                                release character varying(255) NOT NULL,
                                name character varying(255),
                                type character varying(255),
                                description text,
                                date timestamp without time zone
);

CREATE SEQUENCE public.videolink_comment_seq
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE SEQUENCE public.videolink_seq
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE TABLE public.view_count (
                                   item_id character varying(255) NOT NULL,
                                   counter integer DEFAULT 0 NOT NULL
);

CREATE SEQUENCE public.wallmessageid_seq
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE SEQUENCE public.wiki_editor_seq
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE TABLE public.wiki_editor (
                                    id bigint DEFAULT nextval('public.wiki_editor_seq'::regclass) NOT NULL,
                                    parent_item_id bigint NOT NULL,
                                    parent_item_type character varying(32) NOT NULL,
                                    old_group_id bigint
);

CREATE TABLE public.wiki_editor_backup (
                                           wiki_editor_id bigint NOT NULL,
                                           wiki_text text NOT NULL,
                                           author_id bigint NOT NULL,
                                           create_date timestamp without time zone DEFAULT now() NOT NULL
);

CREATE SEQUENCE public.wiki_editor_version_seq
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE TABLE public.wiki_editor_version (
                                            id bigint DEFAULT nextval('public.wiki_editor_version_seq'::regclass) NOT NULL,
                                            wiki_editor_id bigint NOT NULL,
                                            wiki_text text NOT NULL,
                                            version integer DEFAULT 0 NOT NULL,
                                            author_id bigint DEFAULT '-1'::integer NOT NULL,
                                            create_date timestamp without time zone DEFAULT now() NOT NULL,
                                            old_page_id bigint,
                                            restored_version integer
);

CREATE SEQUENCE public.wiki_group_seq
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE SEQUENCE public.wiki_page_backup_seq
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE SEQUENCE public.wiki_page_seq
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE SEQUENCE public.wiki_page_version_seq
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE TABLE public.workstream_attachment (
                                              entry_id bigint NOT NULL,
                                              "position" integer DEFAULT 0 NOT NULL,
                                              item_id character varying(255) NOT NULL
);

CREATE SEQUENCE public.workstream_entries_seq
    START WITH 0
    INCREMENT BY 1
    MINVALUE 0
    NO MAXVALUE
    CACHE 1;

CREATE TABLE public.workstream_entry (
                                         id bigint DEFAULT nextval('public.workstream_entries_seq'::regclass) NOT NULL,
                                         type character varying(32) NOT NULL,
                                         created_at timestamp without time zone NOT NULL,
                                         modified_at timestamp without time zone NOT NULL,
                                         origin_item_id character varying(255) NOT NULL,
                                         subject_item_id character varying(255) NOT NULL,
                                         author_id character varying(255) NOT NULL,
                                         pinned character(1) DEFAULT 'f'::bpchar NOT NULL,
                                         pin_author bigint,
                                         pin_date timestamp without time zone,
                                         published character(1) NOT NULL,
                                         published_at timestamp without time zone
);

CREATE TABLE public.workstream_entry_attribute (
                                                   entry_id bigint NOT NULL,
                                                   key character varying(24) NOT NULL,
                                                   value text NOT NULL
);

CREATE TABLE public.workstream_pinned_hide_date (
                                                    person_id bigint NOT NULL,
                                                    origin_item_id character varying(255) NOT NULL,
                                                    hide_date timestamp without time zone NOT NULL
);

CREATE TABLE public.workstream_read (
                                        person_id bigint NOT NULL,
                                        read_date timestamp without time zone NOT NULL,
                                        item_id character varying(255) NOT NULL
);

ALTER TABLE ONLY public.account_failed_logins
    ADD CONSTRAINT account_failed_logins_pkey PRIMARY KEY (person_id);

ALTER TABLE ONLY public.account_lock
    ADD CONSTRAINT account_lock_pkey PRIMARY KEY (person_id);

ALTER TABLE ONLY public.additional_app_info
    ADD CONSTRAINT additional_app_info_pkey PRIMARY KEY (app_id);

ALTER TABLE ONLY public.another_juco_version_table
    ADD CONSTRAINT another_juco_version_table_pkey PRIMARY KEY (id);

ALTER TABLE ONLY public.app
    ADD CONSTRAINT app_pkey PRIMARY KEY (id);

ALTER TABLE ONLY public.audit_log_param
    ADD CONSTRAINT audit_log_param_pk PRIMARY KEY (audit_id, "position");

ALTER TABLE ONLY public.audit_log
    ADD CONSTRAINT audit_log_pk PRIMARY KEY (id);

ALTER TABLE ONLY public.base_property
    ADD CONSTRAINT base_property_pkey PRIMARY KEY (id);

ALTER TABLE ONLY public.base_property
    ADD CONSTRAINT base_property_property_name_key UNIQUE (property_name, container_name);

ALTER TABLE ONLY public.chat_search_queue
    ADD CONSTRAINT chat_search_queue_pk PRIMARY KEY (id);

ALTER TABLE ONLY public.conversation_attachment
    ADD CONSTRAINT conv_attachment_pk PRIMARY KEY (id);

ALTER TABLE ONLY public.conversation_message
    ADD CONSTRAINT conv_message_pk PRIMARY KEY (id);

ALTER TABLE ONLY public.conversation_participant
    ADD CONSTRAINT conv_participant_pk UNIQUE (conversation_id, participant_id);

ALTER TABLE ONLY public.conversation_settings
    ADD CONSTRAINT conv_settings_pk PRIMARY KEY (conversation_id, participant_id);

ALTER TABLE ONLY public.conversation_change
    ADD CONSTRAINT conversation_change_pkey PRIMARY KEY (id);

ALTER TABLE ONLY public.conversation_change
    ADD CONSTRAINT conversation_change_uq UNIQUE (conversation_id, participant_id, change_type);

ALTER TABLE ONLY public.conversation
    ADD CONSTRAINT conversation_creation_id_key UNIQUE (creation_id);

ALTER TABLE ONLY public.conversation_message
    ADD CONSTRAINT conversation_message_creation_id_key UNIQUE (creation_id);

ALTER TABLE ONLY public.conversation
    ADD CONSTRAINT conversation_pk PRIMARY KEY (id);

ALTER TABLE ONLY public.conversation_search_queue
    ADD CONSTRAINT conversation_search_queue_pk PRIMARY KEY (id);

ALTER TABLE ONLY public.customer_domain
    ADD CONSTRAINT customer_domain_pkey PRIMARY KEY (customer_domain_name);

ALTER TABLE ONLY public.db_patch_version
    ADD CONSTRAINT db_patch_version_pkey PRIMARY KEY (version);

ALTER TABLE ONLY public.default_album
    ADD CONSTRAINT default_albums_item_id_item_type_key UNIQUE (item_id, item_type);

ALTER TABLE ONLY public.default_app_visibility
    ADD CONSTRAINT default_app_visibility_pk PRIMARY KEY (app_id, visible_for_role);

ALTER TABLE ONLY public.default_layout_entity
    ADD CONSTRAINT default_layout_entity_pk PRIMARY KEY (person_id);

ALTER TABLE ONLY public.delete_action
    ADD CONSTRAINT delete_action_pk PRIMARY KEY (id);

ALTER TABLE ONLY public.delete_action
    ADD CONSTRAINT delete_action_trn_id_uq UNIQUE (transaction_id);

ALTER TABLE ONLY public.document_access_token
    ADD CONSTRAINT document_access_token_pk PRIMARY KEY (token);

ALTER TABLE ONLY public.document_version_judo_id
    ADD CONSTRAINT document_version_judo_id_pkey PRIMARY KEY (version_id);

ALTER TABLE ONLY public.domain_property
    ADD CONSTRAINT domain_property_pkey PRIMARY KEY (id);

ALTER TABLE ONLY public.domain_property
    ADD CONSTRAINT domain_property_property_name_key UNIQUE (base_property, domain);

ALTER TABLE ONLY public.drive_document_item
    ADD CONSTRAINT drive_document_item_pk PRIMARY KEY (id, publicity);

ALTER TABLE ONLY public.entity_internal_invitation
    ADD CONSTRAINT e_i_i_pk PRIMARY KEY (entity_id, person_id);

ALTER TABLE ONLY public.entity_member_role_name
    ADD CONSTRAINT e_m_r_n_pk PRIMARY KEY (name);

ALTER TABLE ONLY public.entity_member_role_name
    ADD CONSTRAINT e_m_r_n_workflow_uq UNIQUE (name, workflow);

ALTER TABLE ONLY public.entity_member_role
    ADD CONSTRAINT e_m_r_pk PRIMARY KEY (entity_id, person_id, role);

ALTER TABLE ONLY public.entity_member_role
    ADD CONSTRAINT e_m_r_workflow_uq UNIQUE (entity_id, person_id, workflow);

ALTER TABLE ONLY public.entity_version_archive_notice
    ADD CONSTRAINT e_v_archive_notice_pk PRIMARY KEY (id);

ALTER TABLE ONLY public.email_notification_setting
    ADD CONSTRAINT email_notification_settings_pkey PRIMARY KEY (person_id);

ALTER TABLE ONLY public.entity_recommendation
    ADD CONSTRAINT ent_rec_pk PRIMARY KEY (entity_id, person_id);

ALTER TABLE ONLY public.entity_anchor
    ADD CONSTRAINT entity_anchor_pk PRIMARY KEY (entity_id, component, anchor);

ALTER TABLE ONLY public.entity_component_open_for
    ADD CONSTRAINT entity_component_open_for_pkey PRIMARY KEY (entity_id, component_type, authority_name);

ALTER TABLE ONLY public.entity_component
    ADD CONSTRAINT entity_component_pkey PRIMARY KEY (entity_id, component_type);

ALTER TABLE ONLY public.entity_dynamic_attribute
    ADD CONSTRAINT entity_dynamic_attribute_pkey UNIQUE (entity_id, attribute_id, "position");

ALTER TABLE ONLY public.entity
    ADD CONSTRAINT entity_external_url_unique UNIQUE (external_domain);

ALTER TABLE ONLY public.entity_link
    ADD CONSTRAINT entity_link_pkey PRIMARY KEY (entity_id, linked_entity_id);

ALTER TABLE ONLY public.entity_to_entity_type_privacy
    ADD CONSTRAINT entity_linked_entity_type_privacy_pkey1 PRIMARY KEY (entity_id, entity_type, authority_name);

ALTER TABLE ONLY public.entity_parents
    ADD CONSTRAINT entity_parents_pk PRIMARY KEY (entity_id, parent_id);

ALTER TABLE ONLY public.entity
    ADD CONSTRAINT entity_pkey PRIMARY KEY (id);

ALTER TABLE ONLY public.entity_status
    ADD CONSTRAINT entity_status_pk PRIMARY KEY (id);

ALTER TABLE ONLY public.entity_status
    ADD CONSTRAINT entity_status_uq UNIQUE (entity_type, status_name);

ALTER TABLE ONLY public.entity
    ADD CONSTRAINT entity_subdomain_unique UNIQUE (subdomain);

ALTER TABLE ONLY public.entity_tenant
    ADD CONSTRAINT entity_tenant_pk PRIMARY KEY (entity_id);

ALTER TABLE ONLY public.entity_theme
    ADD CONSTRAINT entity_theme_pk PRIMARY KEY (entity_id);

ALTER TABLE ONLY public.entity_type_app_mapping
    ADD CONSTRAINT entity_type_app_map_pk PRIMARY KEY (entity_type);

ALTER TABLE ONLY public.entity_type_name
    ADD CONSTRAINT entity_type_name_pkey PRIMARY KEY (name);

ALTER TABLE ONLY public.entity_version_components
    ADD CONSTRAINT entity_version_components_pkey UNIQUE (entity_version_id);

ALTER TABLE ONLY public.entity_version_drive_change
    ADD CONSTRAINT entity_version_drive_change_pkey PRIMARY KEY (entity_version_id);

ALTER TABLE ONLY public.entity_version
    ADD CONSTRAINT entity_version_major_minor_draft_unique UNIQUE (entity_id, major_version, minor_version, draft_version);

ALTER TABLE ONLY public.entity_version
    ADD CONSTRAINT entity_version_pk PRIMARY KEY (id);

ALTER TABLE ONLY public.entity_version_review
    ADD CONSTRAINT entity_version_release_order_unique UNIQUE (entity_version_id, review_order);

ALTER TABLE ONLY public.entity_version_review
    ADD CONSTRAINT entity_version_review_pk PRIMARY KEY (entity_version_id, reviewer_id);

ALTER TABLE ONLY public.entity_waitlist
    ADD CONSTRAINT entity_waitlist_pk PRIMARY KEY (entity_id, person_id);

ALTER TABLE ONLY public.export_item_event_queue
    ADD CONSTRAINT export_item_event_queue_pkey PRIMARY KEY (id);

ALTER TABLE ONLY public.external_invitation
    ADD CONSTRAINT external_invitation_email_key UNIQUE (email, entity_id);

ALTER TABLE ONLY public.external_invitation
    ADD CONSTRAINT external_invitation_pkey PRIMARY KEY (hash);

ALTER TABLE ONLY public.file_type
    ADD CONSTRAINT file_types_primary_key PRIMARY KEY (id);

ALTER TABLE ONLY public.file_type
    ADD CONSTRAINT file_types_unique_extension UNIQUE (extension);

ALTER TABLE ONLY public.greeting
    ADD CONSTRAINT greeting_pkey PRIMARY KEY (sender_id, recipient_id);

ALTER TABLE ONLY public.ical_access_token
    ADD CONSTRAINT ical_access_token_pk PRIMARY KEY (person_id);

ALTER TABLE ONLY public.ical_access_token
    ADD CONSTRAINT ical_access_token_token_uk UNIQUE (token);

ALTER TABLE ONLY public.importedentity
    ADD CONSTRAINT importedentity_externalid_key UNIQUE (externalid);

ALTER TABLE ONLY public.importedentity
    ADD CONSTRAINT importedentity_internalid_key UNIQUE (internalid);

ALTER TABLE ONLY public.importedperson
    ADD CONSTRAINT importedperson_externalid_key UNIQUE (externalid);

ALTER TABLE ONLY public.importedperson
    ADD CONSTRAINT importedperson_internalid_key UNIQUE (internalid);

ALTER TABLE ONLY public.item_event_queue
    ADD CONSTRAINT item_event_queue_pkey PRIMARY KEY (id);

ALTER TABLE ONLY public.item_visibility_queue_event
    ADD CONSTRAINT item_visibility_queue_event_pk PRIMARY KEY (item_visibility_queue_id, event_type);

ALTER TABLE ONLY public.item_visibility_queue
    ADD CONSTRAINT item_visibility_queue_pk PRIMARY KEY (id);

ALTER TABLE ONLY public.item_visibility_role
    ADD CONSTRAINT itm_vis_role_pk PRIMARY KEY (id);

ALTER TABLE ONLY public.juco_version_update_process
    ADD CONSTRAINT j_v_u_p_pkey PRIMARY KEY (version_id, process);

ALTER TABLE ONLY public.jabber_synced
    ADD CONSTRAINT jabber_synced_pkey PRIMARY KEY (person_id);

ALTER TABLE ONLY public.jabber_system_user_password
    ADD CONSTRAINT jabber_system_user_password_pk PRIMARY KEY (username, server);

ALTER TABLE ONLY public.jabber_temp_password
    ADD CONSTRAINT jabber_temp_password_uq UNIQUE (person_id, type);

ALTER TABLE ONLY public.juco_version
    ADD CONSTRAINT juco_version_pk PRIMARY KEY (id);

ALTER TABLE ONLY public.language
    ADD CONSTRAINT language_pkey PRIMARY KEY (id);

ALTER TABLE ONLY public.like_table
    ADD CONSTRAINT like_pk PRIMARY KEY (item_id, liker_id);

ALTER TABLE ONLY public.link
    ADD CONSTRAINT link_pk PRIMARY KEY (id);

ALTER TABLE ONLY public.map_person_person_invite
    ADD CONSTRAINT map_person_person_invite_pkey PRIMARY KEY (person_id, friend_id);

ALTER TABLE ONLY public.map_person_person
    ADD CONSTRAINT map_person_person_pkey PRIMARY KEY (person_id, friend_id);

ALTER TABLE ONLY public.mdl_album_image
    ADD CONSTRAINT mdl_album_image_pkey PRIMARY KEY (id);

ALTER TABLE ONLY public.mdl_album
    ADD CONSTRAINT mdl_album_pkey PRIMARY KEY (id);

ALTER TABLE ONLY public.mdl_comment
    ADD CONSTRAINT mdl_comments_primary_key PRIMARY KEY (id);

ALTER TABLE ONLY public.mdl_document
    ADD CONSTRAINT mdl_document_files_primary_key PRIMARY KEY (id);

ALTER TABLE ONLY public.mdl_document_version
    ADD CONSTRAINT mdl_document_versions_id_primary_key PRIMARY KEY (id);

ALTER TABLE ONLY public.mdl_document_version
    ADD CONSTRAINT mdl_document_versions_primary_key UNIQUE (version, document_id);

ALTER TABLE ONLY public.mdl_video_link
    ADD CONSTRAINT mdl_videolink_pkey PRIMARY KEY (id);

ALTER TABLE ONLY public.migrated_item
    ADD CONSTRAINT migrated_item_pkey PRIMARY KEY (origin_id);

ALTER TABLE ONLY public.multi_wiki
    ADD CONSTRAINT multi_wiki_parent_unq UNIQUE (entity_id);

ALTER TABLE ONLY public.multi_wiki
    ADD CONSTRAINT multi_wiki_pkey PRIMARY KEY (id);

ALTER TABLE ONLY public.multi_wiki_version_section
    ADD CONSTRAINT multi_wiki_section_pkey PRIMARY KEY (id);

ALTER TABLE ONLY public.multi_wiki_section_visibility
    ADD CONSTRAINT multi_wiki_section_visibility_pkey PRIMARY KEY (version_section_id, authority_name);

ALTER TABLE ONLY public.multi_wiki_version
    ADD CONSTRAINT multi_wiki_version_pkey PRIMARY KEY (id);

ALTER TABLE ONLY public.multi_wiki_version_section
    ADD CONSTRAINT multi_wiki_version_section_section_id_version_id_key UNIQUE (section_id, version_id);

ALTER TABLE ONLY public.native
    ADD CONSTRAINT native_pkey PRIMARY KEY (id);

ALTER TABLE ONLY public.news_blog
    ADD CONSTRAINT newsblog_pkey PRIMARY KEY (id);

ALTER TABLE ONLY public.newszoom_config
    ADD CONSTRAINT newszoom_config_pkey PRIMARY KEY (tenant_id);

ALTER TABLE ONLY public.newszoom_section_channel
    ADD CONSTRAINT newszoom_section_channel_pk PRIMARY KEY (section_id, channel_id);

ALTER TABLE ONLY public.newszoom_section
    ADD CONSTRAINT newszoom_section_pkey PRIMARY KEY (id);

ALTER TABLE ONLY public.notification
    ADD CONSTRAINT noti_pk PRIMARY KEY (id);

ALTER TABLE ONLY public.oauth_token
    ADD CONSTRAINT o_t_access_token_pk PRIMARY KEY (access_token);

ALTER TABLE ONLY public.oauth_access_token
    ADD CONSTRAINT oauth_access_token_pk PRIMARY KEY (token_id);

ALTER TABLE ONLY public.oauth_client_details
    ADD CONSTRAINT oauth_client_details_pkey PRIMARY KEY (client_id);

ALTER TABLE ONLY public.oauth_refresh_token
    ADD CONSTRAINT oauth_refresh_token_pk PRIMARY KEY (token_id);

ALTER TABLE ONLY public.open_social_app_data
    ADD CONSTRAINT open_social_app_data_pkey PRIMARY KEY (id);

ALTER TABLE ONLY public.person
    ADD CONSTRAINT person_email_key UNIQUE (email);

ALTER TABLE ONLY public.person_external_ids
    ADD CONSTRAINT person_external_ids_pkey PRIMARY KEY (person_id, external_system_id);

ALTER TABLE ONLY public.person
    ADD CONSTRAINT person_hashed_id_key UNIQUE (hashed_id);

ALTER TABLE ONLY public.person
    ADD CONSTRAINT person_newmail_key UNIQUE (newmail);

ALTER TABLE ONLY public.person_pending_password_reset
    ADD CONSTRAINT person_pending_password_reset_pkey PRIMARY KEY (person_id);

ALTER TABLE ONLY public.person
    ADD CONSTRAINT person_pkey PRIMARY KEY (id);

ALTER TABLE ONLY public.person_role
    ADD CONSTRAINT person_roles_pkey PRIMARY KEY (role_name);

ALTER TABLE ONLY public.person_tenant
    ADD CONSTRAINT person_tenant_pk PRIMARY KEY (person_id, tenant_id);

ALTER TABLE ONLY public.person_to_role
    ADD CONSTRAINT pkc_person_role PRIMARY KEY (person_id, role_name);

ALTER TABLE ONLY public.profile_allowed_component
    ADD CONSTRAINT profile_allowed_component_pkey PRIMARY KEY (component_type);

ALTER TABLE ONLY public.profile_attribute
    ADD CONSTRAINT profile_attribute_pk PRIMARY KEY (id);

ALTER TABLE ONLY public.profile_component
    ADD CONSTRAINT profile_component_privacy_pkey PRIMARY KEY (person_id, component_type);

ALTER TABLE ONLY public.push_target_app
    ADD CONSTRAINT push_target_app_pk PRIMARY KEY (id);

ALTER TABLE ONLY public.push_target_app
    ADD CONSTRAINT push_target_app_uq UNIQUE (token, protocol);

ALTER TABLE ONLY public.raweditor_version
    ADD CONSTRAINT raweditor_version_parent_item_id_key UNIQUE (parent_item_id, parent_item_type, editor_type, version);

ALTER TABLE ONLY public.raweditor_version
    ADD CONSTRAINT raweditor_version_pkey PRIMARY KEY (id);

ALTER TABLE ONLY public.raweditor_version
    ADD CONSTRAINT raweditor_version_raweditor_id_key UNIQUE (raweditor_id, version);

ALTER TABLE ONLY public.search_filter
    ADD CONSTRAINT search_filter_pk PRIMARY KEY (id);

ALTER TABLE ONLY public.search_index_queue
    ADD CONSTRAINT search_index_queue_pkey PRIMARY KEY (id);

ALTER TABLE ONLY public.secret
    ADD CONSTRAINT secret_pk PRIMARY KEY (secret_key);

ALTER TABLE ONLY public.sh_conversation_attachment
    ADD CONSTRAINT sh_conversation_attachment_pkey PRIMARY KEY (id);

ALTER TABLE ONLY public.sh_conversation_message
    ADD CONSTRAINT sh_conversation_message_pkey PRIMARY KEY (id);

ALTER TABLE ONLY public.sh_conversation
    ADD CONSTRAINT sh_conversation_pkey PRIMARY KEY (id);

ALTER TABLE ONLY public.sh_conversation_settings
    ADD CONSTRAINT sh_conversation_settings_pkey PRIMARY KEY (conversation_id, participant_id);

ALTER TABLE ONLY public.sh_entity_internal_invitation
    ADD CONSTRAINT sh_e_i_i_pk PRIMARY KEY (entity_id, person_id, delete_action_id);

ALTER TABLE ONLY public.sh_entity_member_role
    ADD CONSTRAINT sh_e_m_r_pk PRIMARY KEY (entity_id, person_id, role, delete_action_id);

ALTER TABLE ONLY public.sh_email_notification_setting
    ADD CONSTRAINT sh_email_notification_setting_pkey PRIMARY KEY (person_id);

ALTER TABLE ONLY public.sh_entity_component_open_for
    ADD CONSTRAINT sh_entity_component_open_for_pkey PRIMARY KEY (entity_id, component_type, authority_name);

ALTER TABLE ONLY public.sh_entity_component
    ADD CONSTRAINT sh_entity_component_pkey PRIMARY KEY (entity_id, component_type);

ALTER TABLE ONLY public.sh_entity_link
    ADD CONSTRAINT sh_entity_link_pkey PRIMARY KEY (entity_id, linked_entity_id);

ALTER TABLE ONLY public.sh_entity
    ADD CONSTRAINT sh_entity_pkey PRIMARY KEY (id);

ALTER TABLE ONLY public.sh_entity_recommendation
    ADD CONSTRAINT sh_entity_recommendation_pkey PRIMARY KEY (entity_id, person_id);

ALTER TABLE ONLY public.sh_entity_to_entity_type_priva
    ADD CONSTRAINT sh_entity_to_entity_type_priva_pkey PRIMARY KEY (entity_id, entity_type, authority_name);

ALTER TABLE ONLY public.sh_entity_version
    ADD CONSTRAINT sh_entity_version_pkey PRIMARY KEY (id);

ALTER TABLE ONLY public.sh_entity_version_review
    ADD CONSTRAINT sh_entity_version_review_pkey PRIMARY KEY (entity_version_id, reviewer_id);

ALTER TABLE ONLY public.sh_external_invitation
    ADD CONSTRAINT sh_external_invitation_pkey PRIMARY KEY (hash);

ALTER TABLE ONLY public.sh_map_person_person_invite
    ADD CONSTRAINT sh_map_person_person_invite_pkey PRIMARY KEY (person_id, friend_id);

ALTER TABLE ONLY public.sh_map_person_person
    ADD CONSTRAINT sh_map_person_person_pkey PRIMARY KEY (person_id, friend_id);

ALTER TABLE ONLY public.sh_mdl_album_image
    ADD CONSTRAINT sh_mdl_album_image_pkey PRIMARY KEY (id);

ALTER TABLE ONLY public.sh_mdl_album
    ADD CONSTRAINT sh_mdl_album_pkey PRIMARY KEY (id);

ALTER TABLE ONLY public.sh_mdl_comment
    ADD CONSTRAINT sh_mdl_comment_pkey PRIMARY KEY (id);

ALTER TABLE ONLY public.sh_mdl_document
    ADD CONSTRAINT sh_mdl_document_pkey PRIMARY KEY (id);

ALTER TABLE ONLY public.sh_mdl_document_version
    ADD CONSTRAINT sh_mdl_document_version_pkey PRIMARY KEY (id);

ALTER TABLE ONLY public.sh_mdl_video_link
    ADD CONSTRAINT sh_mdl_video_link_pkey PRIMARY KEY (id);

ALTER TABLE ONLY public.sh_multi_wiki
    ADD CONSTRAINT sh_multi_wiki_pkey PRIMARY KEY (id);

ALTER TABLE ONLY public.sh_multi_wiki_section_visibili
    ADD CONSTRAINT sh_multi_wiki_section_visibili_pkey PRIMARY KEY (version_section_id, authority_name);

ALTER TABLE ONLY public.sh_multi_wiki_version
    ADD CONSTRAINT sh_multi_wiki_version_pkey PRIMARY KEY (id);

ALTER TABLE ONLY public.sh_multi_wiki_version_section
    ADD CONSTRAINT sh_multi_wiki_version_section_pkey PRIMARY KEY (id);

ALTER TABLE ONLY public.sh_news_blog
    ADD CONSTRAINT sh_news_blog_pkey PRIMARY KEY (id);

ALTER TABLE ONLY public.sh_person_external_ids
    ADD CONSTRAINT sh_person_external_ids_pkey PRIMARY KEY (person_id, external_system_id);

ALTER TABLE ONLY public.sh_person
    ADD CONSTRAINT sh_person_pkey PRIMARY KEY (id);

ALTER TABLE ONLY public.sh_person_to_role
    ADD CONSTRAINT sh_person_to_role_pkey PRIMARY KEY (person_id, role_name);

ALTER TABLE ONLY public.sh_profile_component
    ADD CONSTRAINT sh_profile_component_pkey PRIMARY KEY (person_id, component_type);

ALTER TABLE ONLY public.sh_raweditor_version
    ADD CONSTRAINT sh_raweditor_version_pkey PRIMARY KEY (id);

ALTER TABLE ONLY public.sh_survey
    ADD CONSTRAINT sh_survey_pkey PRIMARY KEY (id);

ALTER TABLE ONLY public.sh_survey_predefined_answer
    ADD CONSTRAINT sh_survey_predefined_answer_pkey PRIMARY KEY (id);

ALTER TABLE ONLY public.sh_survey_question
    ADD CONSTRAINT sh_survey_question_pkey PRIMARY KEY (id);

ALTER TABLE ONLY public.sh_user_setting
    ADD CONSTRAINT sh_user_setting_pkey1 PRIMARY KEY (person_id, key);

ALTER TABLE ONLY public.sh_workstream_attachment
    ADD CONSTRAINT sh_wa_pk PRIMARY KEY (item_id);

ALTER TABLE ONLY public.sh_wiki_editor_backup
    ADD CONSTRAINT sh_wiki_editor_backup_pkey PRIMARY KEY (wiki_editor_id);

ALTER TABLE ONLY public.sh_wiki_editor
    ADD CONSTRAINT sh_wiki_editor_pkey PRIMARY KEY (id);

ALTER TABLE ONLY public.sh_wiki_editor_version
    ADD CONSTRAINT sh_wiki_editor_version_pkey PRIMARY KEY (id);

ALTER TABLE ONLY public.sh_workstream_entry
    ADD CONSTRAINT sh_workstream_entry_pkey PRIMARY KEY (id);

ALTER TABLE ONLY public.shadow_table
    ADD CONSTRAINT shadow_table_pk PRIMARY KEY (table_name);

ALTER TABLE ONLY public.subscriptions
    ADD CONSTRAINT subscriptions_pk PRIMARY KEY (item_id, subscriber_id);

ALTER TABLE ONLY public.survey
    ADD CONSTRAINT survey_pkey PRIMARY KEY (id);

ALTER TABLE ONLY public.survey_predefined_answer
    ADD CONSTRAINT survey_predefined_multiple_choice_answer_pkey PRIMARY KEY (id);

ALTER TABLE ONLY public.survey_question
    ADD CONSTRAINT survey_question_pkey PRIMARY KEY (id);

ALTER TABLE ONLY public.sync_app
    ADD CONSTRAINT sync_app_pk PRIMARY KEY (id);

ALTER TABLE ONLY public.task_filter_order
    ADD CONSTRAINT task_filter_order_pk PRIMARY KEY (filter_id, task_id);

ALTER TABLE ONLY public.task_filter_order
    ADD CONSTRAINT task_filter_order_pos_uk UNIQUE (filter_id, "position") DEFERRABLE;

ALTER TABLE ONLY public.task_list_permissions
    ADD CONSTRAINT task_list_perm_pk PRIMARY KEY (task_list_id, role);

ALTER TABLE ONLY public.task_list
    ADD CONSTRAINT task_list_pkey PRIMARY KEY (id);

ALTER TABLE ONLY public.tasks
    ADD CONSTRAINT tasks_pk PRIMARY KEY (id);

ALTER TABLE ONLY public.tenant_app_visibility
    ADD CONSTRAINT tenant_app_visibility_pk PRIMARY KEY (app_id, tenant_id, visible_for_role);

ALTER TABLE ONLY public.tenant
    ADD CONSTRAINT tenant_pkey PRIMARY KEY (id);

ALTER TABLE ONLY public.tenant_theme
    ADD CONSTRAINT tenant_theme_pkey PRIMARY KEY (tenant_id, theme_attribute);

ALTER TABLE ONLY public.theme
    ADD CONSTRAINT theme_pk PRIMARY KEY (id);

ALTER TABLE ONLY public.translation
    ADD CONSTRAINT translation_pkey PRIMARY KEY (language_id, native_id);

ALTER TABLE ONLY public.person
    ADD CONSTRAINT uniqhash UNIQUE (hashed_id);

ALTER TABLE ONLY public.home_screen_apps
    ADD CONSTRAINT user_app_setting_pk PRIMARY KEY (person_id, app_id);

ALTER TABLE ONLY public.user_setting
    ADD CONSTRAINT user_setting_pkey PRIMARY KEY (person_id, key);

ALTER TABLE ONLY public.version
    ADD CONSTRAINT version_pkey PRIMARY KEY (id, release);

ALTER TABLE ONLY public.workstream_attachment
    ADD CONSTRAINT wa_entry_position_uk UNIQUE (entry_id, "position");

ALTER TABLE ONLY public.workstream_attachment
    ADD CONSTRAINT wa_pk PRIMARY KEY (item_id);

ALTER TABLE ONLY public.wiki_editor_backup
    ADD CONSTRAINT wiki_editor_backup_pkey PRIMARY KEY (wiki_editor_id);

ALTER TABLE ONLY public.wiki_editor
    ADD CONSTRAINT wiki_editor_parent_item_unique UNIQUE (parent_item_id, parent_item_type);

ALTER TABLE ONLY public.wiki_editor
    ADD CONSTRAINT wiki_editor_pkey PRIMARY KEY (id);

ALTER TABLE ONLY public.wiki_editor_version
    ADD CONSTRAINT wiki_editor_version_pkey PRIMARY KEY (id);

ALTER TABLE ONLY public.wiki_editor_version
    ADD CONSTRAINT wiki_editor_version_unique UNIQUE (version, wiki_editor_id);

ALTER TABLE ONLY public.workstream_entry
    ADD CONSTRAINT workstream_entries_pkey PRIMARY KEY (id);

ALTER TABLE ONLY public.workstream_entry_attribute
    ADD CONSTRAINT workstream_entry_attribute_pkey PRIMARY KEY (entry_id, key);

ALTER TABLE ONLY public.workstream_pinned_hide_date
    ADD CONSTRAINT workstream_pinned_hide_date_pk PRIMARY KEY (person_id, origin_item_id);

ALTER TABLE ONLY public.workstream_read
    ADD CONSTRAINT workstream_read_pkey PRIMARY KEY (person_id, item_id);

CREATE INDEX chat_search_queue_creation_date_idx ON public.chat_search_queue USING btree (created_at);

CREATE INDEX chat_search_queue_id_lock_progress_idx ON public.chat_search_queue USING btree (id, lock_id, in_progress);

CREATE INDEX chat_search_queue_item_progress_create_date_idx ON public.chat_search_queue USING btree (item_id, item_type, in_progress, created_at);

CREATE INDEX chat_search_queue_modify_date_idx ON public.chat_search_queue USING btree (modified_at);

CREATE INDEX chat_search_queue_progress_modify_date_idx ON public.chat_search_queue USING btree (in_progress, modified_at);

CREATE INDEX cm_convid_cdate_idx ON public.conversation_message USING btree (conversation_id, create_date DESC);

CREATE INDEX cm_create_date_idx ON public.conversation_message USING btree (create_date DESC);

CREATE INDEX cm_modify_date_idx ON public.conversation_message USING btree (modify_date);

CREATE INDEX conv_attachment_doc_id_idx ON public.conversation_attachment USING btree (document_id);

CREATE INDEX conv_attachment_msg_fk_idx ON public.conversation_attachment USING btree (message_id);

CREATE INDEX conversation_change_create_date_idx ON public.conversation_change USING btree (create_date);

CREATE INDEX conversation_change_participant_id_create_date_idx ON public.conversation_change USING btree (participant_id, create_date);

CREATE INDEX conversation_last_message_date_idx ON public.conversation USING btree (last_message_date DESC);

CREATE INDEX conversation_search_queue_creation_date_idx ON public.conversation_search_queue USING btree (created_at);

CREATE INDEX conversation_search_queue_id_lock_progress_idx ON public.conversation_search_queue USING btree (id, lock_id, in_progress);

CREATE INDEX conversation_search_queue_item_progress_create_date_idx ON public.conversation_search_queue USING btree (item_id, item_type, in_progress, created_at);

CREATE INDEX conversation_search_queue_modify_date_idx ON public.conversation_search_queue USING btree (modified_at);

CREATE INDEX conversation_search_queue_progress_modify_date_idx ON public.conversation_search_queue USING btree (in_progress, modified_at);

CREATE INDEX delete_action_deleted_at_id_idx ON public.delete_action USING btree (deleted_at, id);

CREATE INDEX e_m_r_entity_role_idx ON public.entity_member_role USING btree (entity_id, role);

CREATE INDEX e_m_r_person_role_idx ON public.entity_member_role USING btree (person_id, role);

CREATE INDEX e_m_r_role_idx ON public.entity_member_role USING btree (role);

CREATE INDEX entity_anchor_id_anchor_idx ON public.entity_anchor USING btree (entity_id, anchor);

CREATE UNIQUE INDEX entity_parents_parent_idx ON public.entity_parents USING btree (parent_id, entity_id);

CREATE INDEX entity_version_entity_id_index ON public.entity_version USING btree (entity_id);

CREATE INDEX file_types_extension_index ON public.file_type USING btree (extension);

CREATE INDEX idx_conversation_type ON public.conversation USING btree (type);

CREATE INDEX idx_entity_component_entity_id ON public.entity_component USING btree (entity_id);

CREATE INDEX idx_entity_entity_type ON public.entity USING btree (entity_type);

CREATE INDEX idx_external_invitation_entity_id ON public.external_invitation USING btree (entity_id);

CREATE INDEX idx_greeting_recipient_id ON public.greeting USING btree (recipient_id);

CREATE INDEX idx_importedentities_extid ON public.importedentity USING btree (lower((externalid)::text));

CREATE INDEX idx_importedperson_extid ON public.importedperson USING btree (lower((externalid)::text));

CREATE INDEX idx_mdl_album_parent_item ON public.mdl_album USING btree (parent_item_id, parent_item_type, module);

CREATE INDEX idx_mdl_videolink_parent ON public.mdl_video_link USING btree (parent_item_id, parent_item_type);

CREATE INDEX idx_migration_mapping_name ON public.migration_mapping USING btree (migration_name);

CREATE INDEX idx_newsblog_draft ON public.news_blog USING btree (parent_item_id, parent_item_type, author_id, draft);

CREATE INDEX idx_newsblog_parent_item ON public.news_blog USING btree (parent_item_id, parent_item_type, draft);

CREATE INDEX idx_notification_key ON public.notification USING btree (receiver_id, state);

CREATE INDEX idx_oauth_access_token_auth ON public.oauth_access_token USING btree (authentication_id);

CREATE INDEX idx_oauth_access_token_refresh ON public.oauth_access_token USING btree (refresh_token);

CREATE INDEX idx_oauth_access_token_user ON public.oauth_access_token USING btree (client_id, user_name);

CREATE INDEX idx_oauth_token_client ON public.oauth_token USING btree (client_id);

CREATE INDEX idx_oauth_token_refresh ON public.oauth_token USING btree (refresh_token);

CREATE INDEX idx_oauth_token_user ON public.oauth_token USING btree (person_id);

CREATE UNIQUE INDEX idx_person_lower_email ON public.person USING btree (lower((email)::text));

CREATE INDEX idx_survey_parent_item_and_type ON public.survey USING btree (parent_item_id, parent_item_type);

CREATE INDEX idx_user_hash_index ON public.person USING btree (hashed_id);

CREATE INDEX idx_wiki_editor_version_wiki_editor_id ON public.wiki_editor_version USING btree (wiki_editor_id);

CREATE INDEX item_visibility_queue_creation_date_idx ON public.item_visibility_queue USING btree (created_at);

CREATE INDEX item_visibility_queue_id_lock_progress_idx ON public.item_visibility_queue USING btree (id, lock_id, in_progress);

CREATE INDEX item_visibility_queue_item_progress_create_date_idx ON public.item_visibility_queue USING btree (item_id, item_type, in_progress, created_at);

CREATE INDEX item_visibility_queue_modify_date_idx ON public.item_visibility_queue USING btree (modified_at);

CREATE INDEX item_visibility_queue_progress_modify_date_idx ON public.item_visibility_queue USING btree (in_progress, modified_at);

CREATE INDEX itm_vis_role_iid_idx ON public.item_visibility_role USING btree (item_id);

CREATE INDEX itm_vis_role_role_idx ON public.item_visibility_role USING btree (role_name, item_id, origin_type);

CREATE INDEX like_created_at_idx ON public.like_table USING btree (item_id, created_at);

CREATE INDEX like_liker_id_idx ON public.like_table USING btree (liker_id);

CREATE INDEX like_recent_idx ON public.like_table USING btree (item_id, recent);

CREATE INDEX mdl_comments_parent_idx ON public.mdl_comment USING btree (parent_item_id);

CREATE INDEX mdl_document_versions_document_id_idx ON public.mdl_document_version USING btree (document_id);

CREATE INDEX multi_wiki_version_section_version_id_idx ON public.multi_wiki_version_section USING btree (version_id);

CREATE INDEX noti_o_t_e_s_idx ON public.notification USING btree (originator_id, notification_type, entity_id, state);

CREATE INDEX noti_sub_type_idx ON public.notification USING btree (subject_id, notification_type);

CREATE INDEX person_to_roles_in_pogress_idx ON public.jabber_synced USING btree (in_progress);

CREATE INDEX profile_attribute_pers_id_idx ON public.profile_attribute USING btree (person_id);

CREATE INDEX push_target_app_person_idx ON public.push_target_app USING btree (person_id);

CREATE UNIQUE INDEX subs_sid_iid_state_idx ON public.subscriptions USING btree (subscriber_id, item_id, subscription_state);

CREATE INDEX task_filter_order_task_idx ON public.task_filter_order USING btree (task_id, filter_id, "position");

CREATE INDEX tasks_assignee_idx ON public.tasks USING btree (assignee_id, status);

CREATE INDEX tasks_owner_idx ON public.tasks USING btree (owner_id, status);

CREATE INDEX tasks_parent_task_assignee_idx ON public.tasks USING btree (parent_task_id, assignee_id);

CREATE INDEX tasks_parent_task_owner_idx ON public.tasks USING btree (parent_task_id, owner_id);

CREATE INDEX tasks_task_list_id_status_idx ON public.tasks USING btree (task_list_id, status);

CREATE UNIQUE INDEX theme_element_pkey ON public.theme_attribute USING btree (theme_id, attribute_type);

CREATE INDEX we_orig_item_idx ON public.workstream_entry USING btree (origin_item_id, type);

CREATE INDEX workstream_entry_author_idx ON public.workstream_entry USING btree (author_id, type, origin_item_id);

CREATE INDEX workstream_entry_created_idx ON public.workstream_entry USING btree (created_at DESC);

CREATE INDEX workstream_entry_pinned_idx ON public.workstream_entry USING btree (pinned);

CREATE INDEX workstream_entry_subject_idx ON public.workstream_entry USING btree (subject_item_id, type, origin_item_id);

CREATE INDEX workstream_entry_type_ids ON public.workstream_entry USING btree (type);

CREATE INDEX workstream_read_person_idx ON public.workstream_read USING btree (person_id);

CREATE TRIGGER account_failed_logins_t BEFORE UPDATE ON public.account_failed_logins FOR EACH ROW EXECUTE PROCEDURE public.update_mtime();

CREATE TRIGGER account_lock_t BEFORE UPDATE ON public.account_lock FOR EACH ROW EXECUTE PROCEDURE public.update_mtime();

CREATE TRIGGER conversation_ad AFTER DELETE ON public.conversation FOR EACH ROW EXECUTE PROCEDURE public.conversation_ad_fn();

CREATE TRIGGER conversation_attachment_ad AFTER DELETE ON public.conversation_attachment FOR EACH ROW EXECUTE PROCEDURE public.conversation_attachment_ad_fn();

CREATE TRIGGER conversation_message_ad AFTER DELETE ON public.conversation_message FOR EACH ROW EXECUTE PROCEDURE public.conversation_message_ad_fn();

CREATE TRIGGER conversation_participant_ad AFTER DELETE ON public.conversation_participant FOR EACH ROW EXECUTE PROCEDURE public.conversation_participant_ad_fn();

CREATE TRIGGER conversation_settings_ad AFTER DELETE ON public.conversation_settings FOR EACH ROW EXECUTE PROCEDURE public.conversation_settings_ad_fn();

CREATE TRIGGER default_album_ad AFTER DELETE ON public.default_album FOR EACH ROW EXECUTE PROCEDURE public.default_album_ad_fn();

CREATE TRIGGER email_notification_setting_ad AFTER DELETE ON public.email_notification_setting FOR EACH ROW EXECUTE PROCEDURE public.email_notification_setting_ad_fn();

CREATE TRIGGER entity_ad AFTER DELETE ON public.entity FOR EACH ROW EXECUTE PROCEDURE public.entity_ad_fn();

CREATE TRIGGER entity_component_ad AFTER DELETE ON public.entity_component FOR EACH ROW EXECUTE PROCEDURE public.entity_component_ad_fn();

CREATE TRIGGER entity_component_open_for_ad AFTER DELETE ON public.entity_component_open_for FOR EACH ROW EXECUTE PROCEDURE public.entity_component_open_for_ad_fn();

CREATE TRIGGER entity_dynamic_attribute_ad AFTER DELETE ON public.entity_dynamic_attribute FOR EACH ROW EXECUTE PROCEDURE public.entity_dynamic_attribute_ad_fn();

CREATE TRIGGER entity_internal_invitation_ad AFTER DELETE ON public.entity_internal_invitation FOR EACH ROW EXECUTE PROCEDURE public.entity_internal_invitation_ad_fn();

CREATE TRIGGER entity_link_ad AFTER DELETE ON public.entity_link FOR EACH ROW EXECUTE PROCEDURE public.entity_link_ad_fn();

CREATE TRIGGER entity_member_role_ad AFTER DELETE ON public.entity_member_role FOR EACH ROW EXECUTE PROCEDURE public.entity_member_role_ad_fn();

CREATE TRIGGER entity_privacy_ad AFTER DELETE ON public.entity_privacy FOR EACH ROW EXECUTE PROCEDURE public.entity_privacy_ad_fn();

CREATE TRIGGER entity_recommendation_ad AFTER DELETE ON public.entity_recommendation FOR EACH ROW EXECUTE PROCEDURE public.entity_recommendation_ad_fn();

CREATE TRIGGER entity_to_entity_type_privacy_ad AFTER DELETE ON public.entity_to_entity_type_privacy FOR EACH ROW EXECUTE PROCEDURE public.entity_to_entity_type_privacy_ad_fn();

CREATE TRIGGER entity_version_ad AFTER DELETE ON public.entity_version FOR EACH ROW EXECUTE PROCEDURE public.entity_version_ad_fn();

CREATE TRIGGER entity_version_components_ad AFTER DELETE ON public.entity_version_components FOR EACH ROW EXECUTE PROCEDURE public.entity_version_components_ad_fn();

CREATE TRIGGER entity_version_review_ad AFTER DELETE ON public.entity_version_review FOR EACH ROW EXECUTE PROCEDURE public.entity_version_review_ad_fn();

CREATE TRIGGER entity_waitlist_ad AFTER DELETE ON public.entity_waitlist FOR EACH ROW EXECUTE PROCEDURE public.entity_waitlist_ad_fn();

CREATE TRIGGER external_invitation_ad AFTER DELETE ON public.external_invitation FOR EACH ROW EXECUTE PROCEDURE public.external_invitation_ad_fn();

CREATE TRIGGER like_table_ad AFTER DELETE ON public.like_table FOR EACH ROW EXECUTE PROCEDURE public.like_table_ad_fn();

CREATE TRIGGER map_person_person_ad AFTER DELETE ON public.map_person_person FOR EACH ROW EXECUTE PROCEDURE public.map_person_person_ad_fn();

CREATE TRIGGER map_person_person_invite_ad AFTER DELETE ON public.map_person_person_invite FOR EACH ROW EXECUTE PROCEDURE public.map_person_person_invite_ad_fn();

CREATE TRIGGER mdl_album_ad AFTER DELETE ON public.mdl_album FOR EACH ROW EXECUTE PROCEDURE public.mdl_album_ad_fn();

CREATE TRIGGER mdl_album_image_ad AFTER DELETE ON public.mdl_album_image FOR EACH ROW EXECUTE PROCEDURE public.mdl_album_image_ad_fn();

CREATE TRIGGER mdl_comment_ad AFTER DELETE ON public.mdl_comment FOR EACH ROW EXECUTE PROCEDURE public.mdl_comment_ad_fn();

CREATE TRIGGER mdl_document_ad AFTER DELETE ON public.mdl_document FOR EACH ROW EXECUTE PROCEDURE public.mdl_document_ad_fn();

CREATE TRIGGER mdl_document_version_ad AFTER DELETE ON public.mdl_document_version FOR EACH ROW EXECUTE PROCEDURE public.mdl_document_version_ad_fn();

CREATE TRIGGER mdl_video_link_ad AFTER DELETE ON public.mdl_video_link FOR EACH ROW EXECUTE PROCEDURE public.mdl_video_link_ad_fn();

CREATE TRIGGER multi_wiki_ad AFTER DELETE ON public.multi_wiki FOR EACH ROW EXECUTE PROCEDURE public.multi_wiki_ad_fn();

CREATE TRIGGER multi_wiki_section_visibility_ad AFTER DELETE ON public.multi_wiki_section_visibility FOR EACH ROW EXECUTE PROCEDURE public.multi_wiki_section_visibility_ad_fn();

CREATE TRIGGER multi_wiki_version_ad AFTER DELETE ON public.multi_wiki_version FOR EACH ROW EXECUTE PROCEDURE public.multi_wiki_version_ad_fn();

CREATE TRIGGER multi_wiki_version_section_ad AFTER DELETE ON public.multi_wiki_version_section FOR EACH ROW EXECUTE PROCEDURE public.multi_wiki_version_section_ad_fn();

CREATE TRIGGER news_blog_ad AFTER DELETE ON public.news_blog FOR EACH ROW EXECUTE PROCEDURE public.news_blog_ad_fn();

CREATE TRIGGER person_ad AFTER DELETE ON public.person FOR EACH ROW EXECUTE PROCEDURE public.person_ad_fn();

CREATE TRIGGER person_external_ids_ad AFTER DELETE ON public.person_external_ids FOR EACH ROW EXECUTE PROCEDURE public.person_external_ids_ad_fn();

CREATE TRIGGER person_pending_password_reset_t BEFORE UPDATE ON public.person_pending_password_reset FOR EACH ROW EXECUTE PROCEDURE public.update_mtime();

CREATE TRIGGER person_to_role_ad AFTER DELETE ON public.person_to_role FOR EACH ROW EXECUTE PROCEDURE public.person_to_role_ad_fn();

CREATE TRIGGER profile_attribute_ad AFTER DELETE ON public.profile_attribute FOR EACH ROW EXECUTE PROCEDURE public.profile_attribute_ad_fn();

CREATE TRIGGER profile_component_ad AFTER DELETE ON public.profile_component FOR EACH ROW EXECUTE PROCEDURE public.profile_component_ad_fn();

CREATE TRIGGER raweditor_version_ad AFTER DELETE ON public.raweditor_version FOR EACH ROW EXECUTE PROCEDURE public.raweditor_version_ad_fn();

CREATE TRIGGER scheduled_workstream_entry_t BEFORE UPDATE ON public.scheduled_workstream_entry FOR EACH ROW EXECUTE PROCEDURE public.update_mtime();

CREATE TRIGGER subscriptions_ad AFTER DELETE ON public.subscriptions FOR EACH ROW EXECUTE PROCEDURE public.subscriptions_ad_fn();

CREATE TRIGGER survey_ad AFTER DELETE ON public.survey FOR EACH ROW EXECUTE PROCEDURE public.survey_ad_fn();

CREATE TRIGGER survey_participant_answer_ad AFTER DELETE ON public.survey_participant_answer FOR EACH ROW EXECUTE PROCEDURE public.survey_participant_answer_ad_fn();

CREATE TRIGGER survey_predefined_answer_ad AFTER DELETE ON public.survey_predefined_answer FOR EACH ROW EXECUTE PROCEDURE public.survey_predefined_answer_ad_fn();

CREATE TRIGGER survey_question_ad AFTER DELETE ON public.survey_question FOR EACH ROW EXECUTE PROCEDURE public.survey_question_ad_fn();

CREATE TRIGGER task_list_ad AFTER DELETE ON public.task_list FOR EACH ROW EXECUTE PROCEDURE public.task_list_ad_fn();

CREATE TRIGGER task_list_permissions_ad AFTER DELETE ON public.task_list_permissions FOR EACH ROW EXECUTE PROCEDURE public.task_list_permissions_ad_fn();

CREATE TRIGGER tasks_ad AFTER DELETE ON public.tasks FOR EACH ROW EXECUTE PROCEDURE public.tasks_ad_fn();

CREATE TRIGGER user_setting_ad AFTER DELETE ON public.user_setting FOR EACH ROW EXECUTE PROCEDURE public.user_setting_ad_fn();

CREATE TRIGGER wiki_editor_ad AFTER DELETE ON public.wiki_editor FOR EACH ROW EXECUTE PROCEDURE public.wiki_editor_ad_fn();

CREATE TRIGGER wiki_editor_backup_ad AFTER DELETE ON public.wiki_editor_backup FOR EACH ROW EXECUTE PROCEDURE public.wiki_editor_backup_ad_fn();

CREATE TRIGGER wiki_editor_version_ad AFTER DELETE ON public.wiki_editor_version FOR EACH ROW EXECUTE PROCEDURE public.wiki_editor_version_ad_fn();

CREATE TRIGGER workstream_attachment_ad AFTER DELETE ON public.workstream_attachment FOR EACH ROW EXECUTE PROCEDURE public.workstream_attachment_ad_fn();

CREATE TRIGGER workstream_entry_ad AFTER DELETE ON public.workstream_entry FOR EACH ROW EXECUTE PROCEDURE public.workstream_entry_ad_fn();

CREATE TRIGGER workstream_entry_attribute_ad AFTER DELETE ON public.workstream_entry_attribute FOR EACH ROW EXECUTE PROCEDURE public.workstream_entry_attribute_ad_fn();

ALTER TABLE ONLY public.account_failed_logins
    ADD CONSTRAINT account_failed_logins_person_id_fkey FOREIGN KEY (person_id) REFERENCES public.person(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.account_lock
    ADD CONSTRAINT account_lock_person_id_fkey FOREIGN KEY (person_id) REFERENCES public.person(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.additional_app_info
    ADD CONSTRAINT additional_app_info_app_id_fk FOREIGN KEY (app_id) REFERENCES public.app(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.audit_log_param
    ADD CONSTRAINT audit_log_param_audit_id_fk FOREIGN KEY (audit_id) REFERENCES public.audit_log(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.conversation_attachment
    ADD CONSTRAINT conv_attachment_msg_fk FOREIGN KEY (message_id) REFERENCES public.conversation_message(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.conversation_message
    ADD CONSTRAINT conv_message_conv_fk FOREIGN KEY (conversation_id) REFERENCES public.conversation(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.conversation_message
    ADD CONSTRAINT conv_message_pers_fk FOREIGN KEY (from_person_id) REFERENCES public.person(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.conversation_participant
    ADD CONSTRAINT conv_participant_conv_fk FOREIGN KEY (conversation_id) REFERENCES public.conversation(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.conversation_participant
    ADD CONSTRAINT conv_participant_part_fk FOREIGN KEY (participant_id) REFERENCES public.person(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.conversation_settings
    ADD CONSTRAINT conv_settings_conv_fk FOREIGN KEY (conversation_id) REFERENCES public.conversation(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.conversation_settings
    ADD CONSTRAINT conv_settings_part_fk FOREIGN KEY (participant_id) REFERENCES public.person(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.conversation_change
    ADD CONSTRAINT conversation_change_participant_id_fk FOREIGN KEY (participant_id) REFERENCES public.person(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.default_album
    ADD CONSTRAINT default_albums_album_id_fkey FOREIGN KEY (album_id) REFERENCES public.mdl_album(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.default_app_visibility
    ADD CONSTRAINT default_app_visibility_fk FOREIGN KEY (app_id) REFERENCES public.app(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.default_layout_entity
    ADD CONSTRAINT default_layout_entity_id_fk FOREIGN KEY (entity_id) REFERENCES public.entity(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.document_version_judo_id
    ADD CONSTRAINT document_version_judo_id_version_id_fkey FOREIGN KEY (version_id) REFERENCES public.mdl_document_version(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.domain_property
    ADD CONSTRAINT domain_property_base_property_fkey FOREIGN KEY (base_property) REFERENCES public.base_property(id);

ALTER TABLE ONLY public.entity_internal_invitation
    ADD CONSTRAINT e_i_i_entity_fk FOREIGN KEY (entity_id) REFERENCES public.entity(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.entity_internal_invitation
    ADD CONSTRAINT e_i_i_inviter_fk FOREIGN KEY (inviter_id) REFERENCES public.person(id);

ALTER TABLE ONLY public.entity_internal_invitation
    ADD CONSTRAINT e_i_i_person_fk FOREIGN KEY (person_id) REFERENCES public.person(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.entity_member_role
    ADD CONSTRAINT e_m_entity_fk FOREIGN KEY (entity_id) REFERENCES public.entity(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.entity_member_role
    ADD CONSTRAINT e_m_person_fk FOREIGN KEY (person_id) REFERENCES public.person(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.entity_member_role
    ADD CONSTRAINT e_m_r_role_fk FOREIGN KEY (role, workflow) REFERENCES public.entity_member_role_name(name, workflow);

ALTER TABLE ONLY public.email_notification_setting
    ADD CONSTRAINT email_notification_settings_person_id_fkey FOREIGN KEY (person_id) REFERENCES public.person(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.entity_recommendation
    ADD CONSTRAINT ent_rec_entity_fk FOREIGN KEY (entity_id) REFERENCES public.entity(id);

ALTER TABLE ONLY public.entity_recommendation
    ADD CONSTRAINT ent_rec_person_fk FOREIGN KEY (person_id) REFERENCES public.person(id);

ALTER TABLE ONLY public.entity_recommendation
    ADD CONSTRAINT ent_rec_recommender_fk FOREIGN KEY (recommender_id) REFERENCES public.person(id);

ALTER TABLE ONLY public.entity
    ADD CONSTRAINT entity_assignee_pers_fk FOREIGN KEY (assignee_id) REFERENCES public.person(id) ON DELETE SET NULL;

ALTER TABLE ONLY public.entity_component
    ADD CONSTRAINT entity_component_entity_id_fkey FOREIGN KEY (entity_id) REFERENCES public.entity(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.entity_component_open_for
    ADD CONSTRAINT entity_component_open_for_entity_id_fkey FOREIGN KEY (entity_id) REFERENCES public.entity(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.entity_dynamic_attribute
    ADD CONSTRAINT entity_dynamic_attribute_entity_fk FOREIGN KEY (entity_id) REFERENCES public.entity(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.entity
    ADD CONSTRAINT entity_entity_type_name_fkey FOREIGN KEY (entity_type) REFERENCES public.entity_type_name(name);

ALTER TABLE ONLY public.entity_link
    ADD CONSTRAINT entity_link_entity_id_fkey FOREIGN KEY (entity_id) REFERENCES public.entity(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.entity_link
    ADD CONSTRAINT entity_link_linked_entity_id_fkey FOREIGN KEY (linked_entity_id) REFERENCES public.entity(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.entity_to_entity_type_privacy
    ADD CONSTRAINT entity_linked_entity_type_privacy_entity_id_fkey1 FOREIGN KEY (entity_id) REFERENCES public.entity(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.entity_to_entity_type_privacy
    ADD CONSTRAINT entity_linked_entity_type_privacy_entity_type_name_fkey FOREIGN KEY (entity_type) REFERENCES public.entity_type_name(name);

ALTER TABLE ONLY public.entity_privacy
    ADD CONSTRAINT entity_privacy_entity_id_fkey FOREIGN KEY (entity_id) REFERENCES public.entity(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.entity_theme
    ADD CONSTRAINT entity_theme_entity_id_fk FOREIGN KEY (entity_id) REFERENCES public.entity(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.entity_theme
    ADD CONSTRAINT entity_theme_theme_fk FOREIGN KEY (theme_id) REFERENCES public.theme(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.entity_type_app_mapping
    ADD CONSTRAINT entity_type_app_map_app_id_fk FOREIGN KEY (app_id) REFERENCES public.app(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.entity_type_app_mapping
    ADD CONSTRAINT entity_type_app_map_etype_fk FOREIGN KEY (entity_type) REFERENCES public.entity_type_name(name) ON DELETE CASCADE;

ALTER TABLE ONLY public.entity_version
    ADD CONSTRAINT entity_version_entity_id_fkey FOREIGN KEY (entity_id) REFERENCES public.entity(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.entity_version_components
    ADD CONSTRAINT entity_version_fkey FOREIGN KEY (entity_version_id) REFERENCES public.entity_version(id);

ALTER TABLE ONLY public.entity_version
    ADD CONSTRAINT entity_version_id_fkey FOREIGN KEY (person_id) REFERENCES public.person(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.entity_version_review
    ADD CONSTRAINT entity_version_id_fkey FOREIGN KEY (entity_version_id) REFERENCES public.entity_version(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.entity_version_review
    ADD CONSTRAINT entity_version_review_inviter_fkey FOREIGN KEY (inviter_id) REFERENCES public.person(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.entity_version_review
    ADD CONSTRAINT entity_version_review_person_fkey FOREIGN KEY (reviewer_id) REFERENCES public.person(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.entity_version
    ADD CONSTRAINT entity_version_status_fk FOREIGN KEY (entity_status_id) REFERENCES public.entity_status(id);

ALTER TABLE ONLY public.entity_waitlist
    ADD CONSTRAINT entity_waitlist_entity_fk FOREIGN KEY (entity_id) REFERENCES public.entity(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.entity_waitlist
    ADD CONSTRAINT entity_waitlist_person_fk FOREIGN KEY (person_id) REFERENCES public.person(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.entity_version_drive_change
    ADD CONSTRAINT evdc_entity_version_fk FOREIGN KEY (entity_version_id) REFERENCES public.entity_version(id);

ALTER TABLE ONLY public.external_invitation
    ADD CONSTRAINT external_invitation_entity_id_fkey FOREIGN KEY (entity_id) REFERENCES public.entity(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.external_invitation
    ADD CONSTRAINT external_invitation_inviter_id_fkey FOREIGN KEY (inviter_id) REFERENCES public.person(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.greeting
    ADD CONSTRAINT greeting_recipient_id_fkey FOREIGN KEY (recipient_id) REFERENCES public.person(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.greeting
    ADD CONSTRAINT greeting_sender_id_fkey FOREIGN KEY (sender_id) REFERENCES public.person(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.ical_access_token
    ADD CONSTRAINT ical_access_token_person_id_fk FOREIGN KEY (person_id) REFERENCES public.person(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.importedentity
    ADD CONSTRAINT importedentity_internalid_fkey FOREIGN KEY (internalid) REFERENCES public.entity(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.importedperson
    ADD CONSTRAINT importedperson_internalid_fkey FOREIGN KEY (internalid) REFERENCES public.person(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.item_visibility_queue_event
    ADD CONSTRAINT item_visibility_queue_event_queue_fk FOREIGN KEY (item_visibility_queue_id) REFERENCES public.item_visibility_queue(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.juco_version_update_process
    ADD CONSTRAINT j_v_u_p_j_version_fk FOREIGN KEY (version_id) REFERENCES public.juco_version(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.jabber_synced
    ADD CONSTRAINT jabber_synced_person_id_fkey FOREIGN KEY (person_id) REFERENCES public.person(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.jabber_temp_password
    ADD CONSTRAINT jabber_temp_password_person_id_fkey FOREIGN KEY (person_id) REFERENCES public.person(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.like_table
    ADD CONSTRAINT like_liker_id_fkey FOREIGN KEY (liker_id) REFERENCES public.person(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.map_person_person
    ADD CONSTRAINT map_person_person_friend_id_fkey FOREIGN KEY (friend_id) REFERENCES public.person(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.map_person_person_invite
    ADD CONSTRAINT map_person_person_invite_friend_id_fkey FOREIGN KEY (friend_id) REFERENCES public.person(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.map_person_person_invite
    ADD CONSTRAINT map_person_person_invite_person_id_fkey FOREIGN KEY (person_id) REFERENCES public.person(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.map_person_person
    ADD CONSTRAINT map_person_person_person_id_fkey FOREIGN KEY (person_id) REFERENCES public.person(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.mdl_album_image
    ADD CONSTRAINT mdl_album_image_album_id_fkey FOREIGN KEY (album_id) REFERENCES public.mdl_album(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.mdl_album_image
    ADD CONSTRAINT mdl_album_image_owner_id_fkey FOREIGN KEY (owner_id) REFERENCES public.person(id) ON DELETE SET DEFAULT;

ALTER TABLE ONLY public.mdl_album
    ADD CONSTRAINT mdl_album_owner_id_fkey FOREIGN KEY (owner_id) REFERENCES public.person(id) ON DELETE SET DEFAULT;

ALTER TABLE ONLY public.mdl_comment
    ADD CONSTRAINT mdl_comments_author_foreign_key FOREIGN KEY (author_id) REFERENCES public.person(id) ON DELETE SET DEFAULT;

ALTER TABLE ONLY public.mdl_comment
    ADD CONSTRAINT mdl_comments_editor_fk FOREIGN KEY (editor_id) REFERENCES public.person(id);

ALTER TABLE ONLY public.mdl_document
    ADD CONSTRAINT mdl_document_files_checked_out_by_foreign_key FOREIGN KEY (checked_out_by) REFERENCES public.person(id) ON DELETE SET DEFAULT;

ALTER TABLE ONLY public.mdl_document
    ADD CONSTRAINT mdl_document_files_status_changed_by_foreign_key FOREIGN KEY (status_changed_by) REFERENCES public.person(id) ON DELETE SET DEFAULT;

ALTER TABLE ONLY public.mdl_document_version
    ADD CONSTRAINT mdl_document_versions_document_id_foreign_key FOREIGN KEY (document_id) REFERENCES public.mdl_document(id) ON DELETE RESTRICT;

ALTER TABLE ONLY public.mdl_document_version
    ADD CONSTRAINT mdl_document_versions_file_type_foreign_key FOREIGN KEY (file_type_id) REFERENCES public.file_type(id) ON DELETE RESTRICT;

ALTER TABLE ONLY public.mdl_document_version
    ADD CONSTRAINT mdl_document_versions_uploader_foreign_key FOREIGN KEY (uploaded_by) REFERENCES public.person(id) ON DELETE SET DEFAULT;

ALTER TABLE ONLY public.mdl_video_link
    ADD CONSTRAINT mdl_videolink_author_id_fkey FOREIGN KEY (author_id) REFERENCES public.person(id) ON DELETE SET DEFAULT;

ALTER TABLE ONLY public.multi_wiki
    ADD CONSTRAINT multi_wiki_entity_id_fkey FOREIGN KEY (entity_id) REFERENCES public.entity(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.multi_wiki_version_section
    ADD CONSTRAINT multi_wiki_section_version_id_fkey FOREIGN KEY (version_id) REFERENCES public.multi_wiki_version(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.multi_wiki_section_visibility
    ADD CONSTRAINT multi_wiki_section_visibility_fkey FOREIGN KEY (version_section_id) REFERENCES public.multi_wiki_version_section(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.entity_version_components
    ADD CONSTRAINT multi_wiki_version_fkey FOREIGN KEY (multi_wiki_version_id) REFERENCES public.multi_wiki_version(id);

ALTER TABLE ONLY public.multi_wiki_version
    ADD CONSTRAINT multi_wiki_version_wiki_id_fkey FOREIGN KEY (wiki_id) REFERENCES public.multi_wiki(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.news_blog
    ADD CONSTRAINT newsblog_author_foreign_key FOREIGN KEY (author_id) REFERENCES public.person(id) ON DELETE SET DEFAULT;

ALTER TABLE ONLY public.news_blog
    ADD CONSTRAINT newsblog_image_id_foreign_key FOREIGN KEY (image_id) REFERENCES public.mdl_album_image(id) ON DELETE SET DEFAULT;

ALTER TABLE ONLY public.notification
    ADD CONSTRAINT noti_org_fk FOREIGN KEY (originator_id) REFERENCES public.person(id);

ALTER TABLE ONLY public.notification
    ADD CONSTRAINT noti_rec_fk FOREIGN KEY (receiver_id) REFERENCES public.person(id);

ALTER TABLE ONLY public.notification_attribute
    ADD CONSTRAINT notification_id_fk FOREIGN KEY (notification_id) REFERENCES public.notification(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.oauth_token
    ADD CONSTRAINT o_t_client_fk FOREIGN KEY (client_id) REFERENCES public.oauth_client_details(client_id) ON DELETE CASCADE;

ALTER TABLE ONLY public.oauth_token
    ADD CONSTRAINT o_t_person_fk FOREIGN KEY (person_id) REFERENCES public.person(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.open_social_app_data
    ADD CONSTRAINT open_social_app_data_person_id_fkey FOREIGN KEY (person_id) REFERENCES public.person(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.person_external_ids
    ADD CONSTRAINT person_id_fkey FOREIGN KEY (person_id) REFERENCES public.person(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.person
    ADD CONSTRAINT person_language_id_fkey FOREIGN KEY (language_id) REFERENCES public.language(id);

ALTER TABLE ONLY public.person_pending_password_reset
    ADD CONSTRAINT person_pending_password_reset_person_id_fkey FOREIGN KEY (person_id) REFERENCES public.person(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.person_tenant
    ADD CONSTRAINT person_tenant_fk_p FOREIGN KEY (person_id) REFERENCES public.person(id);

ALTER TABLE ONLY public.person_tenant
    ADD CONSTRAINT person_tenant_fk_t FOREIGN KEY (tenant_id) REFERENCES public.tenant(id);

ALTER TABLE ONLY public.person_to_role
    ADD CONSTRAINT person_to_roles_person_id_fkey FOREIGN KEY (person_id) REFERENCES public.person(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.person_to_role
    ADD CONSTRAINT person_to_roles_role_name_fkey FOREIGN KEY (role_name) REFERENCES public.person_role(role_name) ON DELETE RESTRICT;

ALTER TABLE ONLY public.profile_attribute
    ADD CONSTRAINT profile_attribute_person_id_fk FOREIGN KEY (person_id) REFERENCES public.person(id);

ALTER TABLE ONLY public.profile_component
    ADD CONSTRAINT profile_component_person_id_fkey FOREIGN KEY (person_id) REFERENCES public.person(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.push_target_app
    ADD CONSTRAINT push_target_app_person_fk FOREIGN KEY (person_id) REFERENCES public.person(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.raweditor_version
    ADD CONSTRAINT raweditor_version_author_id_fkey FOREIGN KEY (author_id) REFERENCES public.person(id) ON DELETE SET DEFAULT;

ALTER TABLE ONLY public.search_filter
    ADD CONSTRAINT search_filter_person_id_fk FOREIGN KEY (person_id) REFERENCES public.person(id);

ALTER TABLE ONLY public.newszoom_section_channel
    ADD CONSTRAINT section_id_fk FOREIGN KEY (section_id) REFERENCES public.newszoom_section(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.sh_conversation_attachment
    ADD CONSTRAINT sh_conversation_attachment_delete_action_id_fkey FOREIGN KEY (delete_action_id) REFERENCES public.delete_action(id);

ALTER TABLE ONLY public.sh_conversation
    ADD CONSTRAINT sh_conversation_delete_action_id_fkey FOREIGN KEY (delete_action_id) REFERENCES public.delete_action(id);

ALTER TABLE ONLY public.sh_conversation_message
    ADD CONSTRAINT sh_conversation_message_delete_action_id_fkey FOREIGN KEY (delete_action_id) REFERENCES public.delete_action(id);

ALTER TABLE ONLY public.sh_conversation_participant
    ADD CONSTRAINT sh_conversation_participant_delete_action_id_fkey FOREIGN KEY (delete_action_id) REFERENCES public.delete_action(id);

ALTER TABLE ONLY public.sh_conversation_settings
    ADD CONSTRAINT sh_conversation_settings_delete_action_id_fkey FOREIGN KEY (delete_action_id) REFERENCES public.delete_action(id);

ALTER TABLE ONLY public.sh_default_album
    ADD CONSTRAINT sh_default_album_delete_action_id_fkey FOREIGN KEY (delete_action_id) REFERENCES public.delete_action(id);

ALTER TABLE ONLY public.sh_entity_internal_invitation
    ADD CONSTRAINT sh_e_i_i_delete_fk FOREIGN KEY (delete_action_id) REFERENCES public.delete_action(id);

ALTER TABLE ONLY public.sh_entity_member_role
    ADD CONSTRAINT sh_e_m_r_delete_fk FOREIGN KEY (delete_action_id) REFERENCES public.delete_action(id);

ALTER TABLE ONLY public.sh_email_notification_setting
    ADD CONSTRAINT sh_email_notification_setting_delete_action_id_fkey FOREIGN KEY (delete_action_id) REFERENCES public.delete_action(id);

ALTER TABLE ONLY public.sh_entity_component
    ADD CONSTRAINT sh_entity_component_delete_action_id_fkey FOREIGN KEY (delete_action_id) REFERENCES public.delete_action(id);

ALTER TABLE ONLY public.sh_entity_component_open_for
    ADD CONSTRAINT sh_entity_component_open_for_delete_action_id_fkey FOREIGN KEY (delete_action_id) REFERENCES public.delete_action(id);

ALTER TABLE ONLY public.sh_entity
    ADD CONSTRAINT sh_entity_delete_action_id_fkey FOREIGN KEY (delete_action_id) REFERENCES public.delete_action(id);

ALTER TABLE ONLY public.sh_entity_dynamic_attribute
    ADD CONSTRAINT sh_entity_dynamic_attribute_delete_action_id_fkey FOREIGN KEY (delete_action_id) REFERENCES public.delete_action(id);

ALTER TABLE ONLY public.sh_entity_link
    ADD CONSTRAINT sh_entity_link_delete_action_id_fkey FOREIGN KEY (delete_action_id) REFERENCES public.delete_action(id);

ALTER TABLE ONLY public.sh_entity_privacy
    ADD CONSTRAINT sh_entity_privacy_delete_action_id_fkey FOREIGN KEY (delete_action_id) REFERENCES public.delete_action(id);

ALTER TABLE ONLY public.sh_entity_recommendation
    ADD CONSTRAINT sh_entity_recommendation_delete_action_id_fkey FOREIGN KEY (delete_action_id) REFERENCES public.delete_action(id);

ALTER TABLE ONLY public.sh_entity_to_entity_type_priva
    ADD CONSTRAINT sh_entity_to_entity_type_priva_delete_action_id_fkey FOREIGN KEY (delete_action_id) REFERENCES public.delete_action(id);

ALTER TABLE ONLY public.sh_entity_version_components
    ADD CONSTRAINT sh_entity_version_components_delete_action_id_fkey FOREIGN KEY (delete_action_id) REFERENCES public.delete_action(id);

ALTER TABLE ONLY public.sh_entity_version
    ADD CONSTRAINT sh_entity_version_delete_action_id_fkey FOREIGN KEY (delete_action_id) REFERENCES public.delete_action(id);

ALTER TABLE ONLY public.sh_entity_version_review
    ADD CONSTRAINT sh_entity_version_review_delete_action_id_fkey FOREIGN KEY (delete_action_id) REFERENCES public.delete_action(id);

ALTER TABLE ONLY public.sh_entity_waitlist
    ADD CONSTRAINT sh_entity_waitlist_delete_action_id_fkey FOREIGN KEY (delete_action_id) REFERENCES public.delete_action(id);

ALTER TABLE ONLY public.sh_external_invitation
    ADD CONSTRAINT sh_external_invitation_delete_action_id_fkey FOREIGN KEY (delete_action_id) REFERENCES public.delete_action(id);

ALTER TABLE ONLY public.sh_like_table
    ADD CONSTRAINT sh_like_table_delete_action_id_fkey FOREIGN KEY (delete_action_id) REFERENCES public.delete_action(id);

ALTER TABLE ONLY public.sh_map_person_person
    ADD CONSTRAINT sh_map_person_person_delete_action_id_fkey FOREIGN KEY (delete_action_id) REFERENCES public.delete_action(id);

ALTER TABLE ONLY public.sh_map_person_person_invite
    ADD CONSTRAINT sh_map_person_person_invite_delete_action_id_fkey FOREIGN KEY (delete_action_id) REFERENCES public.delete_action(id);

ALTER TABLE ONLY public.sh_mdl_album
    ADD CONSTRAINT sh_mdl_album_delete_action_id_fkey FOREIGN KEY (delete_action_id) REFERENCES public.delete_action(id);

ALTER TABLE ONLY public.sh_mdl_album_image
    ADD CONSTRAINT sh_mdl_album_image_delete_action_id_fkey FOREIGN KEY (delete_action_id) REFERENCES public.delete_action(id);

ALTER TABLE ONLY public.sh_mdl_comment
    ADD CONSTRAINT sh_mdl_comment_delete_action_id_fkey FOREIGN KEY (delete_action_id) REFERENCES public.delete_action(id);

ALTER TABLE ONLY public.sh_mdl_document
    ADD CONSTRAINT sh_mdl_document_delete_action_id_fkey FOREIGN KEY (delete_action_id) REFERENCES public.delete_action(id);

ALTER TABLE ONLY public.sh_mdl_document_version
    ADD CONSTRAINT sh_mdl_document_version_delete_action_id_fkey FOREIGN KEY (delete_action_id) REFERENCES public.delete_action(id);

ALTER TABLE ONLY public.sh_mdl_video_link
    ADD CONSTRAINT sh_mdl_video_link_delete_action_id_fkey FOREIGN KEY (delete_action_id) REFERENCES public.delete_action(id);

ALTER TABLE ONLY public.sh_multi_wiki
    ADD CONSTRAINT sh_multi_wiki_delete_action_id_fkey FOREIGN KEY (delete_action_id) REFERENCES public.delete_action(id);

ALTER TABLE ONLY public.sh_multi_wiki_section_visibili
    ADD CONSTRAINT sh_multi_wiki_section_visibili_delete_action_id_fkey FOREIGN KEY (delete_action_id) REFERENCES public.delete_action(id);

ALTER TABLE ONLY public.sh_multi_wiki_version
    ADD CONSTRAINT sh_multi_wiki_version_delete_action_id_fkey FOREIGN KEY (delete_action_id) REFERENCES public.delete_action(id);

ALTER TABLE ONLY public.sh_multi_wiki_version_section
    ADD CONSTRAINT sh_multi_wiki_version_section_delete_action_id_fkey FOREIGN KEY (delete_action_id) REFERENCES public.delete_action(id);

ALTER TABLE ONLY public.sh_news_blog
    ADD CONSTRAINT sh_news_blog_delete_action_id_fkey FOREIGN KEY (delete_action_id) REFERENCES public.delete_action(id);

ALTER TABLE ONLY public.sh_person
    ADD CONSTRAINT sh_person_delete_action_id_fkey FOREIGN KEY (delete_action_id) REFERENCES public.delete_action(id);

ALTER TABLE ONLY public.sh_person_external_ids
    ADD CONSTRAINT sh_person_external_ids_delete_action_id_fkey FOREIGN KEY (delete_action_id) REFERENCES public.delete_action(id);

ALTER TABLE ONLY public.sh_person_to_role
    ADD CONSTRAINT sh_person_to_role_delete_action_id_fkey FOREIGN KEY (delete_action_id) REFERENCES public.delete_action(id);

ALTER TABLE ONLY public.sh_profile_attribute
    ADD CONSTRAINT sh_profile_attribute_delete_action_id_fk FOREIGN KEY (delete_action_id) REFERENCES public.delete_action(id);

ALTER TABLE ONLY public.sh_profile_component
    ADD CONSTRAINT sh_profile_component_delete_action_id_fkey FOREIGN KEY (delete_action_id) REFERENCES public.delete_action(id);

ALTER TABLE ONLY public.sh_raweditor_version
    ADD CONSTRAINT sh_raweditor_version_delete_action_id_fkey FOREIGN KEY (delete_action_id) REFERENCES public.delete_action(id);

ALTER TABLE ONLY public.sh_subscriptions
    ADD CONSTRAINT sh_subscriptions_delete_action_id_fkey FOREIGN KEY (delete_action_id) REFERENCES public.delete_action(id);

ALTER TABLE ONLY public.sh_survey
    ADD CONSTRAINT sh_survey_delete_action_id_fkey FOREIGN KEY (delete_action_id) REFERENCES public.delete_action(id);

ALTER TABLE ONLY public.sh_survey_participant_answer
    ADD CONSTRAINT sh_survey_participant_answer_delete_action_id_fkey FOREIGN KEY (delete_action_id) REFERENCES public.delete_action(id);

ALTER TABLE ONLY public.sh_survey_predefined_answer
    ADD CONSTRAINT sh_survey_predefined_answer_delete_action_id_fkey FOREIGN KEY (delete_action_id) REFERENCES public.delete_action(id);

ALTER TABLE ONLY public.sh_survey_question
    ADD CONSTRAINT sh_survey_question_delete_action_id_fkey FOREIGN KEY (delete_action_id) REFERENCES public.delete_action(id);

ALTER TABLE ONLY public.sh_task_list
    ADD CONSTRAINT sh_t_l_delete_action_id_fk FOREIGN KEY (delete_action_id) REFERENCES public.delete_action(id);

ALTER TABLE ONLY public.sh_task_list_permissions
    ADD CONSTRAINT sh_t_l_p_delete_action_id_fk FOREIGN KEY (delete_action_id) REFERENCES public.delete_action(id);

ALTER TABLE ONLY public.sh_tasks
    ADD CONSTRAINT sh_tasks_delete_action_id_fk FOREIGN KEY (delete_action_id) REFERENCES public.delete_action(id);

ALTER TABLE ONLY public.sh_user_setting
    ADD CONSTRAINT sh_user_setting_delete_action_id_fkey1 FOREIGN KEY (delete_action_id) REFERENCES public.delete_action(id);

ALTER TABLE ONLY public.sh_wiki_editor_backup
    ADD CONSTRAINT sh_wiki_editor_backup_delete_action_id_fkey FOREIGN KEY (delete_action_id) REFERENCES public.delete_action(id);

ALTER TABLE ONLY public.sh_wiki_editor
    ADD CONSTRAINT sh_wiki_editor_delete_action_id_fkey FOREIGN KEY (delete_action_id) REFERENCES public.delete_action(id);

ALTER TABLE ONLY public.sh_wiki_editor_version
    ADD CONSTRAINT sh_wiki_editor_version_delete_action_id_fkey FOREIGN KEY (delete_action_id) REFERENCES public.delete_action(id);

ALTER TABLE ONLY public.sh_workstream_attachment
    ADD CONSTRAINT sh_workstream_attachment_delete_action_id_fkey FOREIGN KEY (delete_action_id) REFERENCES public.delete_action(id);

ALTER TABLE ONLY public.sh_workstream_entry_attribute
    ADD CONSTRAINT sh_workstream_entry_attribute_delete_action_id_fkey FOREIGN KEY (delete_action_id) REFERENCES public.delete_action(id);

ALTER TABLE ONLY public.sh_workstream_entry
    ADD CONSTRAINT sh_workstream_entry_delete_action_id_fkey FOREIGN KEY (delete_action_id) REFERENCES public.delete_action(id);

ALTER TABLE ONLY public.subscriptions
    ADD CONSTRAINT subscriptions_subscriber_id_fkey FOREIGN KEY (subscriber_id) REFERENCES public.person(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.survey
    ADD CONSTRAINT survey_owner_id_fkey FOREIGN KEY (owner_id) REFERENCES public.person(id) ON DELETE SET DEFAULT;

ALTER TABLE ONLY public.survey_participant_answer
    ADD CONSTRAINT survey_participant_freetext_answer_question_id_fkey FOREIGN KEY (survey_question_id) REFERENCES public.survey_question(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.survey_participant_answer
    ADD CONSTRAINT survey_participant_multiple_choice_answer_answer_id_foreign_key FOREIGN KEY (survey_predefined_answer_id) REFERENCES public.survey_predefined_answer(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.survey_participant_answer
    ADD CONSTRAINT survey_participant_multiple_choice_answer_participant_id_fkey FOREIGN KEY (participant_id) REFERENCES public.person(id) ON DELETE SET DEFAULT;

ALTER TABLE ONLY public.survey_predefined_answer
    ADD CONSTRAINT survey_predefined_multiple_choice_answer_question_id_fkey FOREIGN KEY (survey_question_id) REFERENCES public.survey_question(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.survey_question
    ADD CONSTRAINT survey_question_survey_id_fkey FOREIGN KEY (survey_id) REFERENCES public.survey(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.sync_app
    ADD CONSTRAINT sync_app_person_id_fk FOREIGN KEY (person_id) REFERENCES public.person(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.task_filter_order
    ADD CONSTRAINT task_filter_order_task_id_fk FOREIGN KEY (task_id) REFERENCES public.tasks(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.tasks
    ADD CONSTRAINT task_list_id_fkey FOREIGN KEY (task_list_id) REFERENCES public.task_list(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.task_list_permissions
    ADD CONSTRAINT task_list_permissions_task_list_id_fkey FOREIGN KEY (task_list_id) REFERENCES public.task_list(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.tasks
    ADD CONSTRAINT tasks_parent_task_id_fkey FOREIGN KEY (parent_task_id) REFERENCES public.tasks(id);

ALTER TABLE ONLY public.tenant_app_visibility
    ADD CONSTRAINT tenant_app_visibility_fk FOREIGN KEY (app_id) REFERENCES public.app(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.newszoom_section
    ADD CONSTRAINT tenant_id_fk FOREIGN KEY (tenant_id) REFERENCES public.tenant(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.tenant_theme
    ADD CONSTRAINT tenant_theme_tenant_id_fkey FOREIGN KEY (tenant_id) REFERENCES public.tenant(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.theme_attribute
    ADD CONSTRAINT theme_element_theme_fk FOREIGN KEY (theme_id) REFERENCES public.theme(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.translation
    ADD CONSTRAINT translation_language_id_fkey FOREIGN KEY (language_id) REFERENCES public.language(id);

ALTER TABLE ONLY public.translation
    ADD CONSTRAINT translation_native_id_fkey FOREIGN KEY (native_id) REFERENCES public.native(id);

ALTER TABLE ONLY public.home_screen_apps
    ADD CONSTRAINT user_app_setting_person_id_fk FOREIGN KEY (person_id) REFERENCES public.person(id);

ALTER TABLE ONLY public.user_setting
    ADD CONSTRAINT user_setting_person_id_fkey FOREIGN KEY (person_id) REFERENCES public.person(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.wiki_editor_backup
    ADD CONSTRAINT wiki_editor_backup_id_foreign_key FOREIGN KEY (wiki_editor_id) REFERENCES public.wiki_editor(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.wiki_editor_version
    ADD CONSTRAINT wiki_editor_version_wiki_editor_id_foreign_key FOREIGN KEY (wiki_editor_id) REFERENCES public.wiki_editor(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.wiki_editor_backup
    ADD CONSTRAINT wiki_page_backup_author_foreign_key FOREIGN KEY (author_id) REFERENCES public.person(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.wiki_editor_version
    ADD CONSTRAINT wiki_page_version_author_foreign_key FOREIGN KEY (author_id) REFERENCES public.person(id) ON DELETE SET DEFAULT;

ALTER TABLE ONLY public.workstream_attachment
    ADD CONSTRAINT workstream_attachments_entry_id_fkey FOREIGN KEY (entry_id) REFERENCES public.workstream_entry(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.workstream_entry_attribute
    ADD CONSTRAINT workstream_entry_attributes_entry_id_fkey FOREIGN KEY (entry_id) REFERENCES public.workstream_entry(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.scheduled_workstream_entry
    ADD CONSTRAINT workstream_entry_fk FOREIGN KEY (workstream_entry_id) REFERENCES public.workstream_entry(id) ON DELETE CASCADE;

ALTER TABLE ONLY public.workstream_pinned_hide_date
    ADD CONSTRAINT workstream_pinned_hide_date_person_id_fk FOREIGN KEY (person_id) REFERENCES public.person(id);

ALTER TABLE ONLY public.workstream_read
    ADD CONSTRAINT workstream_read_pid_fk FOREIGN KEY (person_id) REFERENCES public.person(id);

