/*
 * Decompiled with CFR 0.152.
 */
package de.justsoftware.chat.api.service;

import com.google.common.base.MoreObjects;
import de.justsoftware.chat.api.service.ClientVersion;
import de.justsoftware.chat.api.service.ClientVersionCheckResult;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.annotation.CheckForNull;
import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.servlet.http.HttpServletRequest;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

@Service
@ParametersAreNonnullByDefault
public class ClientVersionsService {
    public static final long API_V2_ANDROID_MIN_APP_VERSION_CODE = 30000L;
    public static final long API_V2_IOS_MIN_APP_VERSION_CODE = 3000000L;
    private static final Logger LOG = LoggerFactory.getLogger(ClientVersionsService.class);
    private static final Pattern ANDROID_VERSION_CODE_PATTERN = Pattern.compile("\\bJustConnect/(\\d+)\\b.*\\(Android/.*");
    private static final Pattern IOS_VERSION_CODE_PATTERN = Pattern.compile("\\bJustConnect/(\\d+\\.\\d+(?:\\.\\d+)?) Build/\\d+ iOS/");
    private static final Pattern IOS_APP_VERSION_PATTERN = Pattern.compile("(\\d+)\\.(\\d+)(?:\\.(\\d+))?");

    @Nonnull
    public ClientVersionCheckResult checkVersion(ClientVersion version) {
        switch (version.getPlatform()) {
            case ANDROID: {
                if (version.getVersionCode() >= 30000L) {
                    return ClientVersionCheckResult.VALID_VERSION_CODE_CHECK_RESULT;
                }
                String androidErrorMessage = ClientVersionsService.errorMessage(ClientVersion.Platform.ANDROID, version.getVersionCode(), 30000L);
                return new ClientVersionCheckResult(true, androidErrorMessage);
            }
            case IOS: {
                if (version.getVersionCode() >= 3000000L) {
                    return ClientVersionCheckResult.VALID_VERSION_CODE_CHECK_RESULT;
                }
                String iosErrorMessage = ClientVersionsService.errorMessage(ClientVersion.Platform.IOS, version.getVersionCode(), 3000000L);
                return new ClientVersionCheckResult(true, iosErrorMessage);
            }
            case UNKNOWN: {
                return ClientVersionCheckResult.VALID_VERSION_CODE_CHECK_RESULT;
            }
        }
        return ClientVersionCheckResult.VALID_VERSION_CODE_CHECK_RESULT;
    }

    @Nonnull
    private static String errorMessage(ClientVersion.Platform platform, long versionCode, long minimumVersionCode) {
        return String.format("The app version code is too low. Found version %d, min version for %s is %d", versionCode, platform.displayName(), minimumVersionCode);
    }

    @Nonnull
    public static ClientVersion getVersion(HttpServletRequest request) {
        String userAgent = request.getHeader("User-Agent");
        if (userAgent == null) {
            return ClientVersion.UNKNOWN_PLATFORM_VERSION;
        }
        String androidVersionString = ClientVersionsService.findVersionCode(userAgent, ANDROID_VERSION_CODE_PATTERN);
        if (androidVersionString != null) {
            return new ClientVersion(ClientVersion.Platform.ANDROID, ClientVersionsService.androidVersion(androidVersionString));
        }
        String iosVersionString = ClientVersionsService.findVersionCode(userAgent, IOS_VERSION_CODE_PATTERN);
        if (iosVersionString != null) {
            return new ClientVersion(ClientVersion.Platform.IOS, ClientVersionsService.iosVersion(iosVersionString));
        }
        return ClientVersion.UNKNOWN_PLATFORM_VERSION;
    }

    private static long androidVersion(String versionCode) {
        try {
            return Long.parseLong(versionCode);
        }
        catch (NumberFormatException versionParseException) {
            LOG.warn("Version code \"" + versionCode + "\" cannot be parsed.");
            return -1L;
        }
    }

    private static long iosVersion(String versionCode) {
        Matcher matcher = IOS_APP_VERSION_PATTERN.matcher(versionCode);
        if (!matcher.matches() || matcher.groupCount() < 3) {
            LOG.warn("Version code \"" + versionCode + "\" cannot be parsed.");
            return -1L;
        }
        String majorVersionString = matcher.group(1);
        String minorVersionString = matcher.group(2);
        String bugfixVersionString = (String)MoreObjects.firstNonNull((Object)matcher.group(3), (Object)"0");
        try {
            long version = Long.parseLong(majorVersionString);
            version *= 1000L;
            version += Long.parseLong(minorVersionString);
            version *= 1000L;
            return version += Long.parseLong(bugfixVersionString);
        }
        catch (NumberFormatException versionParseException) {
            LOG.warn("Version code \"" + versionCode + "\" cannot be parsed.");
            return -1L;
        }
    }

    @CheckForNull
    private static String findVersionCode(String userAgent, Pattern pattern) {
        Matcher matcher = pattern.matcher(userAgent);
        boolean found = matcher.find();
        return found ? matcher.group(1) : null;
    }
}

