#!/usr/bin/python

# Called by ejabberd to authenticate users
# See: https://git.process-one.net/ejabberd/mainline/blobs/raw/2.1.x/doc/dev.html#htoc9

import os
import os.path
import sys
from struct import *
import logging
import urllib

def from_ejabberd():
    input_length = sys.stdin.read(2)
    (size,) = unpack('>h', input_length)
    return sys.stdin.read(size).split(':')

def to_ejabberd(bool):
    answer = 0
    if bool:
        answer = 1
    token = pack('>hh', 2, answer)
    sys.stdout.write(token)
    sys.stdout.flush()

def doRestBooleanCall(cfg, path, params):
    return urllib.urlopen(cfg["RESTSERVER_URL"] + path, urllib.urlencode(params)).read() == "true"

def auth(cfg, username, server, password):
    logging.debug("ask auth %s %s ********", username, server)
    result = doRestBooleanCall(cfg, "/internal/chat/auth", {"user": username, "server": server, "password": password})
    logging.info("auth %s %s => %s", username, server, result)
    return result

def isuser(cfg, username, server):
    logging.debug("ask isuser %s %s", username, server)
    return username == "xmpp-system-user" or username.isdigit()

def setpass(cfg, username, server, password):
    return False # We do not support setting the password

def get_run_dir():
    """Return the path to this script."""
    return os.path.dirname(os.path.abspath(sys.argv[0])) + os.sep

def load_cfg():
    result = {}
    f = open(os.path.join(get_run_dir(), "chat.properties"))
    for line in f:
        line = line.strip()
        if line != "" and not line.startswith("#"):
            [key, val] = line.split("=")
            result[key] = val
    return result

def main():
    cfg = load_cfg()

    # create log dir
    try:
        os.makedirs(os.path.dirname(cfg["EXT_AUTH_LOG_FILE"]))
    except OSError:
        pass # ignore "File exists"

    # init logging
    logging.basicConfig(
        format='%(asctime)s %(message)s',
        filename=cfg["EXT_AUTH_LOG_FILE"],
        filemode="a",
        level=logging.INFO
        )

    logging.info("Start in %s", get_run_dir())

    while True:
        logging.debug("loop start")
        data = from_ejabberd()
        logging.debug("got data")
        success = False
        try:
            if data[0] == "auth":
                success = auth(cfg, data[1], data[2], data[3])
            elif data[0] == "isuser":
                success = isuser(cfg, data[1], data[2])
            elif data[0] == "setpass":
                success = setpass(cfg, data[1], data[2], data[3])
        except Exception, err:
            # Never ever die
            logging.error(str(err))
        logging.debug("success=%s", success)
        to_ejabberd(success)
        logging.debug("loop end")

main()
