#!/usr/bin/env bash


help() {
    echo "Usage $0 [db] [full]"
    echo "Database-backup: db=pg or db=ora, default = pg"
    echo "If you want to explicitely trigger a full duplicity backup to ftp space, then add the parameter \"full\""
    echo "If you add no parameter an incremental backup will be triggered unless the last full backup is older than 2 weeks"

}

# Displays an error message (first parameter) and exits
# the backup script with error code 1
error() {
    time=$(date '+%Y-%m-%d %T')
    echo "[ERROR] $time $1">&2
    status "ERROR"
    exit 1
}

info() {
    time=$(date '+%Y-%m-%d %T')
    echo "[INFO] $time $1"
}

warn() {
    time=$(date '+%Y-%m-%d %T')
    echo "[WARN] $time $1">&2
    status "WARN"
}

status() {
    time=$(date '+%Y-%m-%d %T %z')
    echo "[$1] $time" > $BACKUP_STATUS_LOG_FILE
}

# Reads the main properties from backup.properties file
# and sets addtional properties
setProperties() {

    SCRIPT_DIR=$(dirname $0)
    property_file=$SCRIPT_DIR/backup.properties
    info "Reading main properties from $property_file"
    . $property_file
    info "Setting additional properties"
    BACKUP_FILE_SUFFIX=$(date +%Y-%m-%d-%H%M%S)
    DB_BACKUP_DIR=$BACKUP_MAIN_DIR/db
    JUST_DB_BACKUP_FILE=$DB_BACKUP_DIR/$JUST_DB_NAME-$BACKUP_FILE_SUFFIX
    DRIVE_DB_BACKUP_FILE=$DB_BACKUP_DIR/$DRIVE_DB_NAME-$BACKUP_FILE_SUFFIX
    PEOPLE_DB_BACKUP_FILE=$DB_BACKUP_DIR/$PEOPLE_DB_NAME-$BACKUP_FILE_SUFFIX
    IMPORT_DB_BACKUP_FILE=$DB_BACKUP_DIR/$IMPORT_DB_NAME-$BACKUP_FILE_SUFFIX
    NOTIFICATION_DB_BACKUP_FILE=$DB_BACKUP_DIR/$NOTIFICATION_DB_NAME-$BACKUP_FILE_SUFFIX
    WIKI_DB_BACKUP_FILE=$DB_BACKUP_DIR/$WIKI_DB_NAME-$BACKUP_FILE_SUFFIX
    CHAT_DB_BACKUP_FILE=$DB_BACKUP_DIR/$CHAT_DB_NAME-$BACKUP_FILE_SUFFIX
    STORAGESERVER_BACKUP_DIR=$BACKUP_MAIN_DIR/storageserver
    WWW_BACKUP_DIR=$BACKUP_MAIN_DIR/www
    APACHE_BACKUP_DIR=$BACKUP_MAIN_DIR/apache2
    JUSTCONNECT_CONF_BACKUP_DIR=$BACKUP_MAIN_DIR/justconnect_conf
    LOG_BACKUP_DIR=$BACKUP_MAIN_DIR/logs
    ZOOKEEPER_HELP_DIR=$BACKUP_HELP_DIR/zookeeper
    ZOOKEEPER_BACKUP_DIR=$BACKUP_MAIN_DIR/zookeeper
    ZOOKEEPER_BACKUP_FILE=$ZOOKEEPER_BACKUP_DIR/zookeeper-$BACKUP_FILE_SUFFIX.tar.gz
}

mountSnapShot() {
    name=$1
    [ ! -e $1 ] || error "$1 does not exist"
    size=$(vgs --rows --units m   helium | grep VFree | cut -d ' ' -f 4)
    info "Creating snapshot $name with size of $size"
    lvcreate --snapshot --permission r --name backup --size $size $1 || error "Creating snapshot $1 failed"

}

# Creates a database dump and in case of an error exits the backup
dumpDatabase() {
    mkdir -p "$DB_BACKUP_DIR" || error "Could not create $DB_BACKUP_DIR"
    info "Creating database dump ..."
    if [ "$db_type" = "pg" ]; then
        if [ -f "$HOME_JS"/.pgpass ]; then
            if dpkg -s just-databaseserver > /dev/null 2>&1; then
                info "Database = Postgres JUST"
                pg_dump --format custom -v -h "$DB_HOST" -p "$DB_PORT" -U "$JUST_DB_USER" -f "$JUST_DB_BACKUP_FILE" "$JUST_DB_NAME" || error "Failed to dump database into $JUST_DB_BACKUP_FILE."
                chmod 600 "$JUST_DB_BACKUP_FILE"
            fi

            if dpkg -s just-drive-database > /dev/null 2>&1; then
                info "Database = Postgres DRIVE"
                pg_dump --format custom -v -h "$DB_HOST" -p "$DB_PORT" -U "$DRIVE_DB_USER" -f "$DRIVE_DB_BACKUP_FILE" "$DRIVE_DB_NAME" || error "Failed to dump database into $DRIVE_DB_BACKUP_FILE."
                chmod 600 "$DRIVE_DB_BACKUP_FILE"
            fi

            if dpkg -s just-people-database > /dev/null 2>&1; then
                info "Database = Postgres PEOPLE"
                pg_dump --format custom -v -h "$DB_HOST" -p "$DB_PORT" -U "$PEOPLE_DB_USER" -f "$PEOPLE_DB_BACKUP_FILE" "$PEOPLE_DB_NAME" || error "Failed to dump database into $PEOPLE_DB_BACKUP_FILE."
                chmod 600 "$PEOPLE_DB_BACKUP_FILE"
            fi

            if dpkg -s just-import-database > /dev/null 2>&1; then
                info "Database = Postgres IMPORT"
                pg_dump --format custom -v -h "$DB_HOST" -p "$DB_PORT" -U "$IMPORT_DB_USER" -f "$IMPORT_DB_BACKUP_FILE" "$IMPORT_DB_NAME" || error "Failed to dump database into $IMPORT_DB_BACKUP_FILE."
                chmod 600 "$IMPORT_DB_BACKUP_FILE"
            fi

            if dpkg -s just-notification-database > /dev/null 2>&1; then
                info "Database = Postgres NOTIFICATION"
                pg_dump --format custom -v -h "$DB_HOST" -p "$DB_PORT" -U "$NOTIFICATION_DB_USER" -f "$NOTIFICATION_DB_BACKUP_FILE" "$NOTIFICATION_DB_NAME" || error "Failed to dump database into $NOTIFICATION_DB_BACKUP_FILE."
                chmod 600 "$NOTIFICATION_DB_BACKUP_FILE"
            fi

            if dpkg -s just-wiki-database > /dev/null 2>&1; then
                info "Database = Postgres WIKI"
                pg_dump --format custom -v -h "$DB_HOST" -p "$DB_PORT" -U "$WIKI_DB_USER" -f "$WIKI_DB_BACKUP_FILE" "$WIKI_DB_NAME" || error "Failed to dump database into $WIKI_DB_BACKUP_FILE."
                chmod 600 "$WIKI_DB_BACKUP_FILE"
            fi

            if dpkg -s just-chat-database > /dev/null 2>&1; then
                info "Database = Postgres CHAT"
                pg_dump --format custom -v -h "$DB_HOST" -p "$DB_PORT" -U "$CHAT_DB_USER" -f "$CHAT_DB_BACKUP_FILE" "$CHAT_DB_NAME" || error "Failed to dump database into $CHAT_DB_BACKUP_FILE."
                chmod 600 "$CHAT_DB_BACKUP_FILE"
            fi
        fi
    fi
    info "Successfully created database dump."
}

# Backups the log files for Apache, Exim, Tomcat, Just Connect, Solr
backupAllLogFiles() {
    log_files_tmp=/tmp/log-files
    getLogFiles $APACHE_LOG_FILES_DIR > $log_files_tmp
    getLogFiles $EXIM_LOG_FILES_DIR >> $log_files_tmp
    getLogFiles $TOMCAT_LOG_FILES_DIR >> $log_files_tmp
    getLogFiles $TOMCAT_LOG_FILES_DIR >> $log_files_tmp
    getLogFiles $HOME_JS/logs >> $log_files_tmp
    getLogFiles $HOME_JS/search/logs >> $log_files_tmp
    getLogFiles $JUST_LOG_DIR >> $log_files_tmp
    if [ -s $log_files_tmp ]; then
        info "Creating backup of log files..."
        rsync -avz --files-from $log_files_tmp / $LOG_BACKUP_DIR || warn "Could not backup log files to $LOG_BACKUP_DIR"
    fi
}

# Finds the log files that are newer than three days, used by backupAllLogFiles
getLogFiles() {
    log_files_dir=$1
    if [ -n "$log_files_dir" ]; then
        find $log_files_dir -name '*log*' -mtime -3 || warn "Could not get log files from $log_files_dir"
    fi
}

# Creates a backup for the storageserver, www, Apache and
# all log files
backupFiles() {
    info "Starting backup of files ..."
    backupFile $STORAGESERVER_DIR/ $STORAGESERVER_BACKUP_DIR/
    backupFile $WWW_DIR/ $WWW_BACKUP_DIR/
    backupFile $APACHE_DIR/ $APACHE_BACKUP_DIR/
    backupFile $JUSTCONNECT_CONF_DIR/ $JUSTCONNECT_CONF_BACKUP_DIR/
    if [ -d $HOME_JS/kafkabeat/ ]; then
        backupFile $HOME_JS/kafkabeat/ $BACKUP_MAIN_DIR/kafkabeat/
    fi

    backupFileSparseIgnoreErrors $ZOOKEEPER_DATA_DIR/ $ZOOKEEPER_HELP_DIR/ $ZOOKEEPER_BACKUP_FILE

# In general we do not want any log files in our backup. Leaving this for customized Backups
#    backupAllLogFiles
    info "Backup of files done."
}

backupFile() {
    src=$1
    dest=$2
    if [ -n "$src" ]; then
        if [ -d $src ]; then
            info "Creating backup of $src to $dest"
            rsync -av --delete $src $dest || error "Error occured while creating backup of $src to $dest"
        fi
    fi
}

backupFileSparseIgnoreErrors() {
    src=$1
    hlp=$2
    dest=$3
    if [ -n "$src" ]; then
        if [ -d $src ]; then
            info "Creating backup of $src to $dest"
            mkdir -p $hlp
            rsync -avS --delete $src $hlp || true
            mkdir -p "$(dirname $dest)" || error "Could not create $dest"
            tar cvzfS $dest $hlp
        fi
    fi
}

# Checks if files only exist in source
# Only a warning will be triggered since having more files on backup destination is not that critical
checkFilesOnlyInSource() {
    source=$1
    destination=$2
    # Find files that only exist on source
    files=$(rsync -avz --omit-dir-times --existing --ignore-existing --dry-run $source $destination | grep -v "^sending incremental file list" | grep -v "^building file list ... done" | grep -v "^sent .* bytes.*received .* bytes .* bytes/sec" | grep -v "^total size is.*speedup is.*");
    files=$(echo -n "$files")
    if [ -n "$files" ]; then
        error "File backup failed: $files where not copied from $source to $destination"
    fi
    info "All files were copied from $source to $destination"
}

# Checks if files only exist in destination
checkFilesOnlyInDestination() {
    source=$1
    destination=$2
    # Find files that only exist on destination
    number=$(rsync -avz --omit-dir-times --delete --dry-run $source $destination | grep "^deleting" | wc -l)

    [ $number -gt 0 ] && warn "File backup warning: $number files only exists in $destination, but not in $source"
    info "All files in $destination exist also in $source"
}

# Verifies the backup files
verifyFiles() {
    info "Verifying backup ..."
    checkFilesOnlyInSource $STORAGESERVER_DIR $STORAGESERVER_BACKUP_DIR
    checkFilesOnlyInDestination $STORAGESERVER_DIR $STORAGESERVER_BACKUP_DIR
    checkFilesOnlyInSource $WWW_DIR $WWW_BACKUP_DIR
    checkFilesOnlyInDestination $WWW_DIR $WWW_BACKUP_DIR
    checkFilesOnlyInSource $APACHE_DIR $APACHE_BACKUP_DIR
    checkFilesOnlyInDestination $APACHE_DIR $APACHE_BACKUP_DIR
    checkFilesOnlyInSource $JUSTCONNECT_CONF_DIR $JUSTCONNECT_CONF_BACKUP_DIR
    checkFilesOnlyInDestination $JUSTCONNECT_CONF_DIR $JUSTCONNECT_CONF_BACKUP_DIR

    info "Backup is valid."
}

transferBackup() {
    if [ -n "$FTP_SERVER" ]; then

        duplicity_option="--full-if-older-than 2W"

        if [ "$2" = "full" ]; then
            duplicity_option="full"
        fi
        info "Start creating a backup with option \"$duplicity_option\" to $FTP_SERVER"

        ncftp -u $FTP_USERNAME -p $FTP_PASSWORD ftp://$FTP_SERVER << EOF
mkdir $SERVER
EOF

        FTP_PASSWORD=$FTP_PASSWORD PASSPHRASE=$PASSPHRASE duplicity $duplicity_option $BACKUP_MAIN_DIR ncftp+ftp://$FTP_USERNAME@$FTP_SERVER/$SERVER || error "Could not transfer backup to $FTP_SERVER"
        info "Transfer done."
    fi
    if [ -d /mnt/juststoragespace ]; then
        mount /mnt/juststoragespace || error "Error mounting storagespace"
        mountpoint -q /mnt/juststoragespace || "Storagespace not mounted"
        info "Start rdiff-backup"
        rdiff-backup $BACKUP_MAIN_DIR /mnt/juststoragespace/$SERVER/ || error "Error occured while copying backup to storagespace"
        umount /mnt/juststoragespace || error "Error unmounting storagespace"
    fi
}

# Removes old backups of storageserver, www, apache, log files and
# database dumps
removeOldBackups() {
    info "Starting removal of old backup ..."
    info "Removing old backups from $DB_BACKUP_DIR"

    find $DB_BACKUP_DIR -name "$JUST_DB_NAME-*" -mtime "$BACKUP_ROTATE_DAYS" -print0 | xargs --null rm -rf
    find $DB_BACKUP_DIR -name "$DRIVE_DB_NAME-*" -mtime "$BACKUP_ROTATE_DAYS" -print0 | xargs --null rm -rf
    find $DB_BACKUP_DIR -name "$PEOPLE_DB_NAME-*" -mtime "$BACKUP_ROTATE_DAYS" -print0 | xargs --null rm -rf
    find $DB_BACKUP_DIR -name "$IMPORT_DB_NAME-*" -mtime "$BACKUP_ROTATE_DAYS" -print0 | xargs --null rm -rf
    find $DB_BACKUP_DIR -name "$NOTIFICATION_DB_NAME-*" -mtime "$BACKUP_ROTATE_DAYS" -print0 | xargs --null rm -rf
    find $DB_BACKUP_DIR -name "$WIKI_DB_NAME-*" -mtime "$BACKUP_ROTATE_DAYS" -print0 | xargs --null rm -rf
    info "Removing old backups from $ZOOKEEPER_BACKUP_DIR"
    find $ZOOKEEPER_BACKUP_DIR -name "zookeeper-*" -mtime "$BACKUP_ZOOKEEPER_ROTATE_DAYS" -print0 | xargs --null rm -rf
    info "Removal of old backup done."
}

# Removes the old backups from ftp space
removeOldRemoteBackups() {
    if [ -n "$FTP_SERVER" ]; then
        info "Start removing backups from $FTP_SERVER older than $BACKUP_ROTATE_WEEKS weeks"

        FTP_PASSWORD=$FTP_PASSWORD PASSPHRASE=$PASSPHRASE duplicity remove-older-than "$BACKUP_ROTATE_WEEKS"W --force ftp://$FTP_USERNAME@$FTP_SERVER/$SERVER || warn "Could not remove old backups from $FTP_SERVER"
        info "Removal of old remote backups done."
        info "Cleaning extraneous remote files."
        FTP_PASSWORD=$FTP_PASSWORD PASSPHRASE=$PASSPHRASE duplicity cleanup --force ftp://$FTP_USERNAME@$FTP_SERVER/$SERVER || warn "Could not remove old backups from $FTP_SERVER"
        info "Cleaning of extraneous remote files done."
    fi
    if [ -d /mnt/juststoragespace ]; then
        info "deleting old rdiff-backups"
        mount /mnt/juststoragespace || error "Error mounting storagespace"
        mountpoint -q /mnt/juststoragespace || "Storagespace not mounted"
        rdiff-backup --force --remove-older-than "$BACKUP_ROTATE_WEEKS"W /mnt/juststoragespace/$SERVER/ || error "Error occured while deleting old rdiff-backups"
        umount /mnt/juststoragespace || error "Error unmounting storagespace"
    fi
}


checkArguments() {

    db_type="pg"

    if [ $# -eq 1 ] || [ $# -eq 2 ]; then
        if [ "$1" = "ora" ]; then
            db_type="ora"
        elif [ "$1" != "pg" ] && [ "$1" != "ora" ]; then
            help
            exit 1
        elif [ "$1" = "--help" ]; then
            help
            exit 0
        fi
    fi

    if [ $# -eq 2 ]; then
        if [ "$2" != "full" ]; then
            help
            exit 1
        fi
    fi
}

args=$*
checkArguments $args
setProperties
status "STARTED"

dumpDatabase
backupFiles
verifyFiles
removeOldBackups
transferBackup $args
removeOldRemoteBackups
status "FINISHED"
